<?php
/**
 * API para el Dashboard Principal
 * Maneja las peticiones AJAX del dashboard
 */

require_once '../config/database.php';
require_once '../src/BLL/DashboardService.php';
require_once '../src/BLL/AuthService.php';

// Configuración de headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Requested-With');

// Manejar preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    // Verificar sesión
    session_start();
    $authService = new AuthService();
    
    if (!isset($_SESSION['usuario_id'])) {
        throw new Exception('Sesión no válida', 401);
    }

    // Instanciar servicios
    $dashboardService = new DashboardService();
    
    // Obtener acción
    $accion = $_GET['accion'] ?? $_POST['accion'] ?? '';
    
    if (empty($accion)) {
        throw new Exception('Acción no especificada', 400);
    }

    // Procesar según la acción
    switch ($accion) {
        case 'obtener_metricas':
            $periodo = $_GET['periodo'] ?? 'semana';
            $resultado = obtenerMetricasCompletas($dashboardService, $periodo);
            break;
            
        case 'obtener_graficos':
            $periodo = $_GET['periodo'] ?? 'semana';
            $resultado = obtenerDatosGraficos($dashboardService, $periodo);
            break;
            
        case 'obtener_actividad':
            $limite = (int)($_GET['limite'] ?? 10);
            $resultado = obtenerActividadReciente($dashboardService, $limite);
            break;
            
        case 'obtener_top_utensilios':
            $limite = (int)($_GET['limite'] ?? 5);
            $resultado = obtenerTopUtensilios($dashboardService, $limite);
            break;
            
        case 'obtener_alertas':
            $resultado = obtenerAlertasActivas($dashboardService);
            break;
            
        case 'exportar':
            $periodo = $_GET['periodo'] ?? 'semana';
            exportarDashboard($dashboardService, $periodo);
            break;
            
        case 'estadisticas_avanzadas':
            $resultado = obtenerEstadisticasAvanzadas($dashboardService);
            break;
            
        default:
            throw new Exception("Acción no válida: $accion", 400);
    }
    
    // Enviar respuesta exitosa
    if (isset($resultado)) {
        echo json_encode([
            'success' => true,
            'data' => $resultado,
            'timestamp' => date('Y-m-d H:i:s')
        ], JSON_UNESCAPED_UNICODE);
    }

} catch (Exception $e) {
    http_response_code($e->getCode() ?: 500);
    echo json_encode([
        'success' => false,
        'mensaje' => $e->getMessage(),
        'codigo' => $e->getCode() ?: 500,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
}

/**
 * Obtiene todas las métricas del dashboard
 */
function obtenerMetricasCompletas($dashboardService, $periodo) {
    // Métricas principales
    $metricas = $dashboardService->obtenerMetricasPrincipales();
    
    // Top utensilios
    $topUtensilios = $dashboardService->obtenerTopUtensilios(5);
    
    // Actividad reciente
    $actividadReciente = $dashboardService->obtenerActividadReciente(10);
    
    // Datos para gráficos
    $graficos = $dashboardService->obtenerDatosGraficos($periodo);
    
    // Alertas activas
    $alertas = obtenerAlertasDelSistema($dashboardService);
    
    return [
        'metricas' => $metricas,
        'top_utensilios' => $topUtensilios,
        'actividad_reciente' => $actividadReciente,
        'graficos' => $graficos,
        'alertas' => $alertas,
        'periodo' => $periodo
    ];
}

/**
 * Obtiene datos específicos para gráficos
 */
function obtenerDatosGraficos($dashboardService, $periodo) {
    try {
        $datos = $dashboardService->obtenerDatosGraficos($periodo);
        
        // Procesar datos para Chart.js
        $movimientos = [
            'labels' => [],
            'entradas' => [],
            'salidas' => []
        ];
        
        if (isset($datos['movimientos_por_dia'])) {
            foreach ($datos['movimientos_por_dia'] as $dia) {
                $movimientos['labels'][] = date('d/m', strtotime($dia['fecha']));
                $movimientos['entradas'][] = (int)($dia['entradas'] ?? 0);
                $movimientos['salidas'][] = (int)($dia['salidas'] ?? 0);
            }
        }
        
        return [
            'movimientos' => $movimientos,
            'tipos_utensilio' => $datos['utensilios_por_tipo'] ?? [],
            'tendencia' => $datos['tendencia_semanal'] ?? []
        ];
        
    } catch (Exception $e) {
        error_log("Error obteniendo datos gráficos: " . $e->getMessage());
        return [
            'movimientos' => ['labels' => [], 'entradas' => [], 'salidas' => []],
            'tipos_utensilio' => [],
            'tendencia' => []
        ];
    }
}

/**
 * Obtiene actividad reciente del sistema
 */
function obtenerActividadReciente($dashboardService, $limite) {
    try {
        $actividades = $dashboardService->obtenerActividadReciente($limite);
        
        // Procesar actividades para mejorar la presentación
        return array_map(function($actividad) {
            return [
                'tipo' => determinarTipoActividad($actividad),
                'descripcion' => $actividad['accion'] ?? 'Actividad del sistema',
                'detalle' => $actividad['detalle'] ?? '',
                'fecha' => $actividad['fecha_creacion'] ?? $actividad['fecha'],
                'usuario' => $actividad['usuario'] ?? 'Sistema'
            ];
        }, $actividades);
        
    } catch (Exception $e) {
        error_log("Error obteniendo actividad: " . $e->getMessage());
        return [];
    }
}

/**
 * Obtiene top utensilios más usados
 */
function obtenerTopUtensilios($dashboardService, $limite) {
    try {
        return $dashboardService->obtenerTopUtensilios($limite);
    } catch (Exception $e) {
        error_log("Error obteniendo top utensilios: " . $e->getMessage());
        return [];
    }
}

/**
 * Obtiene alertas activas del sistema
 */
function obtenerAlertasActivas($dashboardService) {
    return obtenerAlertasDelSistema($dashboardService);
}

/**
 * Obtiene estadísticas avanzadas
 */
function obtenerEstadisticasAvanzadas($dashboardService) {
    try {
        return [
            'eficiencia_operacional' => $dashboardService->calcularEficienciaOperacional(),
            'tendencias_uso' => $dashboardService->analizarTendenciasUso(),
            'predicciones' => $dashboardService->generarPredicciones(),
            'comparativa_periodos' => $dashboardService->compararPeriodos()
        ];
    } catch (Exception $e) {
        error_log("Error obteniendo estadísticas avanzadas: " . $e->getMessage());
        return [];
    }
}

/**
 * Genera alertas del sistema
 */
function obtenerAlertasDelSistema($dashboardService) {
    $alertas = [];
    
    try {
        // Verificar stock bajo
        $stockBajo = $dashboardService->verificarStockBajo();
        if (!empty($stockBajo)) {
            $alertas[] = [
                'tipo' => 'stock_bajo',
                'nivel' => 'warning',
                'titulo' => 'Stock Bajo',
                'descripcion' => count($stockBajo) . ' utensilio(s) con stock bajo',
                'fecha' => date('Y-m-d'),
                'datos' => $stockBajo
            ];
        }
        
        // Verificar utensilios inactivos
        $inactivos = $dashboardService->verificarUtensiliosInactivos();
        if (!empty($inactivos)) {
            $alertas[] = [
                'tipo' => 'inactividad',
                'nivel' => 'info',
                'titulo' => 'Utensilios Inactivos',
                'descripcion' => count($inactivos) . ' utensilio(s) sin movimiento en 30 días',
                'fecha' => date('Y-m-d'),
                'datos' => $inactivos
            ];
        }
        
        // Verificar mantenimiento pendiente
        $mantenimiento = $dashboardService->verificarMantenimientoPendiente();
        if (!empty($mantenimiento)) {
            $alertas[] = [
                'tipo' => 'mantenimiento',
                'nivel' => 'danger',
                'titulo' => 'Mantenimiento Pendiente',
                'descripcion' => count($mantenimiento) . ' utensilio(s) requieren mantenimiento',
                'fecha' => date('Y-m-d'),
                'datos' => $mantenimiento
            ];
        }
        
        return $alertas;
        
    } catch (Exception $e) {
        error_log("Error generando alertas: " . $e->getMessage());
        return [];
    }
}

/**
 * Determina el tipo de actividad
 */
function determinarTipoActividad($actividad) {
    $accion = strtolower($actividad['accion'] ?? '');
    
    if (strpos($accion, 'crear') !== false || strpos($accion, 'registr') !== false) {
        return 'creacion';
    } elseif (strpos($accion, 'editar') !== false || strpos($accion, 'actualiz') !== false) {
        return 'edicion';
    } elseif (strpos($accion, 'eliminar') !== false || strpos($accion, 'borrar') !== false) {
        return 'eliminacion';
    } elseif (strpos($accion, 'entrada') !== false) {
        return 'entrada';
    } elseif (strpos($accion, 'salida') !== false) {
        return 'salida';
    } elseif (strpos($accion, 'transfer') !== false) {
        return 'transferencia';
    } elseif (strpos($accion, 'mantenimiento') !== false) {
        return 'mantenimiento';
    }
    
    return 'general';
}

/**
 * Exporta datos del dashboard
 */
function exportarDashboard($dashboardService, $periodo) {
    try {
        // Obtener todos los datos
        $datos = obtenerMetricasCompletas($dashboardService, $periodo);
        
        // Configurar headers para descarga
        $filename = "dashboard-{$periodo}-" . date('Y-m-d') . ".json";
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: no-cache, must-revalidate');
        header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
        
        // Agregar metadatos
        $export = [
            'metadata' => [
                'exported_at' => date('Y-m-d H:i:s'),
                'period' => $periodo,
                'version' => '1.0',
                'format' => 'json'
            ],
            'dashboard_data' => $datos
        ];
        
        echo json_encode($export, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit();
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'mensaje' => 'Error al exportar datos: ' . $e->getMessage()
        ], JSON_UNESCAPED_UNICODE);
    }
}

/**
 * Función de utilidad para logging
 */
function logActivity($mensaje, $nivel = 'info') {
    $logFile = '../logs/dashboard-' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$nivel] $mensaje" . PHP_EOL;
    
    // Crear directorio de logs si no existe
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0777, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

/**
 * Función de utilidad para validar parámetros
 */
function validarParametros($parametros, $requeridos) {
    $errores = [];
    
    foreach ($requeridos as $campo) {
        if (!isset($parametros[$campo]) || empty($parametros[$campo])) {
            $errores[] = "El campo '$campo' es requerido";
        }
    }
    
    if (!empty($errores)) {
        throw new Exception('Parámetros faltantes: ' . implode(', ', $errores), 400);
    }
}

/**
 * Función de utilidad para sanitizar entrada
 */
function sanitizarEntrada($data) {
    if (is_array($data)) {
        return array_map('sanitizarEntrada', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}
?>