<?php
/**
 * ========================================
 * SYSTEM MONITOR API
 * Endpoint para el Panel de Administración
 * ========================================
 */

require_once '../config/config.php';
require_once '../src/Utils/CacheManager.php';
require_once '../src/Utils/PerformanceMonitor.php';
require_once '../src/Utils/SecurityManager.php';
require_once '../src/DAL/Database.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Requested-With');

// Manejar preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

/**
 * ========================================
 * SYSTEM MONITOR CLASS
 * ========================================
 */
class SystemMonitorAPI {
    
    private $cache;
    private $performance;
    private $security;
    private $db;
    
    public function __construct() {
        $this->cache = CacheManager::getInstance();
        $this->performance = PerformanceMonitor::getInstance();
        $this->security = new SecurityManager();
        
        try {
            $this->db = new Database();
        } catch (Exception $e) {
            error_log("Error conectando a la base de datos: " . $e->getMessage());
            $this->db = null;
        }
    }
    
    /**
     * Maneja las solicitudes de la API
     */
    public function handleRequest() {
        try {
            $accion = $_GET['accion'] ?? '';
            
            switch ($accion) {
                case 'system_overview':
                    return $this->getSystemOverview();
                    
                case 'performance_metrics':
                    return $this->getPerformanceMetrics();
                    
                case 'security_events':
                    $limite = (int)($_GET['limite'] ?? 10);
                    return $this->getSecurityEvents($limite);
                    
                case 'cache_stats':
                    return $this->getCacheStats();
                    
                case 'clear_cache':
                    return $this->clearCache();
                    
                case 'optimization_report':
                    return $this->getOptimizationReport();
                    
                case 'health_check':
                    return $this->getHealthCheck();
                    
                default:
                    return $this->error('Acción no válida', 400);
            }
            
        } catch (Exception $e) {
            error_log("Error en SystemMonitorAPI: " . $e->getMessage());
            return $this->error('Error interno del servidor', 500);
        }
    }
    
    /**
     * Obtiene el resumen completo del sistema
     */
    private function getSystemOverview() {
        $overview = [
            'health' => $this->getHealthStatus(),
            'performance' => $this->getPerformanceData(),
            'security' => $this->getSecurityStatus(),
            'cache' => $this->getCacheStatistics(),
            'summary' => $this->getSystemSummary()
        ];
        
        return $this->success($overview);
    }
    
    /**
     * Estado de salud del sistema
     */
    private function getHealthStatus() {
        $checks = [];
        $overall_status = 'ok';
        
        // Verificar conexión a base de datos
        $checks['database'] = [
            'name' => 'Base de Datos',
            'status' => $this->db !== null ? 'ok' : 'error',
            'message' => $this->db !== null ? 'Conectado' : 'Error de conexión'
        ];
        
        // Verificar archivos críticos
        $critical_files = [
            '../config/config.php',
            '../src/DAL/Database.php',
            '../src/Utils/CacheManager.php'
        ];
        
        $missing_files = 0;
        foreach ($critical_files as $file) {
            if (!file_exists($file)) {
                $missing_files++;
            }
        }
        
        $checks['files'] = [
            'name' => 'Archivos del Sistema',
            'status' => $missing_files === 0 ? 'ok' : 'error',
            'message' => $missing_files === 0 ? 'Todos presentes' : "{$missing_files} archivos faltantes"
        ];
        
        // Verificar permisos de escritura
        $writable_dirs = ['../logs', '../cache'];
        $non_writable = 0;
        
        foreach ($writable_dirs as $dir) {
            if (!is_dir($dir)) {
                @mkdir($dir, 0755, true);
            }
            if (!is_writable($dir)) {
                $non_writable++;
            }
        }
        
        $checks['permissions'] = [
            'name' => 'Permisos del Sistema',
            'status' => $non_writable === 0 ? 'ok' : 'warning',
            'message' => $non_writable === 0 ? 'Correctos' : "{$non_writable} directorios sin permisos"
        ];
        
        // Verificar uso de memoria
        $memory_usage = memory_get_usage(true);
        $memory_limit = $this->parseBytes(ini_get('memory_limit'));
        $memory_percent = ($memory_usage / $memory_limit) * 100;
        
        $checks['memory'] = [
            'name' => 'Uso de Memoria',
            'status' => $memory_percent < 80 ? 'ok' : ($memory_percent < 95 ? 'warning' : 'error'),
            'message' => round($memory_percent, 1) . '% utilizada'
        ];
        
        // Determinar estado general
        foreach ($checks as $check) {
            if ($check['status'] === 'error') {
                $overall_status = 'error';
                break;
            } elseif ($check['status'] === 'warning' && $overall_status !== 'error') {
                $overall_status = 'warning';
            }
        }
        
        return [
            'overall_status' => $overall_status,
            'checks' => $checks,
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Datos de rendimiento del sistema
     */
    private function getPerformanceData() {
        $stats = $this->performance->getPerformanceStats();
        
        return [
            'statistics' => $stats,
            'summary' => [
                'score' => $this->calculatePerformanceScore($stats),
                'status' => $this->getPerformanceStatus($stats),
                'last_update' => date('Y-m-d H:i:s')
            ]
        ];
    }
    
    /**
     * Determina el estado de rendimiento
     */
    private function getPerformanceStatus($stats) {
        $score = $this->calculatePerformanceScore($stats);
        
        if ($score >= 90) return 'excellent';
        if ($score >= 70) return 'good';
        if ($score >= 50) return 'fair';
        return 'poor';
    }
    
    /**
     * Calcula el puntaje de rendimiento
     */
    private function calculatePerformanceScore($stats) {
        $score = 100;
        
        // Penalizar por tiempo de consulta promedio alto
        if (isset($stats['queries']['avg_time'])) {
            if ($stats['queries']['avg_time'] > 0.5) {
                $score -= 30;
            } elseif ($stats['queries']['avg_time'] > 0.1) {
                $score -= 15;
            }
        }
        
        // Penalizar por uso alto de memoria
        if (isset($stats['execution']['memory_used_mb'])) {
            if ($stats['execution']['memory_used_mb'] > 100) {
                $score -= 20;
            } elseif ($stats['execution']['memory_used_mb'] > 50) {
                $score -= 10;
            }
        }
        
        // Penalizar por muchas consultas
        if (isset($stats['queries']['total_count'])) {
            if ($stats['queries']['total_count'] > 200) {
                $score -= 15;
            } elseif ($stats['queries']['total_count'] > 100) {
                $score -= 8;
            }
        }
        
        return max(0, min(100, $score));
    }
    
    /**
     * Estado de seguridad del sistema
     */
    private function getSecurityStatus() {
        $security_level = 'secure';
        $recent_events = 0;
        
        // Contar eventos de seguridad recientes (últimas 24 horas)
        $log_file = '../logs/security.log';
        if (file_exists($log_file)) {
            $lines = file($log_file, FILE_IGNORE_NEW_LINES);
            $yesterday = date('Y-m-d', strtotime('-1 day'));
            
            foreach ($lines as $line) {
                if (strpos($line, date('Y-m-d')) !== false) {
                    $recent_events++;
                }
            }
            
            // Determinar nivel de seguridad basado en eventos
            if ($recent_events > 50) {
                $security_level = 'high_risk';
            } elseif ($recent_events > 20) {
                $security_level = 'medium_risk';
            } elseif ($recent_events > 5) {
                $security_level = 'low_risk';
            }
        }
        
        return [
            'security_level' => $security_level,
            'recent_events' => $recent_events,
            'last_scan' => date('Y-m-d H:i:s'),
            'status' => $security_level === 'secure' ? 'ok' : 'warning'
        ];
    }
    
    /**
     * Estadísticas de caché
     */
    private function getCacheStatistics() {
        $cache_dir = '../cache';
        $total_size = 0;
        $file_count = 0;
        
        if (is_dir($cache_dir)) {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($cache_dir)
            );
            
            foreach ($iterator as $file) {
                if ($file->isFile()) {
                    $total_size += $file->getSize();
                    $file_count++;
                }
            }
        }
        
        return [
            'total_size_mb' => round($total_size / (1024 * 1024), 2),
            'memory_items' => $file_count,
            'status' => 'active',
            'last_cleanup' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Resumen del sistema
     */
    private function getSystemSummary() {
        return [
            'php_version' => phpversion(),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'uptime' => [
                'seconds' => time() - ($_SERVER['REQUEST_TIME'] ?? time()),
                'formatted' => $this->formatUptime(time() - ($_SERVER['REQUEST_TIME'] ?? time()))
            ],
            'server_load' => $this->getServerLoad(),
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Obtiene métricas de rendimiento detalladas
     */
    private function getPerformanceMetrics() {
        $stats = $this->performance->getPerformanceStats();
        $report = $this->performance->generateReport();
        
        return $this->success([
            'statistics' => $stats,
            'report' => $report,
            'score' => $this->calculatePerformanceScore($stats),
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }
    
    /**
     * Obtiene eventos de seguridad recientes
     */
    private function getSecurityEvents($limite = 10) {
        $events = [];
        $log_file = '../logs/security.log';
        
        if (file_exists($log_file)) {
            $lines = array_reverse(file($log_file, FILE_IGNORE_NEW_LINES));
            $count = 0;
            
            foreach ($lines as $line) {
                if ($count >= $limite) break;
                
                $parts = explode(' | ', $line);
                if (count($parts) >= 3) {
                    $events[] = [
                        'timestamp' => $parts[0] ?? '',
                        'event' => $parts[1] ?? '',
                        'details' => json_decode($parts[2] ?? '{}', true)
                    ];
                    $count++;
                }
            }
        }
        
        return $this->success(['events' => $events]);
    }
    
    /**
     * Obtiene estadísticas de caché
     */
    private function getCacheStats() {
        $stats = $this->getCacheStatistics();
        return $this->success($stats);
    }
    
    /**
     * Limpia el caché del sistema
     */
    private function clearCache() {
        try {
            $cleared = $this->cache->clear();
            
            return $this->success([
                'message' => 'Caché limpiado exitosamente',
                'cleared_items' => $cleared ? 1 : 0,
                'timestamp' => date('Y-m-d H:i:s')
            ]);
            
        } catch (Exception $e) {
            return $this->error('Error al limpiar el caché: ' . $e->getMessage());
        }
    }
    
    /**
     * Genera reporte de optimización
     */
    private function getOptimizationReport() {
        $report = $this->performance->generateReport();
        $recommendations = [];
        
        // Generar recomendaciones basadas en el rendimiento
        if (isset($report['recommendations'])) {
            $recommendations = $report['recommendations'];
        } else {
            $stats = $this->performance->getPerformanceStats();
            
            // Recomendaciones basadas en estadísticas
            if ($stats['queries']['avg_time'] > 0.1) {
                $recommendations[] = [
                    'type' => 'queries',
                    'severity' => 'high',
                    'message' => 'Consultas lentas detectadas',
                    'value' => round($stats['queries']['avg_time'], 3) . 's promedio'
                ];
            }
            
            if ($stats['execution']['memory_used_mb'] > 50) {
                $recommendations[] = [
                    'type' => 'memory',
                    'severity' => 'medium',
                    'message' => 'Alto uso de memoria',
                    'value' => $stats['execution']['memory_used_mb'] . ' MB'
                ];
            }
            
            if ($stats['queries']['total_count'] > 100) {
                $recommendations[] = [
                    'type' => 'cache',
                    'severity' => 'medium',
                    'message' => 'Muchas consultas - considerar más caché',
                    'value' => $stats['queries']['total_count'] . ' consultas'
                ];
            }
        }
        
        return $this->success([
            'recommendations' => $recommendations,
            'report' => $report,
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }
    
    /**
     * Verificación de salud rápida
     */
    private function getHealthCheck() {
        $health = $this->getHealthStatus();
        
        return $this->success([
            'status' => $health['overall_status'],
            'checks_passed' => count(array_filter($health['checks'], function($check) {
                return $check['status'] === 'ok';
            })),
            'total_checks' => count($health['checks']),
            'timestamp' => $health['timestamp']
        ]);
    }
    
    /**
     * Convierte bytes de string a número
     */
    private function parseBytes($size) {
        $unit = preg_replace('/[^bkmgtpezy]/i', '', $size);
        $size = preg_replace('/[^0-9\.]/', '', $size);
        
        if ($unit) {
            return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
        }
        
        return round($size);
    }
    
    /**
     * Formatea tiempo de actividad
     */
    private function formatUptime($seconds) {
        $days = floor($seconds / 86400);
        $hours = floor(($seconds % 86400) / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        
        if ($days > 0) {
            return "{$days}d {$hours}h {$minutes}m";
        } elseif ($hours > 0) {
            return "{$hours}h {$minutes}m";
        } else {
            return "{$minutes}m";
        }
    }
    
    /**
     * Obtiene carga del servidor (simulada)
     */
    private function getServerLoad() {
        // En Windows no tenemos sys_getloadavg()
        // Simulamos basándose en el uso de memoria
        $memory_usage = memory_get_usage(true);
        $memory_limit = $this->parseBytes(ini_get('memory_limit'));
        $load = ($memory_usage / $memory_limit);
        
        return [
            '1_min' => round($load, 2),
            '5_min' => round($load * 0.9, 2),
            '15_min' => round($load * 0.8, 2)
        ];
    }
    
    /**
     * Respuesta exitosa
     */
    private function success($data, $mensaje = 'OK') {
        return [
            'success' => true,
            'mensaje' => $mensaje,
            'data' => $data,
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Respuesta de error
     */
    private function error($mensaje, $codigo = 400) {
        http_response_code($codigo);
        return [
            'success' => false,
            'mensaje' => $mensaje,
            'codigo' => $codigo,
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
}


/**
 * ========================================
 * EJECUCIÓN DE LA API
 * ========================================
 */
try {
    $api = new SystemMonitorAPI();
    $resultado = $api->handleRequest();
    echo json_encode($resultado, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    error_log("Error fatal en system-monitor.php: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'mensaje' => 'Error interno del servidor',
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
}/**
 * Obtiene uptime estimado del sistema
 */
function getSystemUptime() {
    $startFile = '../cache/system_start.txt';
    
    if (!file_exists($startFile)) {
        file_put_contents($startFile, time());
    }
    
    $startTime = (int) file_get_contents($startFile);
    $uptime = time() - $startTime;
    
    $days = floor($uptime / 86400);
    $hours = floor(($uptime % 86400) / 3600);
    $minutes = floor(($uptime % 3600) / 60);
    
    return [
        'seconds' => $uptime,
        'formatted' => sprintf('%d días, %d horas, %d minutos', $days, $hours, $minutes)
    ];
}

/**
 * Obtiene carga del servidor (si está disponible)
 */
function getServerLoad() {
    if (function_exists('sys_getloadavg')) {
        $load = sys_getloadavg();
        return [
            '1_min' => $load[0],
            '5_min' => $load[1],
            '15_min' => $load[2]
        ];
    }
    
    return null;
}
?>