<?php
/**
 * API endpoint para movimientos
 */

// Configurar headers para API
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Incluir configuración
require_once __DIR__ . '/../../config/config.php';

// Verificar que sea una petición AJAX
if (!isset($_SERVER['HTTP_X_REQUESTED_WITH']) || strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) !== 'xmlhttprequest') {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Solo se permiten peticiones AJAX']);
    exit;
}

// Verificar autenticación
$authService = new AuthService();
if (!$authService->estaAutenticado()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autenticado']);
    exit;
}

$usuario = $authService->getUsuarioActual();

try {
    $method = $_SERVER['REQUEST_METHOD'];
    $path = $_SERVER['PATH_INFO'] ?? '';
    
    // Remover /api/movimientos del path
    $path = str_replace('/api/movimientos', '', $path);
    $path = trim($path, '/');
    
    // Instanciar servicios
    $movimientoService = new MovimientoService();
    $inventarioService = new InventarioService();
    
    switch ($method) {
        case 'GET':
            handleGet($path, $movimientoService, $inventarioService);
            break;
            
        case 'POST':
            handlePost($path, $movimientoService, $authService, $usuario);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['success' => false, 'message' => 'Método no permitido']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

/**
 * Maneja peticiones GET
 */
function handleGet($path, $movimientoService, $inventarioService) {
    $segments = explode('/', $path);
    
    switch ($segments[0]) {
        case 'motivos':
            $tipo = $segments[1] ?? 'entrada';
            $motivos = $movimientoService->obtenerMotivosSugeridos($tipo);
            echo json_encode($motivos);
            break;
            
        case '':
        case 'listar':
            $filtros = $_GET;
            $movimientos = $movimientoService->obtenerHistorial($filtros);
            echo json_encode(['success' => true, 'data' => $movimientos]);
            break;
            
        case 'detalle':
            if (!isset($segments[1]) || !is_numeric($segments[1])) {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'ID de movimiento requerido']);
                return;
            }
            
            try {
                $movimiento = $movimientoService->obtenerPorId($segments[1]);
                echo json_encode(['success' => true, 'data' => $movimiento]);
            } catch (Exception $e) {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            break;
            
        default:
            // Obtener movimiento específico por ID
            if (is_numeric($segments[0])) {
                $id = (int)$segments[0];
                $movimiento = $movimientoService->obtenerPorId($id);
                if ($movimiento) {
                    echo json_encode(['success' => true, 'data' => $movimiento]);
                } else {
                    http_response_code(404);
                    echo json_encode(['success' => false, 'message' => 'Movimiento no encontrado']);
                }
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Endpoint no encontrado']);
            }
    }
}

/**
 * Maneja peticiones POST
 */
function handlePost($path, $movimientoService, $authService, $usuario) {
    // Verificar permisos
    if (!$usuario->puedeGestionar('movimientos')) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'No tiene permisos para registrar movimientos']);
        return;
    }
    
    switch ($path) {
        case 'registrar':
            registrarMovimiento($movimientoService, $authService);
            break;
            
        case 'corregir':
            corregirMovimiento($movimientoService, $authService);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Endpoint no encontrado']);
    }
}

/**
 * Registra un nuevo movimiento
 */
function registrarMovimiento($movimientoService, $authService) {
    // Verificar CSRF token
    $token = $_POST['csrf_token'] ?? '';
    if (!$authService->verificarCSRFToken($token)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Token CSRF inválido']);
        return;
    }
    
    // Validar datos requeridos
    $datos = [
        'utensilio_id' => (int)($_POST['utensilio_id'] ?? 0),
        'tipo_movimiento' => $_POST['tipo_movimiento'] ?? '',
        'cantidad' => (int)($_POST['cantidad'] ?? 0),
        'motivo' => $_POST['motivo'] ?? '',
        'observaciones' => $_POST['observaciones'] ?? '',
        'ubicacion_destino' => $_POST['ubicacion_destino'] ?? ''
    ];
    
    // Validaciones básicas
    if (!$datos['utensilio_id']) {
        echo json_encode(['success' => false, 'message' => 'Debe seleccionar un utensilio']);
        return;
    }
    
    if (!in_array($datos['tipo_movimiento'], ['entrada', 'salida'])) {
        echo json_encode(['success' => false, 'message' => 'Tipo de movimiento inválido']);
        return;
    }
    
    if ($datos['cantidad'] <= 0) {
        echo json_encode(['success' => false, 'message' => 'La cantidad debe ser mayor a 0']);
        return;
    }
    
    if (!$datos['motivo']) {
        echo json_encode(['success' => false, 'message' => 'Debe especificar un motivo']);
        return;
    }
    
    try {
        // Registrar el movimiento usando el servicio correspondiente
        if ($datos['tipo_movimiento'] === 'entrada') {
            $id = $movimientoService->registrarEntrada(
                $datos['utensilio_id'],
                $datos['cantidad'],
                $datos['motivo'],
                $datos['observaciones']
            );
        } else {
            $id = $movimientoService->registrarSalida(
                $datos['utensilio_id'],
                $datos['cantidad'],
                $datos['motivo'],
                $datos['observaciones']
            );
        }
        
        echo json_encode([
            'success' => true, 
            'message' => 'Movimiento registrado exitosamente',
            'id' => $id
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Corrige un movimiento existente
 */
function corregirMovimiento($movimientoService, $authService) {
    // Verificar CSRF token
    $token = $_POST['csrf_token'] ?? '';
    if (!$authService->verificarCSRFToken($token)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Token CSRF inválido']);
        return;
    }
    
    $id = (int)($_POST['movimiento_id'] ?? 0);
    $observacion = $_POST['observacion'] ?? '';
    
    if (!$id) {
        echo json_encode(['success' => false, 'message' => 'ID de movimiento requerido']);
        return;
    }
    
    if (!$observacion) {
        echo json_encode(['success' => false, 'message' => 'Observación de corrección requerida']);
        return;
    }
    
    try {
        $movimientoService->corregirMovimiento($id, $observacion);
        echo json_encode(['success' => true, 'message' => 'Movimiento corregido exitosamente']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}