<?php
/**
 * API para el sistema de reportes
 * Maneja la generación, visualización y descarga de reportes
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type, X-CSRF-Token');

require_once '../../config/config.php';
require_once '../../src/BLL/ReporteService.php';
require_once '../../src/BLL/AuthService.php';

try {
    // Verificar autenticación
    $authService = new AuthService();
    $authService->verificarSesion();
    
    $reporteService = new ReporteService();
    $usuario = $authService->getUsuarioActual();
    
    $method = $_SERVER['REQUEST_METHOD'];
    $path = $_GET['path'] ?? '';
    
    switch ($method) {
        case 'GET':
            handleGet($path, $reporteService);
            break;
            
        case 'POST':
            handlePost($path, $reporteService, $authService, $usuario);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['success' => false, 'message' => 'Método no permitido']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

/**
 * Maneja peticiones GET
 */
function handleGet($path, $reporteService) {
    $segments = explode('/', $path);
    
    switch ($segments[0]) {
        case 'recientes':
            $reportes = obtenerReportesRecientes();
            echo json_encode($reportes);
            break;
            
        case 'tipos':
            $tipos = obtenerTiposReporte();
            echo json_encode($tipos);
            break;
            
        case 'estadisticas':
            $stats = $reporteService->generarReporteEjecutivo();
            echo json_encode(['success' => true, 'data' => $stats]);
            break;
            
        default:
            if (isset($segments[0]) && is_numeric($segments[0])) {
                // Obtener reporte específico por ID
                $reporte = obtenerReportePorId($segments[0]);
                echo json_encode($reporte);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Endpoint no encontrado']);
            }
            break;
    }
}

/**
 * Maneja peticiones POST
 */
function handlePost($path, $reporteService, $authService, $usuario) {
    $segments = explode('/', $path);
    
    switch ($segments[0]) {
        case 'generar':
            generarNuevoReporte($reporteService, $usuario);
            break;
            
        case 'alertas':
            if ($segments[1] === 'exportar') {
                exportarAlertasCSV($reporteService);
            }
            break;
            
        case 'programar':
            programarReporteAutomatico($reporteService, $usuario);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Endpoint no encontrado']);
    }
}

/**
 * Genera un nuevo reporte
 */
function generarNuevoReporte($reporteService, $usuario) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Datos inválidos']);
        return;
    }
    
    $tipo = $input['tipo_reporte'] ?? '';
    $formato = $input['formato'] ?? 'html';
    $fechaInicio = $input['fecha_inicio'] ?? null;
    $fechaFin = $input['fecha_fin'] ?? null;
    $filtros = [
        'incluir_inactivos' => isset($input['incluir_inactivos']),
        'solo_alertas' => isset($input['solo_alertas'])
    ];
    
    try {
        switch ($tipo) {
            case 'inventario':
                $data = $reporteService->generarReporteInventario();
                $titulo = 'Reporte de Inventario';
                $descripcion = 'Estado actual del inventario de utensilios';
                break;
                
            case 'movimientos':
                if (!$fechaInicio || !$fechaFin) {
                    throw new Exception('Fechas de inicio y fin son requeridas para el reporte de movimientos');
                }
                $data = $reporteService->generarReporteMovimientos($fechaInicio, $fechaFin, $filtros);
                $titulo = 'Reporte de Movimientos';
                $descripcion = "Movimientos del período {$fechaInicio} al {$fechaFin}";
                break;
                
            case 'usuarios':
                $data = $reporteService->generarReporteUsuarios($fechaInicio, $fechaFin);
                $titulo = 'Reporte de Usuarios';
                $descripcion = 'Actividad y rendimiento por usuario';
                break;
                
            case 'alertas':
                $data = $reporteService->generarReporteAlertas();
                $titulo = 'Reporte de Alertas';
                $descripcion = 'Estado de alertas y notificaciones del sistema';
                break;
                
            case 'ejecutivo':
                $periodo = determinarPeriodo($fechaInicio, $fechaFin);
                $data = $reporteService->generarReporteEjecutivo($periodo);
                $titulo = 'Reporte Ejecutivo';
                $descripcion = "Resumen ejecutivo del período {$periodo}";
                break;
                
            default:
                throw new Exception('Tipo de reporte no válido');
        }
        
        // Generar contenido según formato
        $contenido = generarContenidoReporte($data, $formato, $tipo);
        
        $reporte = [
            'titulo' => $titulo,
            'descripcion' => $descripcion,
            'tipo' => $tipo,
            'formato' => $formato,
            'contenido' => $contenido,
            'nombre' => strtolower(str_replace(' ', '_', $titulo)),
            'fecha_generacion' => date('Y-m-d H:i:s'),
            'usuario' => $usuario->nombre,
            'autoPrint' => $formato === 'pdf'
        ];
        
        // Registrar el reporte generado
        registrarReporteGenerado($reporte, $usuario->id);
        
        echo json_encode(['success' => true, 'data' => $reporte]);
        
    } catch (Exception $e) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Genera contenido del reporte según el formato
 */
function generarContenidoReporte($data, $formato, $tipo) {
    switch ($formato) {
        case 'html':
            return generarHTMLReporte($data, $tipo);
            
        case 'csv':
            return generarCSVReporte($data, $tipo);
            
        case 'pdf':
            return generarPDFReporte($data, $tipo);
            
        case 'excel':
            return generarExcelReporte($data, $tipo);
            
        default:
            return generarHTMLReporte($data, $tipo);
    }
}

/**
 * Genera reporte en formato HTML
 */
function generarHTMLReporte($data, $tipo) {
    $html = '<div class="report-content">';
    
    // Resumen
    if (isset($data['resumen'])) {
        $html .= '<div class="report-section">';
        $html .= '<h3><i class="fas fa-chart-bar me-2"></i>Resumen</h3>';
        $html .= '<div class="row">';
        
        foreach ($data['resumen'] as $key => $value) {
            if (is_array($value)) {
                $html .= '<div class="col-md-6 mb-3">';
                $html .= '<div class="card">';
                $html .= '<div class="card-body">';
                $html .= '<h6>' . ucfirst(str_replace('_', ' ', $key)) . '</h6>';
                
                foreach ($value as $subKey => $subValue) {
                    $html .= '<div class="d-flex justify-content-between">';
                    $html .= '<span>' . ucfirst(str_replace('_', ' ', $subKey)) . ':</span>';
                    $html .= '<strong>' . (is_numeric($subValue) ? number_format($subValue) : $subValue) . '</strong>';
                    $html .= '</div>';
                }
                
                $html .= '</div></div></div>';
            }
        }
        
        $html .= '</div></div>';
    }
    
    // Detalles
    if (isset($data['detalles']) && !empty($data['detalles'])) {
        $html .= '<div class="report-section">';
        $html .= '<h3><i class="fas fa-list me-2"></i>Detalles</h3>';
        $html .= '<div class="table-responsive">';
        $html .= '<table class="table table-striped table-hover">';
        
        // Encabezados
        $html .= '<thead class="table-dark">';
        $html .= '<tr>';
        foreach (array_keys($data['detalles'][0]) as $header) {
            $html .= '<th>' . ucfirst(str_replace('_', ' ', $header)) . '</th>';
        }
        $html .= '</tr>';
        $html .= '</thead>';
        
        // Datos
        $html .= '<tbody>';
        foreach ($data['detalles'] as $row) {
            $html .= '<tr>';
            foreach ($row as $cell) {
                $html .= '<td>' . htmlspecialchars($cell) . '</td>';
            }
            $html .= '</tr>';
        }
        $html .= '</tbody>';
        
        $html .= '</table>';
        $html .= '</div></div>';
    }
    
    // Alertas específicas para reporte de alertas
    if ($tipo === 'alertas' && isset($data['alertas'])) {
        $html .= '<div class="report-section">';
        $html .= '<h3><i class="fas fa-exclamation-triangle me-2"></i>Alertas por Tipo</h3>';
        
        $alertasPorTipo = [];
        foreach ($data['alertas'] as $alerta) {
            $alertasPorTipo[$alerta['tipo']][] = $alerta;
        }
        
        foreach ($alertasPorTipo as $tipo => $alertas) {
            $badgeClass = $tipo === 'CRITICA' ? 'bg-danger' : ($tipo === 'ADVERTENCIA' ? 'bg-warning' : 'bg-info');
            
            $html .= '<div class="mb-4">';
            $html .= '<h4><span class="badge ' . $badgeClass . '">' . $tipo . '</span> (' . count($alertas) . ')</h4>';
            
            foreach ($alertas as $alerta) {
                $html .= '<div class="alert alert-' . ($tipo === 'CRITICA' ? 'danger' : ($tipo === 'ADVERTENCIA' ? 'warning' : 'info')) . '">';
                $html .= '<strong>' . htmlspecialchars($alerta['utensilio']) . '</strong>: ';
                $html .= htmlspecialchars($alerta['mensaje']);
                $html .= '<br><small>Recomendación: ' . htmlspecialchars($alerta['recomendacion']) . '</small>';
                $html .= '</div>';
            }
            
            $html .= '</div>';
        }
        
        $html .= '</div>';
    }
    
    $html .= '</div>';
    return $html;
}

/**
 * Genera reporte en formato CSV
 */
function generarCSVReporte($data, $tipo) {
    if (!isset($data['detalles']) || empty($data['detalles'])) {
        return "Sin datos disponibles\n";
    }
    
    $output = '';
    
    // Encabezados
    $headers = array_keys($data['detalles'][0]);
    $output .= implode(',', array_map(function($h) {
        return '"' . str_replace('"', '""', ucfirst(str_replace('_', ' ', $h))) . '"';
    }, $headers)) . "\n";
    
    // Datos
    foreach ($data['detalles'] as $row) {
        $csvRow = array_map(function($cell) {
            return '"' . str_replace('"', '""', $cell) . '"';
        }, array_values($row));
        
        $output .= implode(',', $csvRow) . "\n";
    }
    
    return $output;
}

/**
 * Genera reporte en formato PDF (simulado)
 */
function generarPDFReporte($data, $tipo) {
    // En una implementación real, usarías una librería como TCPDF o DOMPDF
    // Por ahora retornamos el HTML que se puede convertir a PDF
    return generarHTMLReporte($data, $tipo);
}

/**
 * Genera reporte en formato Excel (simulado)
 */
function generarExcelReporte($data, $tipo) {
    // En una implementación real, usarías PhpSpreadsheet
    // Por ahora retornamos CSV que es compatible con Excel
    return generarCSVReporte($data, $tipo);
}

/**
 * Determina el período basado en las fechas
 */
function determinarPeriodo($fechaInicio, $fechaFin) {
    if (!$fechaInicio || !$fechaFin) {
        return 'mes';
    }
    
    $inicio = new DateTime($fechaInicio);
    $fin = new DateTime($fechaFin);
    $diff = $inicio->diff($fin)->days;
    
    if ($diff <= 7) return 'semana';
    if ($diff <= 31) return 'mes';
    if ($diff <= 92) return 'trimestre';
    return 'año';
}

/**
 * Obtiene reportes recientes (simulado)
 */
function obtenerReportesRecientes() {
    // En una implementación real, esto vendría de la base de datos
    return [];
}

/**
 * Obtiene tipos de reporte disponibles
 */
function obtenerTiposReporte() {
    return [
        ['id' => 'inventario', 'nombre' => 'Inventario', 'descripcion' => 'Estado del inventario'],
        ['id' => 'movimientos', 'nombre' => 'Movimientos', 'descripcion' => 'Entradas y salidas'],
        ['id' => 'usuarios', 'nombre' => 'Usuarios', 'descripcion' => 'Actividad de usuarios'],
        ['id' => 'alertas', 'nombre' => 'Alertas', 'descripcion' => 'Notificaciones del sistema'],
        ['id' => 'ejecutivo', 'nombre' => 'Ejecutivo', 'descripcion' => 'Resumen completo']
    ];
}

/**
 * Obtiene reporte por ID (simulado)
 */
function obtenerReportePorId($id) {
    // En una implementación real, esto vendría de la base de datos
    return [
        'titulo' => 'Reporte de Ejemplo',
        'contenido' => '<h3>Este es un reporte de ejemplo</h3>',
        'fecha' => date('Y-m-d H:i:s')
    ];
}

/**
 * Exporta alertas en formato CSV
 */
function exportarAlertasCSV($reporteService) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="alertas_' . date('Y-m-d') . '.csv"');
    
    $alertas = $reporteService->generarReporteAlertas();
    
    if (empty($alertas['alertas'])) {
        echo "Sin alertas disponibles\n";
        return;
    }
    
    // Encabezados
    echo "Tipo,Nivel,Utensilio,Cantidad Actual,Stock Mínimo,Mensaje,Recomendación\n";
    
    // Datos
    foreach ($alertas['alertas'] as $alerta) {
        echo sprintf(
            '"%s","%s","%s",%d,%d,"%s","%s"' . "\n",
            $alerta['tipo'],
            $alerta['nivel'],
            $alerta['utensilio'],
            $alerta['cantidad_actual'],
            $alerta['stock_minimo'],
            str_replace('"', '""', $alerta['mensaje']),
            str_replace('"', '""', $alerta['recomendacion'])
        );
    }
}

/**
 * Programa reporte automático (simulado)
 */
function programarReporteAutomatico($reporteService, $usuario) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    // En una implementación real, esto se guardaría en la base de datos
    // y se ejecutaría mediante un cron job
    
    echo json_encode([
        'success' => true, 
        'message' => 'Reporte programado exitosamente',
        'data' => ['id' => uniqid(), 'estado' => 'programado']
    ]);
}

/**
 * Registra el reporte generado (simulado)
 */
function registrarReporteGenerado($reporte, $usuarioId) {
    // En una implementación real, esto se guardaría en la base de datos
    error_log("Reporte generado: " . json_encode([
        'tipo' => $reporte['tipo'],
        'usuario_id' => $usuarioId,
        'fecha' => $reporte['fecha_generacion']
    ]));
}