// JavaScript para el Panel de Administración del Sistema

/**
 * ========================================
 * ADMIN PANEL MANAGER
 * ========================================
 */
class AdminPanelManager {
    constructor() {
        this.charts = {};
        this.refreshInterval = null;
        this.autoRefreshEnabled = true;
        
        this.init();
    }

    /**
     * Inicializa el panel de administración
     */
    init() {
        this.setupEventListeners();
        this.loadSystemOverview();
        this.initializeCharts();
        this.startAutoRefresh();
        this.animateElements();
    }

    /**
     * Configura los event listeners
     */
    setupEventListeners() {
        // Botones principales
        document.getElementById('refreshSystem')?.addEventListener('click', () => {
            this.refreshSystemData();
        });

        document.getElementById('clearCache')?.addEventListener('click', () => {
            this.showConfirmation('¿Limpiar todo el caché del sistema?', () => {
                this.clearSystemCache();
            });
        });

        document.getElementById('optimizeSystem')?.addEventListener('click', () => {
            this.showConfirmation('¿Optimizar el sistema completo?', () => {
                this.optimizeSystem();
            });
        });

        document.getElementById('exportReport')?.addEventListener('click', () => {
            this.exportSystemReport();
        });

        // Botones del panel de control
        document.getElementById('clearCacheBtn')?.addEventListener('click', () => {
            this.clearSystemCache();
        });

        document.getElementById('optimizeDatabase')?.addEventListener('click', () => {
            this.optimizeDatabase();
        });

        document.getElementById('generateReport')?.addEventListener('click', () => {
            this.generateOptimizationReport();
        });

        document.getElementById('cleanupLogs')?.addEventListener('click', () => {
            this.cleanupSystemLogs();
        });

        document.getElementById('systemMaintenance')?.addEventListener('click', () => {
            this.showConfirmation('¿Entrar en modo mantenimiento?', () => {
                this.enableMaintenanceMode();
            });
        });

        // Modal de confirmación
        document.getElementById('confirmAction')?.addEventListener('click', () => {
            if (this.pendingAction) {
                this.pendingAction();
                this.hideConfirmation();
            }
        });
    }

    /**
     * Carga el resumen general del sistema
     */
    async loadSystemOverview() {
        try {
            this.showLoading('Cargando estado del sistema...');

            const response = await fetch('../api/system-monitor.php?accion=system_overview', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const data = await response.json();

            if (data.success) {
                this.updateSystemStatus(data.data);
                this.updateHealthMetrics(data.data.health);
                this.updatePerformanceMetrics(data.data.performance);
                this.updateSecurityInfo(data.data.security);
                this.updateCacheInfo(data.data.cache);
                this.updateSystemInfo(data.data.summary);
            } else {
                throw new Error(data.mensaje || 'Error al cargar datos del sistema');
            }

        } catch (error) {
            console.error('Error cargando sistema:', error);
            this.showError('Error al cargar el estado del sistema');
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Actualiza el estado general del sistema
     */
    updateSystemStatus(data) {
        const statusAlert = document.getElementById('systemAlert');
        const health = data.health;
        
        let alertClass = 'alert-success';
        let statusIcon = 'fas fa-check-circle';
        let statusMessage = 'Sistema funcionando correctamente';

        switch (health.overall_status) {
            case 'error':
                alertClass = 'alert-danger';
                statusIcon = 'fas fa-exclamation-triangle';
                statusMessage = 'Se detectaron problemas críticos en el sistema';
                break;
            case 'warning':
                alertClass = 'alert-warning';
                statusIcon = 'fas fa-exclamation-circle';
                statusMessage = 'Sistema estable con algunas advertencias';
                break;
        }

        statusAlert.className = `alert ${alertClass} d-flex align-items-center`;
        statusAlert.innerHTML = `
            <i class="${statusIcon} me-3"></i>
            <strong>${statusMessage}</strong>
            <div class="ms-auto text-muted">${health.timestamp}</div>
        `;
    }

    /**
     * Actualiza las métricas de salud
     */
    updateHealthMetrics(health) {
        // Estado general
        const overallHealth = document.getElementById('overallHealth');
        const healthDetails = document.getElementById('healthDetails');
        
        if (overallHealth && healthDetails) {
            const statusMap = {
                'ok': { text: 'Excelente', class: 'health-excellent' },
                'warning': { text: 'Advertencia', class: 'health-warning' },
                'error': { text: 'Crítico', class: 'health-critical' }
            };
            
            const status = statusMap[health.overall_status] || statusMap['ok'];
            overallHealth.textContent = status.text;
            overallHealth.className = `health-value ${status.class}`;
            
            const issues = Object.values(health.checks).filter(check => 
                check.status && check.status !== 'ok'
            ).length;
            
            healthDetails.textContent = issues > 0 ? `${issues} problema(s)` : 'Todo OK';
        }
    }

    /**
     * Actualiza las métricas de rendimiento
     */
    updatePerformanceMetrics(performance) {
        // Puntaje de rendimiento
        const performanceScore = document.getElementById('performanceScore');
        const performanceDetails = document.getElementById('performanceDetails');
        
        if (performanceScore && performanceDetails && performance.summary) {
            const score = performance.summary.score || 0;
            performanceScore.textContent = `${score}/100`;
            
            let scoreClass = 'health-excellent';
            if (score < 60) scoreClass = 'health-critical';
            else if (score < 80) scoreClass = 'health-warning';
            else if (score < 90) scoreClass = 'health-good';
            
            performanceScore.className = `health-value ${scoreClass}`;
            performanceDetails.textContent = `${performance.statistics.queries.total_count} consultas`;
        }

        // Métricas detalladas
        const responseTime = document.getElementById('responseTime');
        const queryCount = document.getElementById('queryCount');
        const memoryUsage = document.getElementById('memoryUsage');

        if (responseTime && performance.statistics) {
            responseTime.textContent = `${(performance.statistics.execution.total_time * 1000).toFixed(0)} ms`;
        }

        if (queryCount && performance.statistics) {
            queryCount.textContent = performance.statistics.queries.total_count;
        }

        if (memoryUsage && performance.statistics) {
            memoryUsage.textContent = `${performance.statistics.execution.memory_used_mb} MB`;
        }
    }

    /**
     * Actualiza información de seguridad
     */
    updateSecurityInfo(security) {
        const securityLevel = document.getElementById('securityLevel');
        const securityDetails = document.getElementById('securityDetails');
        const securityEventCount = document.getElementById('securityEventCount');

        if (securityLevel && securityDetails) {
            const levelMap = {
                'secure': { text: 'Seguro', class: 'health-excellent' },
                'low_risk': { text: 'Riesgo Bajo', class: 'health-good' },
                'medium_risk': { text: 'Riesgo Medio', class: 'health-warning' },
                'high_risk': { text: 'Riesgo Alto', class: 'health-critical' }
            };

            const level = levelMap[security.security_level] || levelMap['secure'];
            securityLevel.textContent = level.text;
            securityLevel.className = `health-value ${level.class}`;
            securityDetails.textContent = `${security.recent_events} eventos recientes`;
        }

        if (securityEventCount) {
            securityEventCount.textContent = security.recent_events || '0';
        }

        this.loadSecurityEvents();
    }

    /**
     * Actualiza información de caché
     */
    updateCacheInfo(cache) {
        const cacheItems = document.getElementById('cacheItems');
        const cacheSize = document.getElementById('cacheSize');

        if (cacheItems) {
            cacheItems.textContent = cache.memory_items || '0';
        }

        if (cacheSize) {
            cacheSize.textContent = `${cache.total_size_mb || 0} MB`;
        }
    }

    /**
     * Actualiza información del sistema
     */
    updateSystemInfo(summary) {
        const resourceUsage = document.getElementById('resourceUsage');
        const resourceDetails = document.getElementById('resourceDetails');
        const serverInfo = document.getElementById('serverInfo');
        const systemUptime = document.getElementById('systemUptime');
        const lastUpdate = document.getElementById('lastUpdate');

        if (resourceUsage && resourceDetails) {
            // Simulación del uso de recursos basado en el rendimiento
            const usage = Math.min(100, Math.round((summary.server_load?.['1_min'] || 0) * 100));
            resourceUsage.textContent = `${usage}%`;
            
            let usageClass = 'health-excellent';
            if (usage > 90) usageClass = 'health-critical';
            else if (usage > 70) usageClass = 'health-warning';
            else if (usage > 50) usageClass = 'health-good';
            
            resourceUsage.className = `health-value ${usageClass}`;
            resourceDetails.textContent = 'CPU y Memoria';
        }

        if (serverInfo) {
            serverInfo.textContent = navigator.platform || 'Servidor Web';
        }

        if (systemUptime && summary.uptime) {
            systemUptime.textContent = summary.uptime.formatted || 'N/A';
        }

        if (lastUpdate) {
            lastUpdate.textContent = new Date().toLocaleString('es-ES');
        }
    }

    /**
     * Carga eventos de seguridad
     */
    async loadSecurityEvents() {
        try {
            const response = await fetch('../api/system-monitor.php?accion=security_events&limite=10', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            const data = await response.json();

            if (data.success) {
                this.displaySecurityEvents(data.data.events);
            }

        } catch (error) {
            console.error('Error cargando eventos de seguridad:', error);
        }
    }

    /**
     * Muestra eventos de seguridad
     */
    displaySecurityEvents(events) {
        const container = document.getElementById('securityEvents');
        
        if (!container) return;

        if (events.length === 0) {
            container.innerHTML = `
                <div class="text-center py-4 text-success">
                    <i class="fas fa-shield-alt fa-2x mb-3"></i>
                    <div>No hay eventos de seguridad recientes</div>
                </div>
            `;
            return;
        }

        container.innerHTML = events.map(event => {
            const severity = this.getEventSeverity(event.event);
            const icon = this.getEventIcon(event.event);
            
            return `
                <div class="security-event ${severity}">
                    <div class="security-event-title">
                        <i class="${icon} me-2"></i>
                        ${this.formatEventTitle(event.event)}
                    </div>
                    <div class="security-event-details">
                        ${event.details ? JSON.stringify(event.details).substring(0, 100) + '...' : 'Sin detalles adicionales'}
                    </div>
                    <div class="security-event-time">
                        <i class="fas fa-clock me-1"></i>
                        ${new Date(event.timestamp).toLocaleString('es-ES')}
                    </div>
                </div>
            `;
        }).join('');
    }

    /**
     * Obtiene la severidad de un evento
     */
    getEventSeverity(eventType) {
        const severityMap = {
            'login_failed': 'warning',
            'account_locked': 'critical',
            'rate_limit_exceeded': 'critical',
            'login_success': 'info'
        };
        return severityMap[eventType] || 'info';
    }

    /**
     * Obtiene el icono de un evento
     */
    getEventIcon(eventType) {
        const iconMap = {
            'login_failed': 'fas fa-times-circle',
            'account_locked': 'fas fa-lock',
            'rate_limit_exceeded': 'fas fa-exclamation-triangle',
            'login_success': 'fas fa-check-circle'
        };
        return iconMap[eventType] || 'fas fa-info-circle';
    }

    /**
     * Formatea el título de un evento
     */
    formatEventTitle(eventType) {
        const titleMap = {
            'login_failed': 'Intento de login fallido',
            'account_locked': 'Cuenta bloqueada',
            'rate_limit_exceeded': 'Límite de velocidad excedido',
            'login_success': 'Login exitoso'
        };
        return titleMap[eventType] || eventType;
    }

    /**
     * Carga recomendaciones de optimización
     */
    async loadOptimizationRecommendations() {
        try {
            const response = await fetch('../api/system-monitor.php?accion=optimization_report', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            const data = await response.json();

            if (data.success) {
                this.displayRecommendations(data.data.recommendations);
            }

        } catch (error) {
            console.error('Error cargando recomendaciones:', error);
        }
    }

    /**
     * Muestra recomendaciones de optimización
     */
    displayRecommendations(recommendations) {
        const container = document.getElementById('optimizationRecommendations');
        
        if (!container) return;

        if (recommendations.length === 0) {
            container.innerHTML = `
                <div class="text-center py-4 text-success">
                    <i class="fas fa-check-circle fa-2x mb-3"></i>
                    <div>Sistema optimizado correctamente</div>
                </div>
            `;
            return;
        }

        container.innerHTML = recommendations.map(rec => `
            <div class="recommendation-item ${rec.severity}-priority">
                <div class="recommendation-title">
                    ${this.getPriorityIcon(rec.severity)} ${rec.message}
                </div>
                <div class="recommendation-description">
                    ${rec.value || ''}
                </div>
                <div class="recommendation-action">
                    Acción recomendada: ${this.getActionText(rec.type)}
                </div>
            </div>
        `).join('');
    }

    /**
     * Obtiene icono de prioridad
     */
    getPriorityIcon(severity) {
        const iconMap = {
            'high': '<i class="fas fa-exclamation-triangle text-danger"></i>',
            'medium': '<i class="fas fa-exclamation-circle text-warning"></i>',
            'low': '<i class="fas fa-info-circle text-info"></i>'
        };
        return iconMap[severity] || iconMap['low'];
    }

    /**
     * Obtiene texto de acción
     */
    getActionText(type) {
        const actionMap = {
            'memory': 'Optimizar uso de memoria y implementar paginación',
            'queries': 'Revisar y optimizar consultas de base de datos',
            'cache': 'Implementar más caché en operaciones frecuentes',
            'performance': 'Revisar código y optimizar algoritmos'
        };
        return actionMap[type] || 'Revisar configuración del sistema';
    }

    /**
     * Inicializa gráficos
     */
    initializeCharts() {
        this.initPerformanceChart();
    }

    /**
     * Inicializa gráfico de rendimiento
     */
    initPerformanceChart() {
        const ctx = document.getElementById('performanceChart');
        if (!ctx) return;

        this.charts.performance = new Chart(ctx, {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'Tiempo de Respuesta (ms)',
                    data: [],
                    borderColor: '#3498db',
                    backgroundColor: 'rgba(52, 152, 219, 0.1)',
                    tension: 0.3,
                    fill: true
                }, {
                    label: 'Uso de Memoria (MB)',
                    data: [],
                    borderColor: '#e74c3c',
                    backgroundColor: 'rgba(231, 76, 60, 0.1)',
                    tension: 0.3,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }

    /**
     * Limpia caché del sistema
     */
    async clearSystemCache() {
        try {
            this.showLoading('Limpiando caché...');

            const response = await fetch('../api/system-monitor.php?accion=clear_cache', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            const data = await response.json();

            if (data.success) {
                this.showSuccess('Caché limpiado exitosamente');
                this.refreshSystemData();
            } else {
                throw new Error(data.mensaje);
            }

        } catch (error) {
            console.error('Error limpiando caché:', error);
            this.showError('Error al limpiar el caché');
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Actualiza datos del sistema
     */
    refreshSystemData() {
        this.loadSystemOverview();
        this.loadOptimizationRecommendations();
    }

    /**
     * Auto-refresh del sistema
     */
    startAutoRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
        }

        this.refreshInterval = setInterval(() => {
            if (this.autoRefreshEnabled) {
                this.refreshSystemData();
            }
        }, 60000); // 1 minuto
    }

    /**
     * Anima elementos al cargar
     */
    animateElements() {
        document.querySelectorAll('.health-card').forEach((card, index) => {
            card.style.animationDelay = `${index * 0.1}s`;
            card.classList.add('animate-fadeInUp');
        });

        document.querySelectorAll('.admin-card').forEach((card, index) => {
            card.style.animationDelay = `${(index + 4) * 0.1}s`;
            card.classList.add('animate-slideInLeft');
        });
    }

    /**
     * Muestra confirmación
     */
    showConfirmation(message, callback) {
        const modal = new bootstrap.Modal(document.getElementById('confirmationModal'));
        document.getElementById('confirmationMessage').textContent = message;
        this.pendingAction = callback;
        modal.show();
    }

    /**
     * Oculta confirmación
     */
    hideConfirmation() {
        const modal = bootstrap.Modal.getInstance(document.getElementById('confirmationModal'));
        modal?.hide();
        this.pendingAction = null;
    }

    /**
     * Muestra loading
     */
    showLoading(message) {
        const systemAlert = document.getElementById('systemAlert');
        systemAlert.className = 'alert alert-info d-flex align-items-center';
        systemAlert.innerHTML = `
            <div class="spinner-border spinner-border-sm me-3" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <strong>${message}</strong>
        `;
    }

    /**
     * Oculta loading
     */
    hideLoading() {
        // El loading se oculta cuando se actualiza el estado del sistema
    }

    /**
     * Muestra mensaje de éxito
     */
    showSuccess(message) {
        this.showToast('Éxito', message, 'success');
    }

    /**
     * Muestra mensaje de error
     */
    showError(message) {
        this.showToast('Error', message, 'danger');
    }

    /**
     * Muestra toast
     */
    showToast(title, message, type = 'info') {
        const toastId = 'toast-' + Date.now();
        const icon = type === 'success' ? 'check-circle' : type === 'danger' ? 'exclamation-circle' : 'info-circle';
        
        const toastHtml = `
            <div id="${toastId}" class="toast align-items-center text-bg-${type} border-0" role="alert">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="fas fa-${icon} me-2"></i>
                        <strong>${title}</strong><br>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        `;

        let container = document.querySelector('.toast-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'toast-container position-fixed top-0 end-0 p-3';
            document.body.appendChild(container);
        }

        container.insertAdjacentHTML('beforeend', toastHtml);
        const toastElement = container.lastElementChild;
        const toast = new bootstrap.Toast(toastElement);
        toast.show();

        toastElement.addEventListener('hidden.bs.toast', () => {
            toastElement.remove();
        });
    }
}

/**
 * ========================================
 * INICIALIZACIÓN
 * ========================================
 */
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar panel de administración
    window.adminPanel = new AdminPanelManager();

    // Cargar recomendaciones al cargar la página
    setTimeout(() => {
        window.adminPanel.loadOptimizationRecommendations();
    }, 2000);

    console.log('Panel de Administración iniciado correctamente');
});