// JavaScript para el Dashboard Principal

/**
 * ========================================
 * DASHBOARD MANAGER
 * ========================================
 */
class DashboardManager {
    constructor() {
        this.charts = {};
        this.refreshInterval = null;
        this.autoRefreshEnabled = false;
        this.currentPeriodo = 'semana';
        
        this.init();
    }

    /**
     * Inicializa el dashboard
     */
    init() {
        this.setupEventListeners();
        this.loadDashboardData();
        this.initializeCharts();
        this.startAutoRefresh();
        this.animateElements();
    }

    /**
     * Configura los event listeners
     */
    setupEventListeners() {
        // Cambio de período
        document.querySelectorAll('.periodo-selector').forEach(btn => {
            btn.addEventListener('click', (e) => {
                this.changePeriod(e.target.getAttribute('data-periodo'));
            });
        });

        // Auto-refresh toggle
        const autoRefreshBtn = document.getElementById('autoRefreshBtn');
        if (autoRefreshBtn) {
            autoRefreshBtn.addEventListener('click', () => {
                this.toggleAutoRefresh();
            });
        }

        // Actualizar dashboard
        const refreshBtn = document.getElementById('refreshDashboard');
        if (refreshBtn) {
            refreshBtn.addEventListener('click', () => {
                this.loadDashboardData();
            });
        }

        // Exportar datos
        const exportBtn = document.getElementById('exportData');
        if (exportBtn) {
            exportBtn.addEventListener('click', () => {
                this.exportDashboardData();
            });
        }

        // Acciones rápidas
        this.setupQuickActions();
        
        // Modales
        this.setupModals();
    }

    /**
     * Configura las acciones rápidas
     */
    setupQuickActions() {
        // Registro rápido de utensilio
        document.getElementById('quickAddUtensilio')?.addEventListener('click', () => {
            this.showQuickAddUtensilio();
        });

        // Movimiento rápido
        document.getElementById('quickMovimiento')?.addEventListener('click', () => {
            this.showQuickMovimiento();
        });

        // Reporte rápido
        document.getElementById('quickReporte')?.addEventListener('click', () => {
            this.generateQuickReport();
        });
    }

    /**
     * Configura los modales
     */
    setupModals() {
        // Modal de utensilio rápido
        const modalUtensilio = document.getElementById('quickUtensilioModal');
        if (modalUtensilio) {
            const form = modalUtensilio.querySelector('#quickUtensilioForm');
            form?.addEventListener('submit', (e) => {
                e.preventDefault();
                this.saveQuickUtensilio(form);
            });
        }

        // Modal de movimiento rápido
        const modalMovimiento = document.getElementById('quickMovimientoModal');
        if (modalMovimiento) {
            const form = modalMovimiento.querySelector('#quickMovimientoForm');
            form?.addEventListener('submit', (e) => {
                e.preventDefault();
                this.saveQuickMovimiento(form);
            });
        }
    }

    /**
     * Cambia el período de visualización
     */
    changePeriod(periodo) {
        // Actualizar botones activos
        document.querySelectorAll('.periodo-selector').forEach(btn => {
            btn.classList.remove('active');
        });
        document.querySelector(`[data-periodo="${periodo}"]`)?.classList.add('active');

        this.currentPeriodo = periodo;
        this.loadDashboardData();
        this.updateCharts();
    }

    /**
     * Carga los datos del dashboard
     */
    async loadDashboardData() {
        try {
            this.showLoadingState();

            const response = await fetch(`../api/dashboard.php?accion=obtener_metricas&periodo=${this.currentPeriodo}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const data = await response.json();

            if (data.success) {
                this.updateMetrics(data.metricas);
                this.updateTopUtensilios(data.top_utensilios || []);
                this.updateActivity(data.actividad_reciente || []);
                this.updateChartData(data.graficos || {});
                this.updateAlerts(data.alertas || []);
            } else {
                throw new Error(data.mensaje || 'Error al cargar datos');
            }

        } catch (error) {
            console.error('Error cargando dashboard:', error);
            this.showError('Error al cargar los datos del dashboard');
        } finally {
            this.hideLoadingState();
        }
    }

    /**
     * Actualiza las métricas principales
     */
    updateMetrics(metricas) {
        const metricElements = {
            'total_utensilios': document.getElementById('totalUtensilios'),
            'utensilios_activos': document.getElementById('utensiliosActivos'),
            'movimientos_hoy': document.getElementById('movimientosHoy'),
            'alertas_stock': document.getElementById('alertasStock')
        };

        Object.keys(metricElements).forEach(key => {
            const element = metricElements[key];
            if (element && metricas[key] !== undefined) {
                this.animateCounter(element, metricas[key]);
            }
        });
    }

    /**
     * Anima un contador
     */
    animateCounter(element, targetValue) {
        const currentValue = parseInt(element.textContent) || 0;
        const increment = Math.ceil((targetValue - currentValue) / 20);
        const updateCounter = () => {
            const current = parseInt(element.textContent) || 0;
            if (current < targetValue) {
                element.textContent = Math.min(current + increment, targetValue);
                requestAnimationFrame(updateCounter);
            } else {
                element.textContent = targetValue;
            }
        };
        updateCounter();
    }

    /**
     * Actualiza el top de utensilios
     */
    updateTopUtensilios(utensilios) {
        const container = document.getElementById('topUtensiliosList');
        if (!container) return;

        if (utensilios.length === 0) {
            container.innerHTML = '<div class="text-center text-muted py-3">No hay datos disponibles</div>';
            return;
        }

        container.innerHTML = utensilios.map((utensilio, index) => `
            <div class="list-group-item d-flex align-items-center">
                <div class="rank-badge me-3">
                    <span class="rank-number">${index + 1}</span>
                </div>
                <div class="flex-grow-1">
                    <div class="utensilio-name">${utensilio.nombre}</div>
                    <small class="text-muted">${utensilio.tipo || 'Sin categoría'}</small>
                </div>
                <div class="usage-percentage">
                    <span class="badge bg-primary">${utensilio.total_movimientos || 0}</span>
                </div>
            </div>
        `).join('');
    }

    /**
     * Actualiza la actividad reciente
     */
    updateActivity(actividades) {
        const container = document.getElementById('actividadTimeline');
        if (!container) return;

        if (actividades.length === 0) {
            container.innerHTML = '<div class="text-center text-muted py-3">No hay actividad reciente</div>';
            return;
        }

        container.innerHTML = actividades.map(actividad => {
            const iconClass = this.getActivityIcon(actividad.tipo);
            const colorClass = this.getActivityColor(actividad.tipo);
            
            return `
                <div class="timeline-item">
                    <div class="timeline-marker ${colorClass}">
                        <i class="${iconClass}"></i>
                    </div>
                    <div class="timeline-title">${actividad.descripcion}</div>
                    <div class="timeline-description text-muted">${actividad.detalle || ''}</div>
                    <div class="timeline-meta">
                        <small class="text-muted">
                            <i class="fas fa-clock me-1"></i>
                            ${this.formatDateTime(actividad.fecha)}
                        </small>
                        <small class="text-muted ms-2">
                            <i class="fas fa-user me-1"></i>
                            ${actividad.usuario}
                        </small>
                    </div>
                </div>
            `;
        }).join('');
    }

    /**
     * Obtiene el icono para un tipo de actividad
     */
    getActivityIcon(tipo) {
        const icons = {
            'creacion': 'fas fa-plus',
            'edicion': 'fas fa-edit',
            'eliminacion': 'fas fa-trash',
            'entrada': 'fas fa-arrow-down',
            'salida': 'fas fa-arrow-up',
            'transferencia': 'fas fa-exchange-alt',
            'mantenimiento': 'fas fa-wrench'
        };
        return icons[tipo] || 'fas fa-info-circle';
    }

    /**
     * Obtiene el color para un tipo de actividad
     */
    getActivityColor(tipo) {
        const colors = {
            'creacion': 'bg-success',
            'edicion': 'bg-info',
            'eliminacion': 'bg-danger',
            'entrada': 'bg-primary',
            'salida': 'bg-warning',
            'transferencia': 'bg-secondary',
            'mantenimiento': 'bg-dark'
        };
        return colors[tipo] || 'bg-secondary';
    }

    /**
     * Actualiza las alertas
     */
    updateAlerts(alertas) {
        const container = document.getElementById('alertasContainer');
        if (!container) return;

        if (alertas.length === 0) {
            container.innerHTML = '<div class="alert alert-success alert-sm">✅ No hay alertas pendientes</div>';
            return;
        }

        container.innerHTML = alertas.map(alerta => `
            <div class="alert alert-${alerta.nivel} alert-sm">
                <strong>${alerta.titulo}</strong><br>
                ${alerta.descripcion}
                ${alerta.fecha ? `<br><small>Desde: ${this.formatDate(alerta.fecha)}</small>` : ''}
            </div>
        `).join('');
    }

    /**
     * Inicializa los gráficos
     */
    initializeCharts() {
        this.initMovimientosChart();
        // Aquí se pueden agregar más gráficos
    }

    /**
     * Inicializa el gráfico de movimientos
     */
    initMovimientosChart() {
        const ctx = document.getElementById('movimientosChart');
        if (!ctx) return;

        this.charts.movimientos = new Chart(ctx, {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'Entradas',
                    data: [],
                    borderColor: '#28a745',
                    backgroundColor: 'rgba(40, 167, 69, 0.1)',
                    tension: 0.3,
                    fill: true
                }, {
                    label: 'Salidas',
                    data: [],
                    borderColor: '#dc3545',
                    backgroundColor: 'rgba(220, 53, 69, 0.1)',
                    tension: 0.3,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    title: {
                        display: true,
                        text: 'Movimientos de Inventario'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                },
                interaction: {
                    intersect: false,
                },
                elements: {
                    point: {
                        radius: 4,
                        hoverRadius: 6
                    }
                }
            }
        });
    }

    /**
     * Actualiza los datos de los gráficos
     */
    updateChartData(graficos) {
        if (graficos.movimientos && this.charts.movimientos) {
            this.charts.movimientos.data.labels = graficos.movimientos.labels || [];
            this.charts.movimientos.data.datasets[0].data = graficos.movimientos.entradas || [];
            this.charts.movimientos.data.datasets[1].data = graficos.movimientos.salidas || [];
            this.charts.movimientos.update('active');
        }
    }

    /**
     * Actualiza todos los gráficos
     */
    updateCharts() {
        Object.values(this.charts).forEach(chart => {
            if (chart && chart.update) {
                chart.update();
            }
        });
    }

    /**
     * Muestra el estado de carga
     */
    showLoadingState() {
        document.querySelectorAll('.metric-card').forEach(card => {
            card.classList.add('loading');
        });

        const refreshBtn = document.getElementById('refreshDashboard');
        if (refreshBtn) {
            refreshBtn.disabled = true;
            refreshBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Actualizando...';
        }
    }

    /**
     * Oculta el estado de carga
     */
    hideLoadingState() {
        document.querySelectorAll('.metric-card').forEach(card => {
            card.classList.remove('loading');
        });

        const refreshBtn = document.getElementById('refreshDashboard');
        if (refreshBtn) {
            refreshBtn.disabled = false;
            refreshBtn.innerHTML = '<i class="fas fa-sync-alt me-2"></i>Actualizar';
        }
    }

    /**
     * Toggle del auto-refresh
     */
    toggleAutoRefresh() {
        this.autoRefreshEnabled = !this.autoRefreshEnabled;
        const btn = document.getElementById('autoRefreshBtn');
        
        if (this.autoRefreshEnabled) {
            this.startAutoRefresh();
            btn.innerHTML = '<i class="fas fa-pause me-2"></i>Pausar';
            btn.classList.remove('btn-outline-secondary');
            btn.classList.add('btn-outline-success');
        } else {
            this.stopAutoRefresh();
            btn.innerHTML = '<i class="fas fa-play me-2"></i>Auto';
            btn.classList.remove('btn-outline-success');
            btn.classList.add('btn-outline-secondary');
        }
    }

    /**
     * Inicia el auto-refresh
     */
    startAutoRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
        }
        
        this.refreshInterval = setInterval(() => {
            if (this.autoRefreshEnabled) {
                this.loadDashboardData();
            }
        }, 30000); // 30 segundos
    }

    /**
     * Detiene el auto-refresh
     */
    stopAutoRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
            this.refreshInterval = null;
        }
    }

    /**
     * Anima elementos al cargar
     */
    animateElements() {
        // Animar métricas
        document.querySelectorAll('.metric-card').forEach((card, index) => {
            card.style.animationDelay = `${index * 0.1}s`;
            card.classList.add('fade-in-up');
        });

        // Animar cards
        document.querySelectorAll('.dashboard-card').forEach((card, index) => {
            card.style.animationDelay = `${(index + 4) * 0.1}s`;
            card.classList.add('slide-in-right');
        });
    }

    /**
     * Muestra error
     */
    showError(mensaje) {
        const toast = this.createToast('Error', mensaje, 'danger');
        this.showToast(toast);
    }

    /**
     * Muestra éxito
     */
    showSuccess(mensaje) {
        const toast = this.createToast('Éxito', mensaje, 'success');
        this.showToast(toast);
    }

    /**
     * Crea un toast
     */
    createToast(titulo, mensaje, tipo = 'info') {
        const toastId = 'toast-' + Date.now();
        const icon = tipo === 'success' ? 'check-circle' : tipo === 'danger' ? 'exclamation-circle' : 'info-circle';
        
        return `
            <div id="${toastId}" class="toast align-items-center text-bg-${tipo} border-0" role="alert">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="fas fa-${icon} me-2"></i>
                        <strong>${titulo}</strong><br>
                        ${mensaje}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        `;
    }

    /**
     * Muestra un toast
     */
    showToast(toastHtml) {
        let container = document.querySelector('.toast-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'toast-container position-fixed top-0 end-0 p-3';
            document.body.appendChild(container);
        }

        container.insertAdjacentHTML('beforeend', toastHtml);
        const toastElement = container.lastElementChild;
        const toast = new bootstrap.Toast(toastElement);
        toast.show();

        toastElement.addEventListener('hidden.bs.toast', () => {
            toastElement.remove();
        });
    }

    /**
     * Formatea fecha y hora
     */
    formatDateTime(dateString) {
        const date = new Date(dateString);
        return date.toLocaleString('es-ES', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    /**
     * Formatea fecha
     */
    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('es-ES', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    }

    /**
     * Exporta los datos del dashboard
     */
    async exportDashboardData() {
        try {
            const response = await fetch(`../api/dashboard.php?accion=exportar&periodo=${this.currentPeriodo}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            if (response.ok) {
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `dashboard-${this.currentPeriodo}-${new Date().toISOString().split('T')[0]}.pdf`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
                
                this.showSuccess('Datos exportados correctamente');
            } else {
                throw new Error('Error al exportar datos');
            }
        } catch (error) {
            console.error('Error exportando:', error);
            this.showError('Error al exportar los datos');
        }
    }

    /**
     * Muestra el modal de utensilio rápido
     */
    showQuickAddUtensilio() {
        const modal = new bootstrap.Modal(document.getElementById('quickUtensilioModal'));
        modal.show();
    }

    /**
     * Muestra el modal de movimiento rápido
     */
    showQuickMovimiento() {
        const modal = new bootstrap.Modal(document.getElementById('quickMovimientoModal'));
        modal.show();
    }

    /**
     * Genera un reporte rápido
     */
    generateQuickReport() {
        window.open('../views/reportes.php?quick=true', '_blank');
    }

    /**
     * Guarda utensilio rápido
     */
    async saveQuickUtensilio(form) {
        try {
            const formData = new FormData(form);
            
            const response = await fetch('../api/utensilios.php', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            const result = await response.json();

            if (result.success) {
                this.showSuccess('Utensilio registrado correctamente');
                form.reset();
                bootstrap.Modal.getInstance(document.getElementById('quickUtensilioModal')).hide();
                this.loadDashboardData(); // Actualizar dashboard
            } else {
                throw new Error(result.mensaje || 'Error al registrar utensilio');
            }
        } catch (error) {
            console.error('Error guardando utensilio:', error);
            this.showError('Error al registrar el utensilio');
        }
    }

    /**
     * Guarda movimiento rápido
     */
    async saveQuickMovimiento(form) {
        try {
            const formData = new FormData(form);
            
            const response = await fetch('../api/movimientos.php', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            const result = await response.json();

            if (result.success) {
                this.showSuccess('Movimiento registrado correctamente');
                form.reset();
                bootstrap.Modal.getInstance(document.getElementById('quickMovimientoModal')).hide();
                this.loadDashboardData(); // Actualizar dashboard
            } else {
                throw new Error(result.mensaje || 'Error al registrar movimiento');
            }
        } catch (error) {
            console.error('Error guardando movimiento:', error);
            this.showError('Error al registrar el movimiento');
        }
    }
}

/**
 * ========================================
 * INICIALIZACIÓN
 * ========================================
 */
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar dashboard
    window.dashboard = new DashboardManager();

    // Limpiar intervalos al salir
    window.addEventListener('beforeunload', () => {
        if (window.dashboard) {
            window.dashboard.stopAutoRefresh();
        }
    });

    // Manejar visibilidad de página para pausar/reanudar auto-refresh
    document.addEventListener('visibilitychange', () => {
        if (window.dashboard && window.dashboard.autoRefreshEnabled) {
            if (document.hidden) {
                window.dashboard.stopAutoRefresh();
            } else {
                window.dashboard.startAutoRefresh();
            }
        }
    });

    console.log('Dashboard Manager iniciado correctamente');
});