/**
 * JavaScript adicional para formulario de edición de utensilios
 * Sistema de Inventario de Utensilios de Cocina
 */

document.addEventListener('DOMContentLoaded', function() {
    // Solo ejecutar si estamos en la página de edición
    if (typeof valoresOriginales === 'undefined') {
        return;
    }
    
    // Elementos específicos de edición
    const alertasCambios = document.getElementById('alertasCambios');
    const camposInventario = ['cantidad_total', 'cantidad_disponible'];
    
    // Inicializar funcionalidad de edición
    initChangeDetection();
    initFormReset();
    initWarnings();
    
    /**
     * Inicializa la detección de cambios
     */
    function initChangeDetection() {
        // Monitorear cambios en campos críticos
        camposInventario.forEach(campo => {
            const input = document.getElementById(campo);
            if (input) {
                input.addEventListener('input', detectarCambios);
                input.addEventListener('blur', detectarCambios);
            }
        });
        
        // Monitorear otros campos importantes
        const otrosCampos = ['estado', 'ubicacion', 'precio_unitario'];
        otrosCampos.forEach(campo => {
            const input = document.getElementById(campo);
            if (input) {
                input.addEventListener('input', detectarCambiosGenerales);
                input.addEventListener('change', detectarCambiosGenerales);
            }
        });
    }
    
    /**
     * Detecta cambios en campos de inventario críticos
     */
    function detectarCambios() {
        const cambios = [];
        let hayAlertasCriticas = false;
        
        // Verificar cambios en cantidades
        camposInventario.forEach(campo => {
            const input = document.getElementById(campo);
            const valorActual = parseInt(input.value) || 0;
            const valorOriginal = valoresOriginales[campo];
            
            if (valorActual !== valorOriginal) {
                const diferencia = valorActual - valorOriginal;
                const tipo = diferencia > 0 ? 'incremento' : 'decremento';
                const icono = diferencia > 0 ? 'fa-arrow-up text-success' : 'fa-arrow-down text-danger';
                
                cambios.push({
                    campo: campo,
                    tipo: tipo,
                    diferencia: Math.abs(diferencia),
                    valorOriginal: valorOriginal,
                    valorNuevo: valorActual,
                    icono: icono
                });
                
                if (Math.abs(diferencia) > 10) {
                    hayAlertasCriticas = true;
                }
            }
        });
        
        // Mostrar alertas de cambios
        mostrarAlertasCambios(cambios, hayAlertasCriticas);
        
        // Validar coherencia de datos
        validarCoherenciaDatos();
    }
    
    /**
     * Detecta cambios en otros campos importantes
     */
    function detectarCambiosGenerales() {
        const estadoInput = document.getElementById('estado');
        const ubicacionInput = document.getElementById('ubicacion');
        
        let mensajes = [];
        
        // Verificar cambio de estado
        if (estadoInput && estadoInput.value !== valoresOriginales.estado) {
            const estadoAnterior = valoresOriginales.estado;
            const estadoNuevo = estadoInput.value;
            
            if (estadoAnterior === 'bueno' && (estadoNuevo === 'malo' || estadoNuevo === 'fuera_servicio')) {
                mensajes.push('⚠️ El cambio de estado puede requerir registrar una salida del inventario');
            }
            
            if (estadoNuevo === 'fuera_servicio') {
                mensajes.push('🚫 Los utensilios fuera de servicio no estarán disponibles para uso');
            }
        }
        
        // Verificar cambio de ubicación
        if (ubicacionInput && ubicacionInput.value !== valoresOriginales.ubicacion) {
            mensajes.push('📍 El cambio de ubicación se registrará en el historial de auditoría');
        }
        
        // Mostrar mensajes informativos
        if (mensajes.length > 0) {
            mostrarMensajesInformativos(mensajes);
        }
    }
    
    /**
     * Muestra alertas de cambios en inventario
     */
    function mostrarAlertasCambios(cambios, hayAlertasCriticas) {
        if (cambios.length === 0) {
            alertasCambios.style.display = 'none';
            return;
        }
        
        let html = '';
        
        if (hayAlertasCriticas) {
            html += `
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Cambios importantes detectados:</strong> Se han realizado modificaciones significativas en las cantidades.
                </div>
            `;
        }
        
        html += '<div class="alert alert-info">';
        html += '<i class="fas fa-info-circle me-2"></i>';
        html += '<strong>Cambios detectados:</strong>';
        html += '<ul class="mb-0 mt-2">';
        
        cambios.forEach(cambio => {
            const nombreCampo = cambio.campo === 'cantidad_total' ? 'Cantidad Total' : 'Cantidad Disponible';
            html += `
                <li>
                    <i class="fas ${cambio.icono} me-1"></i>
                    <strong>${nombreCampo}:</strong> ${cambio.valorOriginal} → ${cambio.valorNuevo} 
                    (${cambio.tipo} de ${cambio.diferencia} unidades)
                </li>
            `;
        });
        
        html += '</ul>';
        html += '</div>';
        
        alertasCambios.innerHTML = html;
        alertasCambios.style.display = 'block';
        
        // Scroll suave hacia las alertas
        alertasCambios.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    /**
     * Muestra mensajes informativos temporales
     */
    function mostrarMensajesInformativos(mensajes) {
        // Crear toast informativo
        const toastContainer = document.querySelector('.toast-container') || crearToastContainer();
        
        mensajes.forEach((mensaje, index) => {
            setTimeout(() => {
                const toast = crearToast(mensaje, 'info');
                toastContainer.appendChild(toast);
                
                const bsToast = new bootstrap.Toast(toast, { delay: 4000 });
                bsToast.show();
                
                // Limpiar después de ocultar
                toast.addEventListener('hidden.bs.toast', () => {
                    toast.remove();
                });
            }, index * 500); // Mostrar con delay entre mensajes
        });
    }
    
    /**
     * Crea container de toasts si no existe
     */
    function crearToastContainer() {
        const container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
        return container;
    }
    
    /**
     * Crea un toast personalizado
     */
    function crearToast(mensaje, tipo) {
        const toast = document.createElement('div');
        toast.className = 'toast align-items-center border-0';
        
        const colores = {
            'info': 'text-bg-info',
            'warning': 'text-bg-warning',
            'success': 'text-bg-success',
            'danger': 'text-bg-danger'
        };
        
        toast.classList.add(colores[tipo] || colores.info);
        
        toast.innerHTML = `
            <div class="d-flex">
                <div class="toast-body">
                    ${mensaje}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        `;
        
        return toast;
    }
    
    /**
     * Valida la coherencia de los datos ingresados
     */
    function validarCoherenciaDatos() {
        const cantTotal = parseInt(document.getElementById('cantidad_total').value) || 0;
        const cantDisponible = parseInt(document.getElementById('cantidad_disponible').value) || 0;
        const cantMinima = parseInt(document.getElementById('cantidad_minima').value) || 0;
        
        let errores = [];
        
        // Validar que disponible no sea mayor que total
        if (cantDisponible > cantTotal) {
            errores.push('La cantidad disponible no puede ser mayor que la cantidad total');
        }
        
        // Advertir si cantidad mínima es muy alta
        if (cantMinima > cantTotal && cantTotal > 0) {
            errores.push('La cantidad mínima es mayor que la total, esto generará alertas constantes');
        }
        
        // Advertir sobre cambios drásticos
        const cambioTotal = Math.abs(cantTotal - valoresOriginales.cantidad_total);
        const cambioDisponible = Math.abs(cantDisponible - valoresOriginales.cantidad_disponible);
        
        if (cambioTotal > 50) {
            errores.push(`Cambio significativo en cantidad total: ${cambioTotal} unidades`);
        }
        
        if (cambioDisponible > 50) {
            errores.push(`Cambio significativo en cantidad disponible: ${cambioDisponible} unidades`);
        }
        
        // Mostrar errores si los hay
        mostrarErroresValidacion(errores);
    }
    
    /**
     * Muestra errores de validación
     */
    function mostrarErroresValidacion(errores) {
        // Limpiar errores previos
        const erroresPrevios = document.querySelectorAll('.alerta-validacion-custom');
        erroresPrevios.forEach(el => el.remove());
        
        if (errores.length === 0) {
            return;
        }
        
        const alertaValidacion = document.createElement('div');
        alertaValidacion.className = 'alert alert-danger alerta-validacion-custom mt-3';
        
        let html = '<i class="fas fa-exclamation-triangle me-2"></i>';
        html += '<strong>Problemas detectados:</strong>';
        html += '<ul class="mb-0 mt-2">';
        
        errores.forEach(error => {
            html += `<li>${error}</li>`;
        });
        
        html += '</ul>';
        alertaValidacion.innerHTML = html;
        
        // Insertar después del contenedor de alertas de cambios
        alertasCambios.parentNode.insertBefore(alertaValidacion, alertasCambios.nextSibling);
    }
    
    /**
     * Inicializa el reset personalizado del formulario
     */
    function initFormReset() {
        const form = document.getElementById('formUtensilio');
        const btnReset = form.querySelector('button[type="reset"]');
        
        if (btnReset) {
            btnReset.addEventListener('click', function(e) {
                e.preventDefault();
                
                if (confirm('¿Está seguro que desea deshacer todos los cambios y restaurar los valores originales?')) {
                    restaurarValoresOriginales();
                }
            });
        }
    }
    
    /**
     * Restaura los valores originales del formulario
     */
    function restaurarValoresOriginales() {
        Object.keys(valoresOriginales).forEach(campo => {
            const input = document.getElementById(campo);
            if (input) {
                input.value = valoresOriginales[campo];
                
                // Limpiar estados de validación
                input.classList.remove('is-valid', 'is-invalid');
            }
        });
        
        // Limpiar alertas
        alertasCambios.style.display = 'none';
        const erroresPrevios = document.querySelectorAll('.alerta-validacion-custom');
        erroresPrevios.forEach(el => el.remove());
        
        // Recalcular valor total
        const evento = new Event('input', { bubbles: true });
        document.getElementById('cantidad_total').dispatchEvent(evento);
        document.getElementById('precio_unitario').dispatchEvent(evento);
        
        // Mostrar confirmación
        showSuccess('Valores originales restaurados exitosamente');
    }
    
    /**
     * Inicializa advertencias específicas
     */
    function initWarnings() {
        // Advertencia al salir sin guardar si hay cambios
        let hayFormularioCambiado = false;
        
        // Detectar cambios en cualquier campo
        const todosLosCampos = document.querySelectorAll('#formUtensilio input, #formUtensilio select, #formUtensilio textarea');
        todosLosCampos.forEach(campo => {
            campo.addEventListener('input', () => {
                hayFormularioCambiado = true;
            });
        });
        
        // Reset del flag al enviar el formulario
        document.getElementById('formUtensilio').addEventListener('submit', () => {
            hayFormularioCambiado = false;
        });
        
        // Advertencia antes de salir
        window.addEventListener('beforeunload', (e) => {
            if (hayFormularioCambiado) {
                e.preventDefault();
                e.returnValue = '¿Está seguro que desea salir sin guardar los cambios?';
                return e.returnValue;
            }
        });
        
        // Advertencia en enlaces de navegación
        const linksNavegacion = document.querySelectorAll('a[href]:not([href^="#"])');
        linksNavegacion.forEach(link => {
            link.addEventListener('click', (e) => {
                if (hayFormularioCambiado && !confirm('¿Está seguro que desea salir sin guardar los cambios?')) {
                    e.preventDefault();
                }
            });
        });
    }
    
    /**
     * Funciones de utilidad adicionales
     */
    
    // Comparar valores para determinar tipo de cambio
    function determinarTipoCambio(valorOriginal, valorNuevo) {
        const diferencia = valorNuevo - valorOriginal;
        const porcentaje = valorOriginal > 0 ? Math.abs(diferencia / valorOriginal) * 100 : 100;
        
        if (porcentaje > 50) {
            return 'critico';
        } else if (porcentaje > 20) {
            return 'importante';
        } else {
            return 'menor';
        }
    }
    
    // Validar rangos lógicos
    function validarRangos() {
        const cantTotal = parseInt(document.getElementById('cantidad_total').value) || 0;
        const cantDisponible = parseInt(document.getElementById('cantidad_disponible').value) || 0;
        
        // Auto-ajustar disponible si excede total
        if (cantDisponible > cantTotal) {
            document.getElementById('cantidad_disponible').value = cantTotal;
            showWarning('La cantidad disponible se ajustó automáticamente al total');
        }
    }
    
    // Agregar validación de rangos en tiempo real
    document.getElementById('cantidad_total').addEventListener('blur', validarRangos);
    document.getElementById('cantidad_disponible').addEventListener('blur', validarRangos);
    
    // Ejecutar detección inicial
    detectarCambios();
});