/**
 * JavaScript para formulario de utensilios
 * Sistema de Inventario de Utensilios de Cocina
 */

document.addEventListener('DOMContentLoaded', function() {
    // Elementos del formulario
    const form = document.getElementById('formUtensilio');
    const btnGuardar = document.getElementById('btnGuardar');
    const textoGuardar = document.getElementById('textoGuardar');
    const spinnerGuardar = document.getElementById('spinnerGuardar');
    
    // Campos del formulario
    const cantidadTotal = document.getElementById('cantidad_total');
    const cantidadDisponible = document.getElementById('cantidad_disponible');
    const cantidadMinima = document.getElementById('cantidad_minima');
    const precioUnitario = document.getElementById('precio_unitario');
    const valorTotal = document.getElementById('valorTotal');
    const btnUbicacionActual = document.getElementById('btnUbicacionActual');
    
    // Inicializar funcionalidad
    initFormValidation();
    initFieldInteractions();
    initCalculations();
    initLocationFeatures();
    
    /**
     * Inicializa la validación del formulario
     */
    function initFormValidation() {
        // Validación en tiempo real
        const requiredFields = form.querySelectorAll('[required]');
        requiredFields.forEach(field => {
            field.addEventListener('input', validateField);
            field.addEventListener('blur', validateField);
        });
        
        // Validación personalizada para cantidades
        cantidadDisponible.addEventListener('input', validateCantidades);
        cantidadTotal.addEventListener('input', validateCantidades);
        cantidadMinima.addEventListener('input', validateCantidades);
        
        // Envío del formulario
        form.addEventListener('submit', handleFormSubmit);
    }
    
    /**
     * Valida un campo individual
     */
    function validateField(event) {
        const field = event.target;
        const value = field.value.trim();
        let isValid = true;
        let message = '';
        
        // Validaciones específicas por tipo de campo
        switch (field.type) {
            case 'text':
                if (field.hasAttribute('required') && !value) {
                    isValid = false;
                    message = 'Este campo es requerido';
                } else if (field.hasAttribute('maxlength') && value.length > field.getAttribute('maxlength')) {
                    isValid = false;
                    message = `Máximo ${field.getAttribute('maxlength')} caracteres`;
                }
                break;
                
            case 'number':
                const numValue = parseFloat(value);
                const min = parseFloat(field.getAttribute('min') || '0');
                const max = parseFloat(field.getAttribute('max') || 'Infinity');
                
                if (field.hasAttribute('required') && (!value || isNaN(numValue))) {
                    isValid = false;
                    message = 'Debe ingresar un número válido';
                } else if (numValue < min) {
                    isValid = false;
                    message = `El valor mínimo es ${min}`;
                } else if (numValue > max) {
                    isValid = false;
                    message = `El valor máximo es ${max}`;
                }
                break;
                
            case 'select-one':
                if (field.hasAttribute('required') && !value) {
                    isValid = false;
                    message = 'Debe seleccionar una opción';
                }
                break;
        }
        
        // Aplicar estilos de validación
        updateFieldValidation(field, isValid, message);
        
        return isValid;
    }
    
    /**
     * Valida las relaciones entre cantidades
     */
    function validateCantidades() {
        const total = parseInt(cantidadTotal.value) || 0;
        const disponible = parseInt(cantidadDisponible.value) || 0;
        const minima = parseInt(cantidadMinima.value) || 0;
        
        let isValid = true;
        
        // Cantidad disponible no puede ser mayor que total
        if (disponible > total) {
            updateFieldValidation(cantidadDisponible, false, 'No puede ser mayor que la cantidad total');
            isValid = false;
        } else {
            updateFieldValidation(cantidadDisponible, true);
        }
        
        // Cantidad mínima debe ser al menos 1
        if (minima < 1) {
            updateFieldValidation(cantidadMinima, false, 'La cantidad mínima debe ser al menos 1');
            isValid = false;
        } else {
            updateFieldValidation(cantidadMinima, true);
        }
        
        // Advertencia si cantidad mínima es muy alta
        if (minima > total && total > 0) {
            showWarning('La cantidad mínima es mayor que la cantidad total. Esto generará alertas constantes.');
        }
        
        return isValid;
    }
    
    /**
     * Actualiza el estado visual de validación de un campo
     */
    function updateFieldValidation(field, isValid, message = '') {
        const feedbackEl = field.parentNode.querySelector('.invalid-feedback') || 
                          field.closest('.input-group')?.parentNode.querySelector('.invalid-feedback');
        
        if (isValid) {
            field.classList.remove('is-invalid');
            field.classList.add('is-valid');
            if (feedbackEl) {
                feedbackEl.style.display = 'none';
            }
        } else {
            field.classList.remove('is-valid');
            field.classList.add('is-invalid');
            if (feedbackEl && message) {
                feedbackEl.textContent = message;
                feedbackEl.style.display = 'block';
            }
        }
    }
    
    /**
     * Inicializa interacciones entre campos
     */
    function initFieldInteractions() {
        // Sincronizar cantidad disponible con total cuando se crea un nuevo utensilio
        cantidadTotal.addEventListener('input', function() {
            if (!cantidadDisponible.value || cantidadDisponible.value === '0') {
                cantidadDisponible.value = this.value;
                validateField({ target: cantidadDisponible });
            }
        });
        
        // Auto-completar ubicación basada en el tipo
        const tipoSelect = document.getElementById('tipo');
        const ubicacionInput = document.getElementById('ubicacion');
        
        if (tipoSelect) {
            tipoSelect.addEventListener('change', function() {
                if (!ubicacionInput.value) {
                    const sugerencias = {
                        'cuchillos': 'Estación de Corte',
                        'sartenes': 'Estación de Cocción',
                        'ollas': 'Estación de Cocción',
                        'utensilios_mezcla': 'Estación de Preparación',
                        'moldes': 'Área de Repostería',
                        'electrodomesticos': 'Área de Equipos',
                        'vajilla': 'Área de Servicio',
                        'limpieza': 'Área de Limpieza'
                    };
                    
                    if (sugerencias[this.value]) {
                        ubicacionInput.value = sugerencias[this.value];
                        ubicacionInput.focus();
                        ubicacionInput.select();
                    }
                }
            });
        }
    }
    
    /**
     * Inicializa cálculos automáticos
     */
    function initCalculations() {
        function calcularValorTotal() {
            const cantidad = parseInt(cantidadTotal.value) || 0;
            const precio = parseFloat(precioUnitario.value) || 0;
            const total = cantidad * precio;
            
            valorTotal.textContent = `$${total.toLocaleString('es-CO', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            })}`;
            
            // Cambiar color según el valor
            if (total > 100000) {
                valorTotal.classList.remove('text-success', 'text-warning');
                valorTotal.classList.add('text-danger');
            } else if (total > 50000) {
                valorTotal.classList.remove('text-success', 'text-danger');
                valorTotal.classList.add('text-warning');
            } else {
                valorTotal.classList.remove('text-danger', 'text-warning');
                valorTotal.classList.add('text-success');
            }
        }
        
        cantidadTotal.addEventListener('input', calcularValorTotal);
        precioUnitario.addEventListener('input', calcularValorTotal);
        
        // Cálculo inicial
        calcularValorTotal();
    }
    
    /**
     * Inicializa características de ubicación
     */
    function initLocationFeatures() {
        if (btnUbicacionActual) {
            btnUbicacionActual.addEventListener('click', function() {
                // Simulación de ubicación actual (en un entorno real usaríamos geolocalización)
                const ubicacionesComunes = [
                    'Cocina Principal - Estación A',
                    'Cocina Principal - Estación B',
                    'Área de Preparación',
                    'Despensa Principal',
                    'Almacén de Utensilios'
                ];
                
                const ubicacionAleatoria = ubicacionesComunes[Math.floor(Math.random() * ubicacionesComunes.length)];
                document.getElementById('ubicacion').value = ubicacionAleatoria;
                
                showSuccess('Ubicación sugerida aplicada');
            });
        }
    }
    
    /**
     * Maneja el envío del formulario
     */
    function handleFormSubmit(event) {
        event.preventDefault();
        
        // Validar todo el formulario
        const isValid = validateForm();
        
        if (!isValid) {
            showError('Por favor corrija los errores en el formulario antes de continuar');
            return;
        }
        
        // Mostrar estado de carga
        setLoadingState(true);
        
        // Simular delay para mostrar loading
        setTimeout(() => {
            form.submit();
        }, 500);
    }
    
    /**
     * Valida todo el formulario
     */
    function validateForm() {
        const requiredFields = form.querySelectorAll('[required]');
        let isValid = true;
        
        requiredFields.forEach(field => {
            if (!validateField({ target: field })) {
                isValid = false;
            }
        });
        
        // Validaciones especiales
        if (!validateCantidades()) {
            isValid = false;
        }
        
        return isValid;
    }
    
    /**
     * Establece el estado de carga del formulario
     */
    function setLoadingState(isLoading) {
        if (isLoading) {
            btnGuardar.disabled = true;
            textoGuardar.textContent = 'Guardando...';
            spinnerGuardar.classList.remove('d-none');
            form.classList.add('loading');
        } else {
            btnGuardar.disabled = false;
            textoGuardar.textContent = 'Guardar Utensilio';
            spinnerGuardar.classList.add('d-none');
            form.classList.remove('loading');
        }
    }
    
    /**
     * Resetea el formulario
     */
    form.addEventListener('reset', function() {
        setTimeout(() => {
            // Limpiar estados de validación
            form.querySelectorAll('.is-valid, .is-invalid').forEach(field => {
                field.classList.remove('is-valid', 'is-invalid');
            });
            
            // Recalcular valor total
            valorTotal.textContent = '$0.00';
            valorTotal.className = 'form-control-plaintext fw-bold text-success';
            
            showInfo('Formulario reiniciado');
        }, 10);
    });
    
    /**
     * Funciones de notificación
     */
    function showError(message) {
        showNotification(message, 'error');
    }
    
    function showSuccess(message) {
        showNotification(message, 'success');
    }
    
    function showWarning(message) {
        showNotification(message, 'warning');
    }
    
    function showInfo(message) {
        showNotification(message, 'info');
    }
    
    function showNotification(message, type = 'info') {
        // Remover notificaciones existentes
        const existingNotifications = document.querySelectorAll('.notification-toast');
        existingNotifications.forEach(n => n.remove());
        
        // Crear nueva notificación
        const toast = document.createElement('div');
        toast.className = `notification-toast alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show position-fixed`;
        toast.style.cssText = `
            top: 20px;
            right: 20px;
            z-index: 9999;
            min-width: 300px;
            max-width: 500px;
            animation: slideInRight 0.3s ease;
        `;
        
        const icon = {
            'error': 'fas fa-exclamation-triangle',
            'success': 'fas fa-check-circle',
            'warning': 'fas fa-exclamation-circle',
            'info': 'fas fa-info-circle'
        };
        
        toast.innerHTML = `
            <i class="${icon[type]} me-2"></i>
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        document.body.appendChild(toast);
        
        // Auto-remover después de 5 segundos
        setTimeout(() => {
            if (toast.parentNode) {
                toast.classList.remove('show');
                setTimeout(() => toast.remove(), 150);
            }
        }, 5000);
    }
    
    /**
     * Funciones de utilidad
     */
    
    // Formatear números mientras se escriben
    if (precioUnitario) {
        precioUnitario.addEventListener('input', function() {
            // Permitir solo números y punto decimal
            this.value = this.value.replace(/[^0-9.]/g, '');
            
            // Prevenir múltiples puntos decimales
            const parts = this.value.split('.');
            if (parts.length > 2) {
                this.value = parts[0] + '.' + parts.slice(1).join('');
            }
        });
    }
    
    // Prevenir caracteres especiales en campos numéricos
    const numericFields = [cantidadTotal, cantidadDisponible, cantidadMinima];
    numericFields.forEach(field => {
        if (field) {
            field.addEventListener('keypress', function(e) {
                // Solo permitir números
                if (!/[0-9]/.test(e.key) && !['Backspace', 'Delete', 'Tab', 'Enter', 'ArrowLeft', 'ArrowRight'].includes(e.key)) {
                    e.preventDefault();
                }
            });
        }
    });
    
    // Capitalizar primera letra en campos de texto
    const textFields = ['nombre', 'descripcion', 'ubicacion'];
    textFields.forEach(fieldName => {
        const field = document.getElementById(fieldName);
        if (field && field.type === 'text') {
            field.addEventListener('blur', function() {
                if (this.value) {
                    this.value = this.value.charAt(0).toUpperCase() + this.value.slice(1);
                }
            });
        }
    });
});

// CSS para animaciones
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    .notification-toast {
        border: none !important;
        border-radius: 0.375rem !important;
        box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important;
    }
`;
document.head.appendChild(style);