// JavaScript para la página de login
document.addEventListener('DOMContentLoaded', function() {
    
    // Referencias a elementos
    const loginForm = document.getElementById('loginForm');
    const togglePassword = document.getElementById('togglePassword');
    const passwordInput = document.getElementById('password');
    const btnText = document.getElementById('btnText');
    const btnSpinner = document.getElementById('btnSpinner');
    
    // Mostrar/ocultar contraseña
    if (togglePassword && passwordInput) {
        togglePassword.addEventListener('click', function() {
            const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
            passwordInput.setAttribute('type', type);
            
            const icon = this.querySelector('i');
            if (type === 'text') {
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        });
    }
    
    // Manejo del formulario
    if (loginForm) {
        loginForm.addEventListener('submit', function(e) {
            const email = document.getElementById('email').value.trim();
            const password = document.getElementById('password').value;
            
            // Validaciones básicas
            if (!email || !password) {
                e.preventDefault();
                showAlert('Por favor, complete todos los campos', 'danger');
                return false;
            }
            
            if (!isValidEmail(email)) {
                e.preventDefault();
                showAlert('Por favor, ingrese un email válido', 'danger');
                return false;
            }
            
            // Mostrar loading
            showLoading(true);
        });
    }
    
    // Auto-complete para usuarios de prueba (solo en desarrollo)
    if (window.location.hostname === 'localhost') {
        addTestUserClickHandlers();
    }
    
    // Auto-dismiss alerts después de 5 segundos
    setTimeout(function() {
        const alerts = document.querySelectorAll('.alert');
        alerts.forEach(function(alert) {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        });
    }, 5000);
    
    // Focus en el primer campo
    const emailInput = document.getElementById('email');
    if (emailInput) {
        emailInput.focus();
    }
});

/**
 * Valida formato de email
 */
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

/**
 * Muestra/oculta el indicador de carga
 */
function showLoading(show) {
    const btnText = document.getElementById('btnText');
    const btnSpinner = document.getElementById('btnSpinner');
    const submitBtn = document.querySelector('button[type="submit"]');
    
    if (show) {
        btnText.textContent = 'Iniciando...';
        btnSpinner.classList.remove('d-none');
        submitBtn.disabled = true;
    } else {
        btnText.textContent = 'Iniciar Sesión';
        btnSpinner.classList.add('d-none');
        submitBtn.disabled = false;
    }
}

/**
 * Muestra una alerta
 */
function showAlert(message, type = 'info') {
    const alertHtml = `
        <div class="alert alert-${type} alert-dismissible fade show" role="alert">
            <i class="fas fa-${getAlertIcon(type)} me-2"></i>
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `;
    
    const form = document.getElementById('loginForm');
    form.insertAdjacentHTML('beforebegin', alertHtml);
}

/**
 * Obtiene el icono según el tipo de alerta
 */
function getAlertIcon(type) {
    const icons = {
        'success': 'check-circle',
        'danger': 'exclamation-triangle',
        'warning': 'exclamation-circle',
        'info': 'info-circle'
    };
    return icons[type] || 'info-circle';
}

/**
 * Agrega handlers para usuarios de prueba
 */
function addTestUserClickHandlers() {
    const testUsers = document.querySelectorAll('.row.text-center .col-4');
    
    testUsers.forEach(function(userDiv, index) {
        userDiv.style.cursor = 'pointer';
        userDiv.addEventListener('click', function() {
            const emailInput = document.getElementById('email');
            const passwordInput = document.getElementById('password');
            
            const credentials = [
                { email: 'admin@sistema.com', password: 'admin123' },
                { email: 'encargado@cocina.com', password: 'encargado123' },
                { email: 'auditor@sistema.com', password: 'auditor123' }
            ];
            
            if (credentials[index]) {
                emailInput.value = credentials[index].email;
                passwordInput.value = credentials[index].password;
                
                // Efecto visual
                this.style.backgroundColor = 'rgba(102, 126, 234, 0.2)';
                setTimeout(() => {
                    this.style.backgroundColor = '';
                }, 200);
            }
        });
    });
}

/**
 * Manejo de teclado
 */
document.addEventListener('keydown', function(e) {
    // Enter para enviar formulario
    if (e.key === 'Enter') {
        const activeElement = document.activeElement;
        if (activeElement && (activeElement.id === 'email' || activeElement.id === 'password')) {
            const form = document.getElementById('loginForm');
            if (form) {
                form.submit();
            }
        }
    }
    
    // Escape para limpiar campos
    if (e.key === 'Escape') {
        document.getElementById('email').value = '';
        document.getElementById('password').value = '';
        document.getElementById('email').focus();
    }
});

/**
 * Validación en tiempo real
 */
document.getElementById('email')?.addEventListener('blur', function() {
    const email = this.value.trim();
    if (email && !isValidEmail(email)) {
        this.classList.add('is-invalid');
    } else {
        this.classList.remove('is-invalid');
    }
});

/**
 * Limpiar validación al escribir
 */
document.getElementById('email')?.addEventListener('input', function() {
    this.classList.remove('is-invalid');
});