/**
 * JavaScript para módulo de movimientos
 * Sistema de Inventario de Utensilios de Cocina
 */

document.addEventListener('DOMContentLoaded', function() {
    // Elementos principales
    const formFiltros = document.getElementById('formFiltros');
    const modalMovimiento = new bootstrap.Modal(document.getElementById('modalMovimiento'));
    const modalDetalleMovimiento = new bootstrap.Modal(document.getElementById('modalDetalleMovimiento'));
    const formMovimiento = document.getElementById('formMovimiento');
    
    // Campos del formulario de movimiento
    const tipoMovimientoInput = document.getElementById('tipoMovimiento');
    const utensilioSelect = document.getElementById('utensilioMovimiento');
    const cantidadInput = document.getElementById('cantidadMovimiento');
    const motivoSelect = document.getElementById('motivoMovimiento');
    const ayudaCantidad = document.getElementById('ayudaCantidad');
    const infoUtensilio = document.getElementById('infoUtensilio');
    const detalleUtensilio = document.getElementById('detalleUtensilio');
    
    // Inicializar funcionalidades
    initFiltros();
    initMovimientos();
    initNotifications();
    
    /**
     * Inicializa el sistema de filtros
     */
    function initFiltros() {
        // Auto-envío de filtros al cambiar
        const camposFiltros = formFiltros.querySelectorAll('input, select');
        camposFiltros.forEach(campo => {
            campo.addEventListener('change', function() {
                if (this.type !== 'text') {
                    formFiltros.submit();
                }
            });
            
            // Para campos de texto, enviar después de una pausa
            if (campo.type === 'text') {
                let timeoutId;
                campo.addEventListener('input', function() {
                    clearTimeout(timeoutId);
                    timeoutId = setTimeout(() => {
                        formFiltros.submit();
                    }, 1000);
                });
            }
        });
        
        // Botón limpiar filtros
        document.getElementById('btnLimpiarFiltros').addEventListener('click', function() {
            camposFiltros.forEach(campo => {
                if (campo.type === 'text' || campo.type === 'date') {
                    campo.value = '';
                } else if (campo.tagName === 'SELECT') {
                    campo.selectedIndex = 0;
                }
            });
            formFiltros.submit();
        });
    }
    
    /**
     * Inicializa el sistema de movimientos
     */
    function initMovimientos() {
        // Evento de envío del formulario
        formMovimiento.addEventListener('submit', handleMovimientoSubmit);
        
        // Cambios en selección de utensilio
        utensilioSelect.addEventListener('change', handleUtensilioChange);
        
        // Validación en tiempo real de cantidad
        cantidadInput.addEventListener('input', validateCantidad);
        
        // Exponer funciones globalmente
        window.abrirModalMovimiento = abrirModalMovimiento;
        window.verDetalleMovimiento = verDetalleMovimiento;
        window.corregirMovimiento = corregirMovimiento;
        window.exportarMovimientos = exportarMovimientos;
        window.actualizarLista = actualizarLista;
    }
    
    /**
     * Abre el modal para registrar movimiento
     */
    function abrirModalMovimiento(tipo) {
        if (!USUARIO_PUEDE_GESTIONAR) {
            showError('No tiene permisos para registrar movimientos');
            return;
        }
        
        // Configurar modal según tipo
        tipoMovimientoInput.value = tipo;
        
        const iconoTipo = document.getElementById('iconoTipoMovimiento');
        const textoTipo = document.getElementById('textoTipoMovimiento');
        const textoBoton = document.getElementById('textoBotonMovimiento');
        const btnConfirmar = document.getElementById('btnConfirmarMovimiento');
        
        if (tipo === 'entrada') {
            iconoTipo.innerHTML = '<i class="fas fa-plus text-success me-2"></i>';
            textoTipo.textContent = 'Entrada';
            textoBoton.textContent = 'Registrar Entrada';
            btnConfirmar.className = 'btn btn-success';
            cantidadInput.max = 999999; // Sin límite para entradas
        } else {
            iconoTipo.innerHTML = '<i class="fas fa-minus text-danger me-2"></i>';
            textoTipo.textContent = 'Salida';
            textoBoton.textContent = 'Registrar Salida';
            btnConfirmar.className = 'btn btn-danger';
        }
        
        // Limpiar formulario
        formMovimiento.reset();
        tipoMovimientoInput.value = tipo;
        infoUtensilio.classList.add('d-none');
        
        // Cargar motivos según tipo
        cargarMotivos(tipo);
        
        // Mostrar modal
        modalMovimiento.show();
        
        // Focus en utensilio
        setTimeout(() => {
            utensilioSelect.focus();
        }, 300);
    }
    
    /**
     * Carga los motivos según el tipo de movimiento
     */
    async function cargarMotivos(tipo) {
        try {
            const response = await fetch(`${BASE_URL}/api/movimientos/motivos/${tipo}`);
            const motivos = await response.json();
            
            motivoSelect.innerHTML = '<option value="">Seleccionar motivo...</option>';
            
            motivos.forEach(motivo => {
                const option = document.createElement('option');
                option.value = motivo;
                option.textContent = motivo;
                motivoSelect.appendChild(option);
            });
            
        } catch (error) {
            console.error('Error cargando motivos:', error);
            
            // Motivos por defecto
            const motivosPorDefecto = {
                'entrada': [
                    'Compra nueva',
                    'Reposición',
                    'Devolución',
                    'Transferencia desde otra área',
                    'Reparación completada'
                ],
                'salida': [
                    'Uso en servicio',
                    'Préstamo',
                    'Transferencia a otra área',
                    'Reparación/Mantenimiento',
                    'Deterioro/Pérdida',
                    'Descarte'
                ]
            };
            
            motivoSelect.innerHTML = '<option value="">Seleccionar motivo...</option>';
            motivosPorDefecto[tipo].forEach(motivo => {
                const option = document.createElement('option');
                option.value = motivo;
                option.textContent = motivo;
                motivoSelect.appendChild(option);
            });
        }
    }
    
    /**
     * Maneja el cambio de selección de utensilio
     */
    function handleUtensilioChange() {
        const selectedOption = utensilioSelect.selectedOptions[0];
        
        if (!selectedOption || !selectedOption.value) {
            infoUtensilio.classList.add('d-none');
            cantidadInput.max = '';
            ayudaCantidad.textContent = '';
            return;
        }
        
        const disponible = parseInt(selectedOption.dataset.disponible) || 0;
        const total = parseInt(selectedOption.dataset.total) || 0;
        const ubicacion = selectedOption.dataset.ubicacion || '';
        const nombre = selectedOption.textContent.split(' (')[0];
        
        // Configurar límites según tipo de movimiento
        const tipo = tipoMovimientoInput.value;
        
        if (tipo === 'salida') {
            cantidadInput.max = disponible;
            ayudaCantidad.textContent = `Máximo disponible: ${disponible}`;
            ayudaCantidad.className = 'form-text text-muted';
            
            if (disponible === 0) {
                ayudaCantidad.textContent = 'No hay stock disponible para salida';
                ayudaCantidad.className = 'form-text text-danger';
                cantidadInput.disabled = true;
            } else {
                cantidadInput.disabled = false;
            }
        } else {
            cantidadInput.max = 999999;
            ayudaCantidad.textContent = 'Sin límite para entradas';
            ayudaCantidad.className = 'form-text text-success';
            cantidadInput.disabled = false;
        }
        
        // Mostrar información del utensilio
        const porcentajeUso = total > 0 ? ((total - disponible) / total * 100).toFixed(1) : 0;
        
        detalleUtensilio.innerHTML = `
            <div class="row">
                <div class="col-md-6">
                    <strong>Total:</strong> ${total} unidades<br>
                    <strong>Disponible:</strong> <span class="text-success">${disponible}</span><br>
                    <strong>En uso:</strong> <span class="text-warning">${total - disponible}</span>
                </div>
                <div class="col-md-6">
                    <strong>Ubicación:</strong> ${ubicacion}<br>
                    <strong>Uso:</strong> ${porcentajeUso}%<br>
                    ${disponible <= 5 && disponible > 0 ? '<span class="badge bg-warning text-dark">Stock bajo</span>' : ''}
                    ${disponible === 0 ? '<span class="badge bg-danger">Sin stock</span>' : ''}
                </div>
            </div>
        `;
        
        infoUtensilio.classList.remove('d-none');
    }
    
    /**
     * Valida la cantidad ingresada
     */
    function validateCantidad() {
        const cantidad = parseInt(cantidadInput.value);
        const max = parseInt(cantidadInput.max);
        const tipo = tipoMovimientoInput.value;
        
        if (!cantidad || cantidad <= 0) {
            cantidadInput.classList.remove('is-valid', 'is-invalid');
            return;
        }
        
        if (tipo === 'salida' && cantidad > max) {
            cantidadInput.classList.add('is-invalid');
            cantidadInput.classList.remove('is-valid');
            ayudaCantidad.textContent = `Cantidad excede el stock disponible (${max})`;
            ayudaCantidad.className = 'form-text text-danger';
        } else {
            cantidadInput.classList.add('is-valid');
            cantidadInput.classList.remove('is-invalid');
            
            if (tipo === 'salida') {
                ayudaCantidad.textContent = `Máximo disponible: ${max}`;
                ayudaCantidad.className = 'form-text text-muted';
            } else {
                ayudaCantidad.textContent = 'Sin límite para entradas';
                ayudaCantidad.className = 'form-text text-success';
            }
        }
    }
    
    /**
     * Maneja el envío del formulario de movimiento
     */
    function handleMovimientoSubmit(event) {
        event.preventDefault();
        
        // Validar formulario
        if (!validarFormularioMovimiento()) {
            return;
        }
        
        const formData = new FormData(formMovimiento);
        const tipo = formData.get('tipo_movimiento');
        const cantidad = parseInt(formData.get('cantidad'));
        const utensilio = utensilioSelect.selectedOptions[0].textContent.split(' (')[0];
        
        // Confirmación
        const mensaje = `¿Confirmar ${tipo} de ${cantidad} unidades de "${utensilio}"?`;
        if (!confirm(mensaje)) {
            return;
        }
        
        // Enviar datos
        enviarMovimiento(formData);
    }
    
    /**
     * Valida el formulario de movimiento
     */
    function validarFormularioMovimiento() {
        const utensilio = utensilioSelect.value;
        const cantidad = parseInt(cantidadInput.value);
        const motivo = motivoSelect.value;
        
        if (!utensilio) {
            showError('Debe seleccionar un utensilio');
            utensilioSelect.focus();
            return false;
        }
        
        if (!cantidad || cantidad <= 0) {
            showError('Debe ingresar una cantidad válida');
            cantidadInput.focus();
            return false;
        }
        
        if (!motivo) {
            showError('Debe seleccionar un motivo');
            motivoSelect.focus();
            return false;
        }
        
        // Validar límites
        const tipo = tipoMovimientoInput.value;
        const max = parseInt(cantidadInput.max);
        
        if (tipo === 'salida' && cantidad > max) {
            showError(`La cantidad no puede exceder el stock disponible (${max})`);
            cantidadInput.focus();
            return false;
        }
        
        return true;
    }
    
    /**
     * Envía el movimiento al servidor
     */
    async function enviarMovimiento(formData) {
        const btnConfirmar = document.getElementById('btnConfirmarMovimiento');
        const textoBoton = document.getElementById('textoBotonMovimiento');
        const spinner = document.getElementById('spinnerMovimiento');
        
        try {
            // Mostrar loading
            btnConfirmar.disabled = true;
            textoBoton.textContent = 'Procesando...';
            spinner.classList.remove('d-none');
            
            const response = await fetch(`${BASE_URL}/api/movimientos/registrar`, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                showSuccess('Movimiento registrado exitosamente');
                modalMovimiento.hide();
                
                // Recargar página después de un breve delay
                setTimeout(() => {
                    location.reload();
                }, 1500);
            } else {
                showError(result.message || 'Error al registrar el movimiento');
            }
            
        } catch (error) {
            console.error('Error:', error);
            showError('Error de conexión. Intente nuevamente.');
        } finally {
            // Restaurar botón
            btnConfirmar.disabled = false;
            const tipo = tipoMovimientoInput.value;
            textoBoton.textContent = tipo === 'entrada' ? 'Registrar Entrada' : 'Registrar Salida';
            spinner.classList.add('d-none');
        }
    }
    
    /**
     * Ver detalle de un movimiento
     */
    async function verDetalleMovimiento(movimientoId) {
        const contenido = document.getElementById('contenidoDetalleMovimiento');
        
        try {
            showLoading('Cargando detalle...');
            
            const response = await fetch(`${BASE_URL}/api/movimientos/${movimientoId}`);
            const movimiento = await response.json();
            
            if (movimiento.success) {
                const mov = movimiento.data;
                
                contenido.innerHTML = `
                    <dl class="row">
                        <dt class="col-sm-4">ID:</dt>
                        <dd class="col-sm-8">${mov.id}</dd>
                        
                        <dt class="col-sm-4">Fecha:</dt>
                        <dd class="col-sm-8">${new Date(mov.fecha_movimiento).toLocaleString('es-CO')}</dd>
                        
                        <dt class="col-sm-4">Tipo:</dt>
                        <dd class="col-sm-8">
                            <span class="badge bg-${mov.tipo_movimiento === 'entrada' ? 'success' : 'danger'}">
                                ${mov.tipo_movimiento === 'entrada' ? 'Entrada' : 'Salida'}
                            </span>
                        </dd>
                        
                        <dt class="col-sm-4">Utensilio:</dt>
                        <dd class="col-sm-8">
                            <strong>${mov.utensilio_nombre}</strong><br>
                            <small class="text-muted">${mov.utensilio_tipo}</small>
                        </dd>
                        
                        <dt class="col-sm-4">Cantidad:</dt>
                        <dd class="col-sm-8">
                            <strong class="text-${mov.tipo_movimiento === 'entrada' ? 'success' : 'danger'}">
                                ${mov.tipo_movimiento === 'entrada' ? '+' : '-'}${mov.cantidad}
                            </strong>
                        </dd>
                        
                        <dt class="col-sm-4">Motivo:</dt>
                        <dd class="col-sm-8"><strong>${mov.motivo}</strong></dd>
                        
                        ${mov.observaciones ? `
                            <dt class="col-sm-4">Observaciones:</dt>
                            <dd class="col-sm-8">${mov.observaciones}</dd>
                        ` : ''}
                        
                        <dt class="col-sm-4">Usuario:</dt>
                        <dd class="col-sm-8">${mov.usuario_nombre}</dd>
                        
                        ${mov.ubicacion_destino ? `
                            <dt class="col-sm-4">Ubicación destino:</dt>
                            <dd class="col-sm-8">${mov.ubicacion_destino}</dd>
                        ` : ''}
                    </dl>
                `;
                
                modalDetalleMovimiento.show();
            } else {
                showError('Error al cargar el detalle del movimiento');
            }
            
        } catch (error) {
            console.error('Error:', error);
            showError('Error de conexión al cargar el detalle');
        } finally {
            hideLoading();
        }
    }
    
    /**
     * Corregir un movimiento
     */
    function corregirMovimiento(movimientoId) {
        const observacion = prompt('Ingrese la observación de corrección:');
        
        if (!observacion || !observacion.trim()) {
            return;
        }
        
        // Aquí implementarías la lógica de corrección
        showWarning('Funcionalidad de corrección en desarrollo');
    }
    
    /**
     * Exportar movimientos
     */
    function exportarMovimientos() {
        showInfo('Funcionalidad de exportación en desarrollo');
    }
    
    /**
     * Actualizar lista de movimientos
     */
    function actualizarLista() {
        location.reload();
    }
    
    /**
     * Sistema de notificaciones
     */
    function initNotifications() {
        // Crear container de toasts si no existe
        if (!document.querySelector('.toast-container')) {
            const container = document.createElement('div');
            container.className = 'toast-container position-fixed top-0 end-0 p-3';
            container.style.zIndex = '9999';
            document.body.appendChild(container);
        }
    }
    
    function showNotification(message, type = 'info', duration = 5000) {
        const container = document.querySelector('.toast-container');
        
        const icons = {
            'success': 'fas fa-check-circle',
            'error': 'fas fa-exclamation-triangle',
            'warning': 'fas fa-exclamation-circle',
            'info': 'fas fa-info-circle'
        };
        
        const colors = {
            'success': 'text-bg-success',
            'error': 'text-bg-danger',
            'warning': 'text-bg-warning',
            'info': 'text-bg-info'
        };
        
        const toastId = 'toast-' + Date.now();
        
        const toastHTML = `
            <div id="${toastId}" class="toast align-items-center ${colors[type]} border-0" role="alert">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="${icons[type]} me-2"></i>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        `;
        
        container.insertAdjacentHTML('beforeend', toastHTML);
        
        const toastElement = document.getElementById(toastId);
        const toast = new bootstrap.Toast(toastElement, { delay: duration });
        toast.show();
        
        // Limpiar después de que se oculte
        toastElement.addEventListener('hidden.bs.toast', () => {
            toastElement.remove();
        });
    }
    
    function showSuccess(message) {
        showNotification(message, 'success');
    }
    
    function showError(message) {
        showNotification(message, 'error');
    }
    
    function showWarning(message) {
        showNotification(message, 'warning');
    }
    
    function showInfo(message) {
        showNotification(message, 'info');
    }
    
    /**
     * Loading overlay
     */
    function showLoading(message = 'Cargando...') {
        // Remover overlay existente
        hideLoading();
        
        const overlay = document.createElement('div');
        overlay.id = 'loading-overlay';
        overlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.1rem;
        `;
        
        overlay.innerHTML = `
            <div class="text-center">
                <div class="spinner-border mb-3" role="status">
                    <span class="visually-hidden">Cargando...</span>
                </div>
                <div>${message}</div>
            </div>
        `;
        
        document.body.appendChild(overlay);
    }
    
    function hideLoading() {
        const overlay = document.getElementById('loading-overlay');
        if (overlay) {
            overlay.remove();
        }
    }
    
    // Exponer funciones globalmente
    window.showSuccess = showSuccess;
    window.showError = showError;
    window.showWarning = showWarning;
    window.showInfo = showInfo;
    window.showLoading = showLoading;
    window.hideLoading = hideLoading;
    
    // Shortcuts de teclado
    document.addEventListener('keydown', function(e) {
        // Ctrl + E para registrar entrada
        if (e.ctrlKey && e.key === 'e' && USUARIO_PUEDE_GESTIONAR) {
            e.preventDefault();
            abrirModalMovimiento('entrada');
        }
        
        // Ctrl + S para registrar salida
        if (e.ctrlKey && e.key === 's' && USUARIO_PUEDE_GESTIONAR) {
            e.preventDefault();
            abrirModalMovimiento('salida');
        }
        
        // F5 para actualizar (personalizado)
        if (e.key === 'F5') {
            e.preventDefault();
            actualizarLista();
        }
    });
});