/**
 * JavaScript para el Sistema de Reportes
 * Maneja la generación, visualización y exportación de reportes
 */

// Estado global del módulo
const ReportesApp = {
    modal: null,
    currentReport: null,
    
    // Inicialización
    init() {
        this.modal = new bootstrap.Modal(document.getElementById('reporteModal'));
        this.bindEvents();
        this.loadReportesRecientes();
        
        console.log('Sistema de Reportes inicializado');
    },
    
    // Vinculación de eventos
    bindEvents() {
        // Eventos del formulario de reportes
        const tipoSelect = document.querySelector('select[name="tipo_reporte"]');
        if (tipoSelect) {
            tipoSelect.addEventListener('change', this.handleTipoChange.bind(this));
        }
        
        // Atajos de teclado
        document.addEventListener('keydown', this.handleKeyboard.bind(this));
        
        // Eventos de clicks en cards
        document.querySelectorAll('.report-type-card').forEach(card => {
            card.addEventListener('click', (e) => {
                e.preventDefault();
            });
        });
    },
    
    // Manejo de cambios en tipo de reporte
    handleTipoChange(event) {
        const tipo = event.target.value;
        const periodoFields = document.querySelector('.periodo-fields');
        const filtrosFields = document.querySelector('.filtros-adicionales');
        
        // Mostrar/ocultar campos según el tipo
        if (tipo === 'movimientos' || tipo === 'usuarios' || tipo === 'ejecutivo') {
            periodoFields.style.display = 'block';
        } else {
            periodoFields.style.display = 'none';
        }
        
        if (tipo === 'inventario' || tipo === 'alertas') {
            filtrosFields.style.display = 'block';
        } else {
            filtrosFields.style.display = 'none';
        }
        
        // Configurar fechas por defecto
        this.setDefaultDates(tipo);
    },
    
    // Establecer fechas por defecto según el tipo
    setDefaultDates(tipo) {
        const fechaInicio = document.querySelector('input[name="fecha_inicio"]');
        const fechaFin = document.querySelector('input[name="fecha_fin"]');
        
        const hoy = new Date();
        let inicio = new Date();
        
        switch (tipo) {
            case 'ejecutivo':
                inicio.setDate(hoy.getDate() - 30);
                break;
            case 'movimientos':
                inicio.setDate(hoy.getDate() - 7);
                break;
            case 'usuarios':
                inicio.setDate(hoy.getDate() - 30);
                break;
            default:
                inicio.setDate(hoy.getDate() - 7);
        }
        
        fechaInicio.value = inicio.toISOString().split('T')[0];
        fechaFin.value = hoy.toISOString().split('T')[0];
    },
    
    // Manejo de atajos de teclado
    handleKeyboard(event) {
        if (event.ctrlKey) {
            switch (event.key) {
                case 'r':
                    event.preventDefault();
                    this.generarReporteRapido();
                    break;
                case 'e':
                    event.preventDefault();
                    this.exportarReporteActual();
                    break;
                case 'p':
                    event.preventDefault();
                    this.programarReporte();
                    break;
            }
        }
        
        if (event.key === 'Escape' && this.modal._isShown) {
            this.modal.hide();
        }
    }
};

// Funciones principales de reportes
function generarReporte(tipo) {
    const modal = document.getElementById('reporteModal');
    const tipoSelect = modal.querySelector('select[name="tipo_reporte"]');
    
    tipoSelect.value = tipo;
    tipoSelect.dispatchEvent(new Event('change'));
    
    ReportesApp.modal.show();
}

function generarReporteRapido() {
    ReportesApp.modal.show();
}

function programarReporte() {
    showNotification('Función de programación en desarrollo', 'info');
}

function procesarReporte() {
    const form = document.getElementById('reporteForm');
    const formData = new FormData(form);
    
    // Validar formulario
    if (!form.checkValidity()) {
        form.reportValidity();
        return;
    }
    
    const data = Object.fromEntries(formData.entries());
    
    // Mostrar loading
    showLoading('Generando reporte...');
    
    // Simular generación del reporte
    setTimeout(() => {
        generarReporteAjax(data)
            .then(result => {
                hideLoading();
                ReportesApp.modal.hide();
                
                if (data.formato === 'html') {
                    mostrarReporteHTML(result);
                } else {
                    descargarReporte(result, data.formato);
                }
                
                showNotification('Reporte generado exitosamente', 'success');
                actualizarReportesRecientes(data);
            })
            .catch(error => {
                hideLoading();
                showNotification('Error al generar reporte: ' + error.message, 'error');
                console.error('Error generando reporte:', error);
            });
    }, 1000);
}

// Función AJAX para generar reportes
async function generarReporteAjax(data) {
    const response = await fetch(`${BASE_URL}/api/reportes/generar`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': getCSRFToken()
        },
        body: JSON.stringify(data)
    });
    
    if (!response.ok) {
        throw new Error(`Error HTTP: ${response.status}`);
    }
    
    return await response.json();
}

// Mostrar reporte en HTML
function mostrarReporteHTML(reporte) {
    // Crear nueva ventana o tab para mostrar el reporte
    const ventanaReporte = window.open('', '_blank');
    
    if (!ventanaReporte) {
        showNotification('Por favor, permita ventanas emergentes para ver el reporte', 'warning');
        return;
    }
    
    // Generar HTML del reporte
    const htmlContent = generarHTMLReporte(reporte);
    
    ventanaReporte.document.write(htmlContent);
    ventanaReporte.document.close();
}

// Generar HTML del reporte
function generarHTMLReporte(reporte) {
    return `
    <!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>${reporte.titulo} - ${APP_NAME}</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
        <link href="${BASE_URL}/assets/css/reportes.css" rel="stylesheet">
        <style>
            body { padding: 20px; }
            .report-header { margin-bottom: 30px; border-bottom: 2px solid #dee2e6; padding-bottom: 20px; }
            .report-section { margin-bottom: 30px; }
            @media print { body { padding: 0; } }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="report-header text-center">
                <h1>${reporte.titulo}</h1>
                <p class="text-muted">${reporte.descripcion}</p>
                <small>Generado el ${new Date().toLocaleString('es-ES')}</small>
            </div>
            
            <div class="report-content">
                ${reporte.contenido}
            </div>
        </div>
        
        <script>
            // Auto-imprimir si está configurado
            if (${reporte.autoPrint || false}) {
                window.print();
            }
        </script>
    </body>
    </html>`;
}

// Descargar reporte
function descargarReporte(reporte, formato) {
    const blob = new Blob([reporte.contenido], {
        type: getContentType(formato)
    });
    
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    
    a.href = url;
    a.download = `${reporte.nombre}_${new Date().toISOString().split('T')[0]}.${formato}`;
    
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    
    URL.revokeObjectURL(url);
}

// Obtener tipo de contenido por formato
function getContentType(formato) {
    const types = {
        'pdf': 'application/pdf',
        'excel': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'csv': 'text/csv',
        'html': 'text/html'
    };
    
    return types[formato] || 'application/octet-stream';
}

// Cargar reportes recientes
function loadReportesRecientes() {
    fetch(`${BASE_URL}/api/reportes/recientes`)
        .then(response => response.json())
        .then(data => {
            const tbody = document.getElementById('reportesRecientes');
            
            if (data.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="5" class="text-center text-muted py-4">
                            <i class="fas fa-info-circle me-2"></i>
                            No hay reportes recientes disponibles
                        </td>
                    </tr>`;
                return;
            }
            
            tbody.innerHTML = data.map(reporte => `
                <tr>
                    <td>
                        <i class="fas ${getReportIcon(reporte.tipo)} me-2"></i>
                        ${reporte.tipo_nombre}
                    </td>
                    <td>${reporte.periodo || 'N/A'}</td>
                    <td>${reporte.usuario}</td>
                    <td>${new Date(reporte.fecha).toLocaleString('es-ES')}</td>
                    <td>
                        <div class="btn-group btn-group-sm">
                            <button class="btn btn-outline-primary" onclick="verReporte('${reporte.id}')">
                                <i class="fas fa-eye"></i>
                            </button>
                            <button class="btn btn-outline-secondary" onclick="descargarReporteId('${reporte.id}')">
                                <i class="fas fa-download"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `).join('');
        })
        .catch(error => {
            console.error('Error cargando reportes recientes:', error);
        });
}

// Obtener icono por tipo de reporte
function getReportIcon(tipo) {
    const icons = {
        'inventario': 'fa-warehouse',
        'movimientos': 'fa-chart-line',
        'usuarios': 'fa-users',
        'alertas': 'fa-bell',
        'ejecutivo': 'fa-chart-pie'
    };
    
    return icons[tipo] || 'fa-file-alt';
}

// Ver reporte existente
function verReporte(id) {
    fetch(`${BASE_URL}/api/reportes/${id}`)
        .then(response => response.json())
        .then(reporte => {
            mostrarReporteHTML(reporte);
        })
        .catch(error => {
            showNotification('Error al cargar el reporte', 'error');
            console.error('Error:', error);
        });
}

// Descargar reporte por ID
function descargarReporteId(id) {
    window.location.href = `${BASE_URL}/api/reportes/${id}/descargar`;
}

// Actualizar lista de reportes recientes
function actualizarReportesRecientes(nuevoReporte) {
    const tbody = document.getElementById('reportesRecientes');
    const primeraFila = tbody.querySelector('tr');
    
    const nuevaFila = document.createElement('tr');
    nuevaFila.innerHTML = `
        <td>
            <i class="fas ${getReportIcon(nuevoReporte.tipo_reporte)} me-2"></i>
            ${nuevoReporte.tipo_reporte}
        </td>
        <td>${nuevoReporte.fecha_inicio || 'N/A'} - ${nuevoReporte.fecha_fin || 'N/A'}</td>
        <td>Usuario Actual</td>
        <td>${new Date().toLocaleString('es-ES')}</td>
        <td>
            <div class="btn-group btn-group-sm">
                <button class="btn btn-outline-primary" onclick="verReporte('temp')">
                    <i class="fas fa-eye"></i>
                </button>
                <button class="btn btn-outline-secondary" onclick="descargarReporteId('temp')">
                    <i class="fas fa-download"></i>
                </button>
            </div>
        </td>
    `;
    
    // Insertar al principio
    if (primeraFila) {
        tbody.insertBefore(nuevaFila, primeraFila);
    } else {
        tbody.appendChild(nuevaFila);
    }
    
    // Limitar a 5 reportes recientes
    const filas = tbody.querySelectorAll('tr');
    if (filas.length > 5) {
        filas[filas.length - 1].remove();
    }
}

// Exportar alertas
function exportarAlertas() {
    showLoading('Exportando alertas...');
    
    fetch(`${BASE_URL}/api/reportes/alertas/exportar`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': getCSRFToken()
        }
    })
    .then(response => response.blob())
    .then(blob => {
        hideLoading();
        
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        
        a.href = url;
        a.download = `alertas_${new Date().toISOString().split('T')[0]}.csv`;
        
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        
        URL.revokeObjectURL(url);
        showNotification('Alertas exportadas exitosamente', 'success');
    })
    .catch(error => {
        hideLoading();
        showNotification('Error al exportar alertas', 'error');
        console.error('Error:', error);
    });
}

// Funciones de utilidad UI
function showLoading(message = 'Cargando...') {
    const overlay = document.createElement('div');
    overlay.className = 'loading-overlay';
    overlay.id = 'loadingOverlay';
    
    overlay.innerHTML = `
        <div class="text-center">
            <div class="loading-spinner"></div>
            <p class="text-white mt-3">${message}</p>
        </div>
    `;
    
    document.body.appendChild(overlay);
}

function hideLoading() {
    const overlay = document.getElementById('loadingOverlay');
    if (overlay) {
        overlay.remove();
    }
}

function showNotification(message, type = 'info') {
    const alertClass = {
        'success': 'alert-success',
        'error': 'alert-danger',
        'warning': 'alert-warning',
        'info': 'alert-info'
    }[type];
    
    const iconClass = {
        'success': 'fa-check-circle',
        'error': 'fa-exclamation-circle',
        'warning': 'fa-exclamation-triangle',
        'info': 'fa-info-circle'
    }[type];
    
    const notification = document.createElement('div');
    notification.className = `alert ${alertClass} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 10000; min-width: 300px;';
    
    notification.innerHTML = `
        <i class="fas ${iconClass} me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remover después de 5 segundos
    setTimeout(() => {
        if (notification && notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

function getCSRFToken() {
    return document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
}

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    ReportesApp.init();
});

// Exportar funciones para uso global
window.ReportesApp = ReportesApp;
window.generarReporte = generarReporte;
window.generarReporteRapido = generarReporteRapido;
window.programarReporte = programarReporte;
window.procesarReporte = procesarReporte;
window.exportarAlertas = exportarAlertas;