/**
 * JavaScript para vista de detalles de utensilio
 * Sistema de Inventario de Utensilios de Cocina
 */

document.addEventListener('DOMContentLoaded', function() {
    // Elementos de la página
    const modalMovimiento = new bootstrap.Modal(document.getElementById('modalMovimiento'));
    const formMovimiento = document.getElementById('formMovimiento');
    const tipoMovimientoInput = document.getElementById('tipoMovimiento');
    const tipoMovimientoTexto = document.getElementById('tipoMovimientoTexto');
    const textoBotonMovimiento = document.getElementById('textoBotonMovimiento');
    const cantidadMovimientoInput = document.getElementById('cantidadMovimiento');
    
    // Inicializar funcionalidades
    initMovimientos();
    initActions();
    initNotifications();
    
    /**
     * Inicializa el sistema de movimientos
     */
    function initMovimientos() {
        if (formMovimiento) {
            formMovimiento.addEventListener('submit', handleMovimientoSubmit);
        }
    }
    
    /**
     * Inicializa las acciones de la página
     */
    function initActions() {
        // Las funciones se definen globalmente para ser llamadas desde los onclick
        window.registrarMovimiento = registrarMovimiento;
        window.confirmarEliminacion = confirmarEliminacion;
        window.imprimirEtiqueta = imprimirEtiqueta;
        window.exportarDatos = exportarDatos;
    }
    
    /**
     * Registra un movimiento (entrada o salida)
     */
    function registrarMovimiento(tipo) {
        // Configurar modal según el tipo
        tipoMovimientoInput.value = tipo;
        
        if (tipo === 'entrada') {
            tipoMovimientoTexto.textContent = 'Entrada';
            textoBotonMovimiento.textContent = 'Entrada';
            cantidadMovimientoInput.max = 999999; // Sin límite para entradas
            cantidadMovimientoInput.placeholder = 'Cantidad a ingresar';
        } else {
            tipoMovimientoTexto.textContent = 'Salida';
            textoBotonMovimiento.textContent = 'Salida';
            cantidadMovimientoInput.max = cantidadDisponible;
            cantidadMovimientoInput.placeholder = `Máximo ${cantidadDisponible} disponibles`;
        }
        
        // Limpiar formulario
        formMovimiento.reset();
        tipoMovimientoInput.value = tipo;
        
        // Mostrar modal
        modalMovimiento.show();
        
        // Focus en cantidad
        setTimeout(() => {
            cantidadMovimientoInput.focus();
        }, 300);
    }
    
    /**
     * Maneja el envío del formulario de movimiento
     */
    function handleMovimientoSubmit(event) {
        event.preventDefault();
        
        const formData = new FormData(formMovimiento);
        const tipo = formData.get('tipo_movimiento');
        const cantidad = parseInt(formData.get('cantidad'));
        
        // Validaciones
        if (!cantidad || cantidad <= 0) {
            showError('La cantidad debe ser mayor a 0');
            return;
        }
        
        if (tipo === 'salida' && cantidad > cantidadDisponible) {
            showError(`No hay suficiente stock. Disponible: ${cantidadDisponible}`);
            return;
        }
        
        // Mostrar confirmación
        const mensaje = tipo === 'entrada' 
            ? `¿Registrar entrada de ${cantidad} unidades?`
            : `¿Registrar salida de ${cantidad} unidades?`;
            
        if (!confirm(mensaje)) {
            return;
        }
        
        // Enviar datos
        submitMovimiento(formData);
    }
    
    /**
     * Envía el movimiento al servidor
     */
    async function submitMovimiento(formData) {
        try {
            showLoading('Registrando movimiento...');
            
            const response = await fetch(`${BASE_URL}/api/movimientos/registrar`, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                showSuccess('Movimiento registrado exitosamente');
                modalMovimiento.hide();
                
                // Recargar página después de un breve delay
                setTimeout(() => {
                    location.reload();
                }, 1500);
            } else {
                showError(result.message || 'Error al registrar el movimiento');
            }
            
        } catch (error) {
            console.error('Error:', error);
            showError('Error de conexión. Intente nuevamente.');
        } finally {
            hideLoading();
        }
    }
    
    /**
     * Confirma la eliminación del utensilio
     */
    function confirmarEliminacion() {
        const nombre = document.querySelector('h1').textContent.replace(/^\s*[^a-zA-Z]*\s*/, '').trim();
        
        if (confirm(`¿Está seguro que desea eliminar el utensilio "${nombre}"?\n\nEsta acción no se puede deshacer y eliminará también todo el historial de movimientos.`)) {
            eliminarUtensilio();
        }
    }
    
    /**
     * Elimina el utensilio
     */
    async function eliminarUtensilio() {
        try {
            showLoading('Eliminando utensilio...');
            
            const response = await fetch(`${BASE_URL}/api/utensilios/${utensilioId}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-Token': document.querySelector('meta[name="csrf-token"]').content
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                showSuccess('Utensilio eliminado exitosamente');
                
                // Redirigir a la lista
                setTimeout(() => {
                    window.location.href = `${BASE_URL}/utensilios`;
                }, 1500);
            } else {
                showError(result.message || 'Error al eliminar el utensilio');
            }
            
        } catch (error) {
            console.error('Error:', error);
            showError('Error de conexión. Intente nuevamente.');
        } finally {
            hideLoading();
        }
    }
    
    /**
     * Imprime etiqueta del utensilio
     */
    function imprimirEtiqueta() {
        const nombre = document.querySelector('h1').textContent.replace(/^\s*[^a-zA-Z]*\s*/, '').trim();
        const ubicacion = document.querySelector('dd').textContent.trim();
        
        // Crear ventana de impresión
        const ventanaImpresion = window.open('', '_blank');
        
        const contenidoEtiqueta = `
            <!DOCTYPE html>
            <html>
            <head>
                <title>Etiqueta - ${nombre}</title>
                <style>
                    body { 
                        font-family: Arial, sans-serif; 
                        padding: 20px; 
                        font-size: 14px;
                        line-height: 1.4;
                    }
                    .etiqueta {
                        border: 2px solid #000;
                        padding: 15px;
                        width: 300px;
                        margin: 0 auto;
                    }
                    .nombre {
                        font-weight: bold;
                        font-size: 16px;
                        margin-bottom: 10px;
                        text-align: center;
                    }
                    .info {
                        margin: 8px 0;
                    }
                    .codigo {
                        text-align: center;
                        font-family: monospace;
                        background: #f0f0f0;
                        padding: 5px;
                        margin: 10px 0;
                    }
                    @media print {
                        body { margin: 0; padding: 10px; }
                        .etiqueta { width: auto; }
                    }
                </style>
            </head>
            <body>
                <div class="etiqueta">
                    <div class="nombre">${nombre}</div>
                    <div class="info"><strong>ID:</strong> ${utensilioId}</div>
                    <div class="info"><strong>Ubicación:</strong> ${ubicacion}</div>
                    <div class="info"><strong>Fecha:</strong> ${new Date().toLocaleDateString('es-CO')}</div>
                    <div class="codigo">ID-${String(utensilioId).padStart(6, '0')}</div>
                </div>
                <script>
                    window.onload = function() {
                        window.print();
                        window.close();
                    };
                </script>
            </body>
            </html>
        `;
        
        ventanaImpresion.document.write(contenidoEtiqueta);
        ventanaImpresion.document.close();
    }
    
    /**
     * Exporta los datos del utensilio
     */
    async function exportarDatos() {
        try {
            showLoading('Exportando datos...');
            
            const response = await fetch(`${BASE_URL}/api/utensilios/${utensilioId}/export`, {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            if (response.ok) {
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `utensilio-${utensilioId}-${new Date().toISOString().split('T')[0]}.pdf`;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                window.URL.revokeObjectURL(url);
                
                showSuccess('Datos exportados exitosamente');
            } else {
                showError('Error al exportar los datos');
            }
            
        } catch (error) {
            console.error('Error:', error);
            showError('Error de conexión. Intente nuevamente.');
        } finally {
            hideLoading();
        }
    }
    
    /**
     * Sistema de notificaciones
     */
    function initNotifications() {
        // Crear container de toasts si no existe
        if (!document.querySelector('.toast-container')) {
            const container = document.createElement('div');
            container.className = 'toast-container';
            document.body.appendChild(container);
        }
    }
    
    function showNotification(message, type = 'info', duration = 5000) {
        const container = document.querySelector('.toast-container');
        
        const icons = {
            'success': 'fas fa-check-circle',
            'error': 'fas fa-exclamation-triangle',
            'warning': 'fas fa-exclamation-circle',
            'info': 'fas fa-info-circle'
        };
        
        const colors = {
            'success': 'bg-success',
            'error': 'bg-danger',
            'warning': 'bg-warning',
            'info': 'bg-primary'
        };
        
        const toastId = 'toast-' + Date.now();
        
        const toastHTML = `
            <div id="${toastId}" class="toast align-items-center text-white ${colors[type]} border-0" role="alert">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="${icons[type]} me-2"></i>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        `;
        
        container.insertAdjacentHTML('beforeend', toastHTML);
        
        const toastElement = document.getElementById(toastId);
        const toast = new bootstrap.Toast(toastElement, { delay: duration });
        toast.show();
        
        // Limpiar después de que se oculte
        toastElement.addEventListener('hidden.bs.toast', () => {
            toastElement.remove();
        });
    }
    
    function showSuccess(message) {
        showNotification(message, 'success');
    }
    
    function showError(message) {
        showNotification(message, 'error');
    }
    
    function showWarning(message) {
        showNotification(message, 'warning');
    }
    
    function showInfo(message) {
        showNotification(message, 'info');
    }
    
    /**
     * Loading overlay
     */
    function showLoading(message = 'Cargando...') {
        // Remover overlay existente
        hideLoading();
        
        const overlay = document.createElement('div');
        overlay.id = 'loading-overlay';
        overlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.1rem;
        `;
        
        overlay.innerHTML = `
            <div class="text-center">
                <div class="spinner-border mb-3" role="status">
                    <span class="visually-hidden">Cargando...</span>
                </div>
                <div>${message}</div>
            </div>
        `;
        
        document.body.appendChild(overlay);
    }
    
    function hideLoading() {
        const overlay = document.getElementById('loading-overlay');
        if (overlay) {
            overlay.remove();
        }
    }
    
    /**
     * Funciones de utilidad
     */
    
    // Actualizar estado de stock en tiempo real
    function actualizarEstadoStock() {
        const disponibleElements = document.querySelectorAll('[data-cantidad-disponible]');
        disponibleElements.forEach(element => {
            const cantidad = parseInt(element.dataset.cantidadDisponible);
            const minima = parseInt(element.dataset.cantidadMinima || '1');
            
            if (cantidad <= minima) {
                element.classList.add('text-danger');
                element.classList.remove('text-success');
            } else {
                element.classList.add('text-success');
                element.classList.remove('text-danger');
            }
        });
    }
    
    // Formatear números
    function formatearNumero(numero) {
        return new Intl.NumberFormat('es-CO').format(numero);
    }
    
    // Formatear moneda
    function formatearMoneda(valor) {
        return new Intl.NumberFormat('es-CO', {
            style: 'currency',
            currency: 'COP',
            minimumFractionDigits: 0,
            maximumFractionDigits: 2
        }).format(valor);
    }
    
    // Actualizar timestamps relativos
    function actualizarTimestamps() {
        const timestamps = document.querySelectorAll('[data-timestamp]');
        timestamps.forEach(element => {
            const timestamp = element.dataset.timestamp;
            const fecha = new Date(timestamp);
            const ahora = new Date();
            const diferencia = ahora - fecha;
            
            let texto = '';
            
            if (diferencia < 60000) { // Menos de 1 minuto
                texto = 'Hace un momento';
            } else if (diferencia < 3600000) { // Menos de 1 hora
                const minutos = Math.floor(diferencia / 60000);
                texto = `Hace ${minutos} minuto${minutos !== 1 ? 's' : ''}`;
            } else if (diferencia < 86400000) { // Menos de 1 día
                const horas = Math.floor(diferencia / 3600000);
                texto = `Hace ${horas} hora${horas !== 1 ? 's' : ''}`;
            } else {
                texto = fecha.toLocaleDateString('es-CO');
            }
            
            element.textContent = texto;
        });
    }
    
    // Inicializar actualizaciones automáticas
    actualizarEstadoStock();
    actualizarTimestamps();
    
    // Actualizar timestamps cada minuto
    setInterval(actualizarTimestamps, 60000);
    
    // Keyboard shortcuts
    document.addEventListener('keydown', function(e) {
        // Ctrl + E para editar
        if (e.ctrlKey && e.key === 'e') {
            e.preventDefault();
            const editBtn = document.querySelector('a[href*="/editar/"]');
            if (editBtn) {
                editBtn.click();
            }
        }
        
        // Ctrl + P para imprimir etiqueta
        if (e.ctrlKey && e.key === 'p') {
            e.preventDefault();
            imprimirEtiqueta();
        }
        
        // Escape para cerrar modales
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                const modalInstance = bootstrap.Modal.getInstance(openModal);
                if (modalInstance) {
                    modalInstance.hide();
                }
            }
        }
    });
    
    // Exponer funciones globalmente para compatibilidad
    window.showSuccess = showSuccess;
    window.showError = showError;
    window.showWarning = showWarning;
    window.showInfo = showInfo;
    window.showLoading = showLoading;
    window.hideLoading = hideLoading;
});

// Inicialización de tooltips si están disponibles
document.addEventListener('DOMContentLoaded', function() {
    if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }
});