// JavaScript para el módulo de utensilios
document.addEventListener('DOMContentLoaded', function() {
    
    // Referencias a elementos
    const vistaTabla = document.getElementById('tabla-vista');
    const vistaTarjetas = document.getElementById('tarjetas-vista');
    const btnVistaTabla = document.getElementById('vista-tabla');
    const btnVistaTarjetas = document.getElementById('vista-tarjetas');
    
    // Cambio de vista
    if (btnVistaTabla && btnVistaTarjetas) {
        btnVistaTabla.addEventListener('change', function() {
            if (this.checked) {
                mostrarVistaTabla();
            }
        });
        
        btnVistaTarjetas.addEventListener('change', function() {
            if (this.checked) {
                mostrarVistaTarjetas();
            }
        });
    }
    
    // Guardar preferencia de vista
    const vistaPreferida = localStorage.getItem('vista-utensilios');
    if (vistaPreferida === 'tarjetas') {
        btnVistaTarjetas.checked = true;
        mostrarVistaTarjetas();
    }
    
    // Auto-submit del formulario de filtros con debounce
    const filtroInputs = document.querySelectorAll('input[name="busqueda"], select[name="tipo"], select[name="estado"], select[name="ubicacion"]');
    let timeoutId;
    
    filtroInputs.forEach(input => {
        input.addEventListener('input', function() {
            clearTimeout(timeoutId);
            timeoutId = setTimeout(() => {
                if (input.name !== 'busqueda' || input.value.length >= 2 || input.value.length === 0) {
                    document.querySelector('form').submit();
                }
            }, 500);
        });
    });
    
    // Checkbox de stock bajo
    const checkboxStockBajo = document.querySelector('input[name="stock_bajo"]');
    if (checkboxStockBajo) {
        checkboxStockBajo.addEventListener('change', function() {
            document.querySelector('form').submit();
        });
    }
    
    // Inicializar tooltips
    initTooltips();
    
    // Cargar notificaciones
    cargarNotificaciones();
    
    // Actualizar notificaciones cada 30 segundos
    setInterval(cargarNotificaciones, 30000);
});

/**
 * Muestra la vista de tabla
 */
function mostrarVistaTabla() {
    const vistaTabla = document.getElementById('tabla-vista');
    const vistaTarjetas = document.getElementById('tarjetas-vista');
    
    vistaTabla.style.display = 'block';
    vistaTarjetas.style.display = 'none';
    
    localStorage.setItem('vista-utensilios', 'tabla');
}

/**
 * Muestra la vista de tarjetas
 */
function mostrarVistaTarjetas() {
    const vistaTabla = document.getElementById('tabla-vista');
    const vistaTarjetas = document.getElementById('tarjetas-vista');
    
    vistaTabla.style.display = 'none';
    vistaTarjetas.style.display = 'block';
    
    localStorage.setItem('vista-utensilios', 'tarjetas');
}

/**
 * Confirma la eliminación de un utensilio
 */
function confirmarEliminacion(id, nombre) {
    const modal = new bootstrap.Modal(document.getElementById('modalEliminar'));
    const nombreElement = document.getElementById('nombreUtensilio');
    const btnEliminar = document.getElementById('btnEliminar');
    
    nombreElement.textContent = nombre;
    
    // Configurar el botón de eliminar
    btnEliminar.onclick = function() {
        eliminarUtensilio(id, modal);
    };
    
    modal.show();
}

/**
 * Elimina un utensilio
 */
async function eliminarUtensilio(id, modal) {
    const btnEliminar = document.getElementById('btnEliminar');
    const textoOriginal = btnEliminar.innerHTML;
    
    try {
        // Mostrar loading
        btnEliminar.disabled = true;
        btnEliminar.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Eliminando...';
        
        const response = await fetch(`${BASE_URL}/utensilios/eliminar/${id}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-Token': getCSRFToken()
            }
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Cerrar modal y recargar página
            modal.hide();
            mostrarNotificacion('Utensilio eliminado exitosamente', 'success');
            setTimeout(() => {
                location.reload();
            }, 1000);
        } else {
            throw new Error(result.message || 'Error al eliminar el utensilio');
        }
        
    } catch (error) {
        console.error('Error:', error);
        mostrarNotificacion(error.message, 'error');
    } finally {
        // Restaurar botón
        btnEliminar.disabled = false;
        btnEliminar.innerHTML = textoOriginal;
    }
}

/**
 * Inicializa los tooltips de Bootstrap
 */
function initTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function(tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
}

/**
 * Muestra una notificación temporal
 */
function mostrarNotificacion(mensaje, tipo = 'info') {
    const alertClass = {
        'success': 'alert-success',
        'error': 'alert-danger',
        'warning': 'alert-warning',
        'info': 'alert-info'
    }[tipo] || 'alert-info';
    
    const iconClass = {
        'success': 'fas fa-check-circle',
        'error': 'fas fa-exclamation-triangle',
        'warning': 'fas fa-exclamation-circle',
        'info': 'fas fa-info-circle'
    }[tipo] || 'fas fa-info-circle';
    
    const alertHtml = `
        <div class="alert ${alertClass} alert-dismissible fade show position-fixed" 
             style="top: 20px; right: 20px; z-index: 1055; min-width: 300px;">
            <i class="${iconClass} me-2"></i>
            ${mensaje}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', alertHtml);
    
    // Auto-dismiss después de 5 segundos
    setTimeout(() => {
        const alert = document.querySelector('.alert.position-fixed');
        if (alert) {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }
    }, 5000);
}

/**
 * Obtiene el token CSRF del meta tag
 */
function getCSRFToken() {
    const token = document.querySelector('meta[name="csrf-token"]');
    return token ? token.getAttribute('content') : '';
}

/**
 * Carga las notificaciones del usuario
 */
async function cargarNotificaciones() {
    try {
        const response = await fetch(`${BASE_URL}/api/notificaciones`, {
            headers: {
                'X-CSRF-Token': getCSRFToken()
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            actualizarNotificaciones(data);
        }
    } catch (error) {
        console.log('Error cargando notificaciones:', error);
    }
}

/**
 * Actualiza el display de notificaciones
 */
function actualizarNotificaciones(notificaciones) {
    const contador = document.getElementById('contadorNotificaciones');
    const lista = document.getElementById('listaNotificaciones');
    
    if (!contador || !lista) return;
    
    const noLeidas = notificaciones.filter(n => !n.leida);
    
    if (noLeidas.length > 0) {
        contador.textContent = noLeidas.length;
        contador.style.display = 'block';
        
        // Actualizar lista
        lista.innerHTML = '';
        noLeidas.slice(0, 5).forEach(notif => {
            const item = document.createElement('li');
            item.className = 'dropdown-item-text border-bottom pb-2 mb-2';
            item.innerHTML = `
                <div class="d-flex align-items-start">
                    <i class="fas fa-${getNotificationIcon(notif.tipo)} text-${getNotificationColor(notif.nivel)} me-2 mt-1"></i>
                    <div class="flex-grow-1">
                        <div class="small fw-bold">${notif.mensaje}</div>
                        <small class="text-muted">${formatearFecha(notif.fecha_creacion)}</small>
                    </div>
                </div>
            `;
            lista.appendChild(item);
        });
    } else {
        contador.style.display = 'none';
        lista.innerHTML = '<li class="dropdown-item-text text-muted text-center py-3">No hay notificaciones nuevas</li>';
    }
}

/**
 * Obtiene el icono según el tipo de notificación
 */
function getNotificationIcon(tipo) {
    const icons = {
        'stock_bajo': 'exclamation-triangle',
        'stock_agotado': 'times-circle',
        'utensilio_dañado': 'wrench',
        'revision_pendiente': 'clock'
    };
    return icons[tipo] || 'bell';
}

/**
 * Obtiene el color según el nivel de notificación
 */
function getNotificationColor(nivel) {
    const colors = {
        'critical': 'danger',
        'warning': 'warning',
        'info': 'info'
    };
    return colors[nivel] || 'secondary';
}

/**
 * Formatea una fecha para mostrar
 */
function formatearFecha(fecha) {
    const ahora = new Date();
    const fechaNotif = new Date(fecha);
    const diffMs = ahora - fechaNotif;
    const diffMins = Math.floor(diffMs / 60000);
    const diffHoras = Math.floor(diffMins / 60);
    const diffDias = Math.floor(diffHoras / 24);
    
    if (diffMins < 1) return 'Ahora';
    if (diffMins < 60) return `Hace ${diffMins} min`;
    if (diffHoras < 24) return `Hace ${diffHoras} h`;
    if (diffDias < 7) return `Hace ${diffDias} días`;
    
    return fechaNotif.toLocaleDateString();
}

/**
 * Funciones de utilidad para filtros
 */
function limpiarFiltros() {
    document.getElementById('busqueda').value = '';
    document.getElementById('tipo').value = '';
    document.getElementById('estado').value = '';
    document.getElementById('ubicacion').value = '';
    document.getElementById('stock_bajo').checked = false;
    document.querySelector('form').submit();
}

/**
 * Exportar resultados
 */
function exportarResultados(formato = 'excel') {
    const params = new URLSearchParams(window.location.search);
    params.set('exportar', formato);
    
    const url = `${BASE_URL}/utensilios/exportar?${params.toString()}`;
    window.open(url, '_blank');
}

/**
 * Imprimir tabla
 */
function imprimirTabla() {
    const contenido = document.getElementById('tabla-vista').innerHTML;
    const ventana = window.open('', '_blank');
    ventana.document.write(`
        <html>
        <head>
            <title>Lista de Utensilios - ${document.title}</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
            <style>
                @media print {
                    .btn { display: none; }
                    .card { border: none; box-shadow: none; }
                }
            </style>
        </head>
        <body>
            <div class="container">
                <h2>Lista de Utensilios</h2>
                <p>Generado el: ${new Date().toLocaleDateString()}</p>
                ${contenido}
            </div>
        </body>
        </html>
    `);
    ventana.document.close();
    ventana.print();
}

// Atajos de teclado
document.addEventListener('keydown', function(e) {
    // Ctrl+N para nuevo utensilio
    if (e.ctrlKey && e.key === 'n') {
        e.preventDefault();
        const enlaceNuevo = document.querySelector('a[href*="/utensilios/nuevo"]');
        if (enlaceNuevo) {
            window.location.href = enlaceNuevo.href;
        }
    }
    
    // Ctrl+F para enfocar búsqueda
    if (e.ctrlKey && e.key === 'f') {
        e.preventDefault();
        const busqueda = document.getElementById('busqueda');
        if (busqueda) {
            busqueda.focus();
            busqueda.select();
        }
    }
});