<?php

/**
 * Servicio de Auditoría
 * Maneja el registro y consulta de eventos de auditoría
 */
class AuditoriaService {
    private $auditoriaRepo;
    private $authService;
    
    public function __construct() {
        $this->auditoriaRepo = new AuditoriaRepository();
        $this->authService = new AuthService();
    }

    /**
     * Registra un evento de auditoría
     */
    public function registrar($tabla, $accion, $idRegistro, $datosAnteriores = null, $datosNuevos = null, $usuarioId = null) {
        try {
            // Obtener usuario actual si no se especifica
            if ($usuarioId === null) {
                $usuario = $this->authService->getUsuarioActual();
                $usuarioId = $usuario ? $usuario->id : null;
            }
            
            $datos = [
                'tabla_afectada' => $tabla,
                'accion' => $accion,
                'id_registro' => $idRegistro,
                'usuario_id' => $usuarioId,
                'datos_anteriores' => $datosAnteriores ? json_encode($datosAnteriores) : null,
                'datos_nuevos' => $datosNuevos ? json_encode($datosNuevos) : null
            ];
            
            return $this->auditoriaRepo->registrar($datos);
            
        } catch (Exception $e) {
            // No lanzar excepción para no interrumpir el flujo principal
            error_log("Error registrando auditoría: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Obtiene registros de auditoría
     */
    public function obtenerRegistros($filtros = [], $limite = 100) {
        $this->authService->verificarPermiso('auditoria_ver');
        
        // Limitar resultados para evitar sobrecarga
        if (!isset($filtros['limite'])) {
            $filtros['limite'] = $limite;
        }
        
        return $this->auditoriaRepo->obtenerConFiltros($filtros);
    }

    /**
     * Obtiene actividad de usuarios
     */
    public function obtenerActividadUsuarios($dias = 30) {
        $this->authService->verificarPermiso('auditoria_ver');
        return $this->auditoriaRepo->getActividadPorUsuario($dias);
    }

    /**
     * Obtiene estadísticas de auditoría
     */
    public function obtenerEstadisticas($dias = 30) {
        $this->authService->verificarPermiso('auditoria_ver');
        return $this->auditoriaRepo->getEstadisticas($dias);
    }

    /**
     * Exporta datos de auditoría para reportes
     */
    public function exportarParaReporte($filtros = []) {
        $this->authService->verificarPermiso('auditoria_ver');
        
        $registros = $this->obtenerRegistros($filtros);
        $datos = [];
        
        foreach ($registros as $registro) {
            $datos[] = [
                'fecha' => date('d/m/Y H:i:s', strtotime($registro['fecha'])),
                'usuario' => $registro['nombre_usuario'] ?? 'Sistema',
                'tabla' => $registro['tabla_afectada'],
                'accion' => $registro['accion'],
                'ip' => $registro['ip_address'] ?? 'N/A'
            ];
        }
        
        return $datos;
    }

    /**
     * Limpia registros antiguos de auditoría
     */
    public function limpiarRegistrosAntiguos($diasRetener = 365) {
        if (!$this->authService->esAdministrador()) {
            throw new Exception('Solo los administradores pueden limpiar registros de auditoría');
        }
        
        // Contar registros antes de eliminar
        $totalAntes = $this->auditoriaRepo->count();
        
        // Eliminar registros antiguos
        $resultado = $this->auditoriaRepo->eliminarRegistrosAntiguos($diasRetener);
        
        // Registrar la limpieza
        $this->registrar('auditoria', 'CLEANUP', 0, null, ['registros_eliminados' => $resultado, 'total_antes' => $totalAntes]);
        
        return $resultado;
    }

    /**
     * Obtiene resumen de cambios para un registro específico
     */
    public function obtenerHistorialRegistro($tabla, $idRegistro) {
        $this->authService->verificarPermiso('auditoria_ver');
        
        $filtros = [
            'tabla' => $tabla,
            'id_registro' => $idRegistro
        ];
        
        return $this->auditoriaRepo->obtenerConFiltros($filtros);
    }
}