<?php

// No require directo - se carga con autoloader

/**
 * Servicio de Autenticación y Autorización con Seguridad Mejorada
 */
class AuthService {
    private $usuarioRepo;
    private $security;
    
    public function __construct() {
        $this->usuarioRepo = new UsuarioRepository();
        $this->security = SecurityManager::getInstance();
    }

    /**
     * Autentica un usuario con seguridad mejorada
     */
    public function login($email, $password) {
        // Verificar rate limiting
        if (!$this->security->checkRateLimit()) {
            throw new Exception('Demasiados intentos. Intente más tarde.');
        }
        
        // Sanitizar y validar datos de entrada
        $email = $this->security->sanitizeInput($email);
        $password = $this->security->sanitizeInput($password);
        
        if (empty($email) || empty($password)) {
            throw new Exception('Email y contraseña son requeridos');
        }

        if (!$this->security->validateInput($email, 'email')) {
            throw new Exception('El formato del email no es válido');
        }

        // Verificar si la cuenta está bloqueada
        if ($this->security->isBlocked($email)) {
            throw new Exception('Cuenta temporalmente bloqueada por múltiples intentos fallidos');
        }

        // Verificar credenciales
        $usuario = $this->usuarioRepo->verificarCredenciales($email, $password);
        
        if (!$usuario) {
            // Registrar intento fallido
            $this->security->recordLoginAttempt($email, false);
            throw new Exception('Credenciales incorrectas');
        }

        // Registrar intento exitoso
        $this->security->recordLoginAttempt($email, true);
        
        // Iniciar sesión
        $this->iniciarSesion($usuario);
        
        // Registrar auditoría
        $this->registrarAuditoria('LOGIN', $usuario->id, 'Inicio de sesión exitoso');
        
        return $usuario;
    }

    /**
     * Cierra la sesión del usuario
     */
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            $userId = $_SESSION['user_id'];
            
            // Registrar auditoría
            $this->registrarAuditoria('LOGOUT', $userId, 'Cierre de sesión');
            
            // Limpiar sesión
            session_unset();
            session_destroy();
            
            return true;
        }
        
        return false;
    }

    /**
     * Verifica si el usuario está autenticado
     */
    public function estaAutenticado() {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }

    /**
     * Obtiene el usuario actual de la sesión
     */
    public function getUsuarioActual() {
        if (!$this->estaAutenticado()) {
            return null;
        }

        $userId = $_SESSION['user_id'];
        return $this->usuarioRepo->getById($userId);
    }

    /**
     * Verifica si el usuario tiene un rol específico
     */
    public function tieneRol($rol) {
        $usuario = $this->getUsuarioActual();
        return $usuario && $usuario->tieneRol($rol);
    }

    /**
     * Verifica si el usuario es administrador
     */
    public function esAdministrador() {
        return $this->tieneRol(Usuario::ROL_ADMINISTRADOR);
    }

    /**
     * Verifica si el usuario es encargado
     */
    public function esEncargado() {
        return $this->tieneRol(Usuario::ROL_ENCARGADO);
    }

    /**
     * Verifica si el usuario es auditor
     */
    public function esAuditor() {
        return $this->tieneRol(Usuario::ROL_AUDITOR);
    }

    /**
     * Verifica permisos para una acción específica
     */
    public function verificarPermiso($accion) {
        $usuario = $this->getUsuarioActual();
        
        if (!$usuario) {
            throw new Exception('Usuario no autenticado');
        }

        $permisos = $this->getPermisosRol($usuario->rol);
        
        if (!in_array($accion, $permisos)) {
            throw new Exception('No tiene permisos para realizar esta acción');
        }

        return true;
    }

    /**
     * Obtiene los permisos según el rol
     */
    private function getPermisosRol($rol) {
        $permisos = [
            Usuario::ROL_ADMINISTRADOR => [
                'usuarios_crear', 'usuarios_editar', 'usuarios_eliminar', 'usuarios_ver',
                'utensilios_crear', 'utensilios_editar', 'utensilios_eliminar', 'utensilios_ver',
                'movimientos_crear', 'movimientos_ver', 'movimientos_editar',
                'reportes_ver', 'reportes_exportar',
                'auditoria_ver', 'sistema_configurar'
            ],
            Usuario::ROL_ENCARGADO => [
                'utensilios_crear', 'utensilios_editar', 'utensilios_ver',
                'movimientos_crear', 'movimientos_ver',
                'reportes_ver'
            ],
            Usuario::ROL_AUDITOR => [
                'utensilios_ver',
                'movimientos_ver',
                'reportes_ver', 'reportes_exportar',
                'auditoria_ver'
            ]
        ];

        return $permisos[$rol] ?? [];
    }

    /**
     * Cambia la contraseña del usuario actual
     */
    public function cambiarPassword($passwordActual, $passwordNueva, $confirmarPassword) {
        $usuario = $this->getUsuarioActual();
        
        if (!$usuario) {
            throw new Exception('Usuario no autenticado');
        }

        // Validar contraseña actual
        if (!password_verify($passwordActual, $usuario->password_hash)) {
            throw new Exception('La contraseña actual es incorrecta');
        }

        // Validar nueva contraseña
        $this->validarPassword($passwordNueva, $confirmarPassword);

        // Cambiar contraseña
        $nuevoHash = password_hash($passwordNueva, PASSWORD_HASH_ALGO);
        $resultado = $this->usuarioRepo->cambiarPassword($usuario->id, $nuevoHash);

        if ($resultado) {
            $this->registrarAuditoria('PASSWORD_CHANGE', $usuario->id, 'Cambio de contraseña');
        }

        return $resultado;
    }

    /**
     * Valida la fortaleza de la contraseña
     */
    private function validarPassword($password, $confirmar) {
        if (empty($password)) {
            throw new Exception('La contraseña es requerida');
        }

        if ($password !== $confirmar) {
            throw new Exception('Las contraseñas no coinciden');
        }

        if (strlen($password) < 6) {
            throw new Exception('La contraseña debe tener al menos 6 caracteres');
        }

        // Opcional: Validaciones adicionales de fortaleza
        if (!preg_match('/[A-Za-z]/', $password)) {
            throw new Exception('La contraseña debe contener al menos una letra');
        }

        if (!preg_match('/[0-9]/', $password)) {
            throw new Exception('La contraseña debe contener al menos un número');
        }

        return true;
    }

    /**
     * Inicia la sesión del usuario
     */
    private function iniciarSesion($usuario) {
        // Regenerar ID de sesión por seguridad
        session_regenerate_id(true);
        
        // Guardar datos en la sesión
        $_SESSION['user_id'] = $usuario->id;
        $_SESSION['user_name'] = $usuario->nombre;
        $_SESSION['user_email'] = $usuario->email;
        $_SESSION['user_role'] = $usuario->rol;
        $_SESSION['login_time'] = time();
        $_SESSION['last_activity'] = time();

        // Generar token CSRF
        $_SESSION[CSRF_TOKEN_NAME] = $this->generarCSRFToken();
    }

    /**
     * Genera un token CSRF
     */
    private function generarCSRFToken() {
        return bin2hex(random_bytes(32));
    }

    /**
     * Verifica el token CSRF
     */
    public function verificarCSRFToken($token) {
        return isset($_SESSION[CSRF_TOKEN_NAME]) && 
               hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }

    /**
     * Obtiene el token CSRF actual
     */
    public function getCSRFToken() {
        return $_SESSION[CSRF_TOKEN_NAME] ?? null;
    }

    /**
     * Verifica si la sesión ha expirado
     */
    public function verificarSesion() {
        if (!$this->estaAutenticado()) {
            return false;
        }

        $tiempoInactividad = time() - ($_SESSION['last_activity'] ?? 0);
        
        if ($tiempoInactividad > SESSION_LIFETIME) {
            $this->logout();
            return false;
        }

        // Actualizar última actividad
        $_SESSION['last_activity'] = time();
        return true;
    }

    /**
     * Registra un intento de login fallido
     */
    private function registrarIntentoFallido($email) {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        
        error_log("Intento de login fallido - Email: {$email}, IP: {$ip}, User Agent: {$userAgent}");
        
        // Aquí podrías implementar un sistema de bloqueo por intentos fallidos
    }

    /**
     * Registra eventos de auditoría
     */
    private function registrarAuditoria($accion, $usuarioId, $detalle) {
        try {
            $auditoriaRepo = new AuditoriaRepository();
            $auditoriaRepo->registrar([
                'tabla_afectada' => 'usuarios',
                'accion' => $accion,
                'id_registro' => $usuarioId,
                'usuario_id' => $usuarioId,
                'datos_nuevos' => json_encode(['detalle' => $detalle]),
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
        } catch (Exception $e) {
            error_log("Error registrando auditoría: " . $e->getMessage());
        }
    }

    /**
     * Middleware para verificar autenticación
     */
    public function requiereAutenticacion() {
        if (!$this->verificarSesion()) {
            header('Location: ' . BASE_URL . '/login');
            exit;
        }
    }

    /**
     * Middleware para verificar permisos
     */
    public function requierePermiso($accion) {
        $this->requiereAutenticacion();
        $this->verificarPermiso($accion);
    }
}