<?php

// No require directo - se carga con autoloader

/**
 * Servicio para el Dashboard principal del sistema
 * Proporciona métricas, estadísticas y datos para widgets
 */
class DashboardService {
    
    private $authService;
    private $movimientoRepo;
    private $utensilioRepo;
    private $usuarioRepo;
    private $auditoriaRepo;
    private $cache;
    
    public function __construct() {
        $this->authService = new AuthService();
        $this->movimientoRepo = new MovimientoRepository();
        $this->utensilioRepo = new UtensilioRepository();
        $this->usuarioRepo = new UsuarioRepository();
        $this->auditoriaRepo = new AuditoriaRepository();
        $this->cache = CacheManager::getInstance();
    }
    
    /**
     * Obtiene métricas principales del dashboard
     */
    public function obtenerMetricasPrincipales() {
        $this->authService->verificarSesion();
        
        $cacheKey = 'dashboard_metricas_principales';
        
        return $this->cache->remember($cacheKey, function() {
        
        // Métricas de inventario
        $totalUtensilios = $this->utensilioRepo->count();
        $totalItems = $this->calcularTotalItems();
        $valorInventario = $this->calcularValorInventario();
        
        // Métricas de movimientos (últimos 30 días)
        $fechaInicio = date('Y-m-d', strtotime('-30 days'));
        $fechaFin = date('Y-m-d');
        
        $movimientosRecientes = $this->movimientoRepo->buscar([
            'fecha_inicio' => $fechaInicio,
            'fecha_fin' => $fechaFin
        ]);
        
        $entradas = 0;
        $salidas = 0;
        $itemsMovidos = 0;
        
        foreach ($movimientosRecientes as $movimiento) {
            if ($movimiento->tipo_movimiento === 'entrada') {
                $entradas++;
            } else {
                $salidas++;
            }
            $itemsMovidos += $movimiento->cantidad;
        }
        
        // Alertas de stock
        $alertas = $this->obtenerAlertas();
        
        // Usuarios activos
        $usuariosActivos = $this->contarUsuariosActivos();
        
        return [
            'inventario' => [
                'total_utensilios' => $totalUtensilios,
                'total_items' => $totalItems,
                'valor_total' => $valorInventario,
                'items_disponibles' => $this->calcularItemsDisponibles()
            ],
            'movimientos' => [
                'total_movimientos' => count($movimientosRecientes),
                'entradas' => $entradas,
                'salidas' => $salidas,
                'items_movidos' => $itemsMovidos,
                'balance' => $entradas - $salidas,
                'promedio_diario' => round(count($movimientosRecientes) / 30, 1)
            ],
            'alertas' => [
                'total' => count($alertas),
                'criticas' => count(array_filter($alertas, function($a) { return $a['nivel'] === 'critica'; })),
                'advertencias' => count(array_filter($alertas, function($a) { return $a['nivel'] === 'advertencia'; })),
                'detalle' => array_slice($alertas, 0, 5) // Top 5 alertas
            ],
            'usuarios' => [
                'activos' => $usuariosActivos,
                'total' => $this->usuarioRepo->count(),
                'conectados_hoy' => $this->contarUsuariosConectadosHoy()
            ],
            'fecha_actualizacion' => date('Y-m-d H:i:s')
        ];
        }, 300); // Cache por 5 minutos
    }
    
    /**
     * Obtiene datos para gráficos de tendencias
     */
    public function obtenerDatosGraficos($periodo = 'semana') {
        $this->authService->verificarSesion();
        
        switch ($periodo) {
            case 'semana':
                return $this->obtenerDatosUltimaSemana();
            case 'mes':
                return $this->obtenerDatosUltimoMes();
            case 'trimestre':
                return $this->obtenerDatosUltimoTrimestre();
            default:
                return $this->obtenerDatosUltimaSemana();
        }
    }
    
    /**
     * Obtiene top utensilios más usados
     */
    public function obtenerTopUtensilios($limite = 10) {
        $this->authService->verificarSesion();
        
        // Obtener movimientos de los últimos 30 días agrupados por utensilio
        $fechaInicio = date('Y-m-d', strtotime('-30 days'));
        
        $sql = "SELECT 
                    u.id,
                    u.nombre,
                    u.tipo,
                    COUNT(m.id) as total_movimientos,
                    SUM(CASE WHEN m.tipo_movimiento = 'salida' THEN m.cantidad ELSE 0 END) as veces_usado,
                    SUM(CASE WHEN m.tipo_movimiento = 'entrada' THEN m.cantidad ELSE 0 END) as veces_devuelto,
                    u.cantidad_disponible,
                    u.cantidad_total
                FROM utensilios u
                LEFT JOIN movimientos m ON u.id = m.id_utensilio AND m.fecha >= ?
                WHERE u.activo = 1
                GROUP BY u.id, u.nombre, u.tipo, u.cantidad_disponible, u.cantidad_total
                ORDER BY total_movimientos DESC, veces_usado DESC
                LIMIT ?";
        
        $data = Database::getInstance()->select($sql, [$fechaInicio, $limite]);
        
        return array_map(function($item) {
            return [
                'id' => $item['id'],
                'nombre' => $item['nombre'],
                'tipo' => $item['tipo'],
                'total_movimientos' => (int)$item['total_movimientos'],
                'veces_usado' => (int)$item['veces_usado'],
                'veces_devuelto' => (int)$item['veces_devuelto'],
                'disponibles' => (int)$item['cantidad_disponible'],
                'total' => (int)$item['cantidad_total'],
                'porcentaje_uso' => $item['cantidad_total'] > 0 ? 
                    round((($item['cantidad_total'] - $item['cantidad_disponible']) / $item['cantidad_total']) * 100, 1) : 0
            ];
        }, $data);
    }
    
    /**
     * Obtiene actividad reciente del sistema
     */
    public function obtenerActividadReciente($limite = 20) {
        $this->authService->verificarSesion();
        
        // Combinar movimientos y auditoría reciente
        $movimientosRecientes = $this->movimientoRepo->getConDetalles(10);
        $auditoriaReciente = [];
        
        $actividades = [];
        
        // Agregar movimientos
        foreach ($movimientosRecientes as $mov) {
            $actividades[] = [
                'tipo' => 'movimiento',
                'fecha' => $mov->fecha,
                'titulo' => ($mov->tipo_movimiento === 'entrada' ? 'Entrada' : 'Salida') . ' de ' . $mov->nombre_utensilio,
                'descripcion' => "Cantidad: {$mov->cantidad}, Motivo: {$mov->motivo}",
                'usuario' => $mov->nombre_usuario ?? 'Sistema',
                'icono' => $mov->tipo_movimiento === 'entrada' ? 'fa-plus-circle' : 'fa-minus-circle',
                'color' => $mov->tipo_movimiento === 'entrada' ? 'success' : 'warning'
            ];
        }
        
        // Agregar eventos de auditoría (por implementar)
        // TODO: Implementar cuando AuditoriaRepository tenga el método obtenerRecientes()
        
        // Ordenar por fecha descendente
        usort($actividades, function($a, $b) {
            return strtotime($b['fecha']) - strtotime($a['fecha']);
        });
        
        return array_slice($actividades, 0, $limite);
    }
    
    /**
     * Obtiene estadísticas de rendimiento del sistema
     */
    public function obtenerEstadisticasRendimiento() {
        $this->authService->verificarSesion();
        
        return [
            'rotacion_inventario' => $this->calcularRotacionInventario(),
            'eficiencia_usuarios' => $this->calcularEficienciaUsuarios(),
            'tiempo_respuesta_promedio' => $this->calcularTiempoRespuesta(),
            'disponibilidad_sistema' => $this->calcularDisponibilidad(),
            'uso_storage' => $this->calcularUsoStorage(),
            'crecimiento_inventario' => $this->calcularCrecimientoInventario()
        ];
    }
    
    /**
     * Obtiene widgets personalizables para el dashboard
     */
    public function obtenerWidgets($usuarioId) {
        $this->authService->verificarSesion();
        
        // En una implementación real, esto vendría de preferencias del usuario
        return [
            'metricas_principales' => ['activo' => true, 'posicion' => 1],
            'grafico_movimientos' => ['activo' => true, 'posicion' => 2],
            'alertas_stock' => ['activo' => true, 'posicion' => 3],
            'top_utensilios' => ['activo' => true, 'posicion' => 4],
            'actividad_reciente' => ['activo' => true, 'posicion' => 5],
            'estadisticas_rendimiento' => ['activo' => false, 'posicion' => 6]
        ];
    }
    
    // Métodos auxiliares privados
    
    private function calcularTotalItems() {
        $sql = "SELECT SUM(cantidad_total) as total FROM utensilios WHERE activo = 1";
        $result = Database::getInstance()->selectOne($sql);
        return (int)($result['total'] ?? 0);
    }
    
    private function calcularItemsDisponibles() {
        $sql = "SELECT SUM(cantidad_disponible) as total FROM utensilios WHERE activo = 1";
        $result = Database::getInstance()->selectOne($sql);
        return (int)($result['total'] ?? 0);
    }
    
    private function calcularItemsEnUso() {
        $sql = "SELECT SUM(cantidad_total - cantidad_disponible) as total FROM utensilios WHERE activo = 1";
        $result = Database::getInstance()->selectOne($sql);
        return (float)($result['total'] ?? 0);
    }
    
    private function obtenerAlertas() {
        $utensilios = $this->utensilioRepo->getAll();
        $alertas = [];
        
        foreach ($utensilios as $utensilio) {
            if ($utensilio->cantidad_disponible == 0) {
                $alertas[] = [
                    'nivel' => 'critica',
                    'tipo' => 'stock_agotado',
                    'mensaje' => "Stock agotado: {$utensilio->nombre}",
                    'utensilio_id' => $utensilio->id,
                    'utensilio_nombre' => $utensilio->nombre
                ];
            } elseif ($utensilio->tieneStockBajo()) {
                $alertas[] = [
                    'nivel' => 'advertencia',
                    'tipo' => 'stock_bajo',
                    'mensaje' => "Stock bajo: {$utensilio->nombre} ({$utensilio->cantidad_disponible} disponibles)",
                    'utensilio_id' => $utensilio->id,
                    'utensilio_nombre' => $utensilio->nombre
                ];
            }
        }
        
        return $alertas;
    }
    
    private function contarUsuariosActivos() {
        $sql = "SELECT COUNT(*) as total FROM usuarios WHERE activo = 1";
        $result = Database::getInstance()->selectOne($sql);
        return (int)($result['total'] ?? 0);
    }
    
    private function contarUsuariosConectadosHoy() {
        // Simulación basada en movimientos del día
        $hoy = date('Y-m-d');
        $sql = "SELECT COUNT(DISTINCT usuario_id) as total FROM movimientos WHERE DATE(fecha) = ?";
        $result = Database::getInstance()->selectOne($sql, [$hoy]);
        return (int)($result['total'] ?? 0);
    }
    
    private function obtenerDatosUltimaSemana() {
        $datos = [];
        
        for ($i = 6; $i >= 0; $i--) {
            $fecha = date('Y-m-d', strtotime("-{$i} days"));
            $datos[] = [
                'fecha' => $fecha,
                'fecha_formato' => date('d/m', strtotime($fecha)),
                'entradas' => $this->contarMovimientosPorFecha($fecha, 'entrada'),
                'salidas' => $this->contarMovimientosPorFecha($fecha, 'salida'),
                'items_movidos' => $this->contarItemsMovidosPorFecha($fecha)
            ];
        }
        
        return $datos;
    }
    
    private function obtenerDatosUltimoMes() {
        $datos = [];
        
        for ($i = 29; $i >= 0; $i--) {
            $fecha = date('Y-m-d', strtotime("-{$i} days"));
            
            // Agrupar por semanas para el mes
            $semana = date('W', strtotime($fecha));
            
            if (!isset($datos[$semana])) {
                $datos[$semana] = [
                    'fecha' => "Sem {$semana}",
                    'entradas' => 0,
                    'salidas' => 0,
                    'items_movidos' => 0
                ];
            }
            
            $datos[$semana]['entradas'] += $this->contarMovimientosPorFecha($fecha, 'entrada');
            $datos[$semana]['salidas'] += $this->contarMovimientosPorFecha($fecha, 'salida');
            $datos[$semana]['items_movidos'] += $this->contarItemsMovidosPorFecha($fecha);
        }
        
        return array_values($datos);
    }
    
    private function obtenerDatosUltimoTrimestre() {
        $datos = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $fecha = date('Y-m', strtotime("-{$i} months"));
            $fechaInicio = $fecha . '-01';
            $fechaFin = date('Y-m-t', strtotime($fechaInicio));
            
            $datos[] = [
                'fecha' => date('M Y', strtotime($fechaInicio)),
                'entradas' => $this->contarMovimientosPorRango($fechaInicio, $fechaFin, 'entrada'),
                'salidas' => $this->contarMovimientosPorRango($fechaInicio, $fechaFin, 'salida'),
                'items_movidos' => $this->contarItemsMovidosPorRango($fechaInicio, $fechaFin)
            ];
        }
        
        return $datos;
    }
    
    private function contarMovimientosPorFecha($fecha, $tipo) {
        $sql = "SELECT COUNT(*) as total FROM movimientos WHERE DATE(fecha) = ? AND tipo_movimiento = ?";
        $result = Database::getInstance()->selectOne($sql, [$fecha, $tipo]);
        return (int)($result['total'] ?? 0);
    }
    
    private function contarItemsMovidosPorFecha($fecha) {
        $sql = "SELECT SUM(cantidad) as total FROM movimientos WHERE DATE(fecha) = ?";
        $result = Database::getInstance()->selectOne($sql, [$fecha]);
        return (int)($result['total'] ?? 0);
    }
    
    private function contarMovimientosPorRango($fechaInicio, $fechaFin, $tipo) {
        $sql = "SELECT COUNT(*) as total FROM movimientos WHERE DATE(fecha) BETWEEN ? AND ? AND tipo_movimiento = ?";
        $result = Database::getInstance()->selectOne($sql, [$fechaInicio, $fechaFin, $tipo]);
        return (int)($result['total'] ?? 0);
    }
    
    private function contarItemsMovidosPorRango($fechaInicio, $fechaFin) {
        $sql = "SELECT SUM(cantidad) as total FROM movimientos WHERE DATE(fecha) BETWEEN ? AND ?";
        $result = Database::getInstance()->selectOne($sql, [$fechaInicio, $fechaFin]);
        return (int)($result['total'] ?? 0);
    }
    
    private function getAuditoriaIcon($accion) {
        $icons = [
            'INSERT' => 'fa-plus',
            'UPDATE' => 'fa-edit',
            'DELETE' => 'fa-trash',
            'LOGIN' => 'fa-sign-in-alt',
            'LOGOUT' => 'fa-sign-out-alt'
        ];
        
        return $icons[$accion] ?? 'fa-file-alt';
    }
    
    private function getAuditoriaColor($accion) {
        $colors = [
            'INSERT' => 'success',
            'UPDATE' => 'info',
            'DELETE' => 'danger',
            'LOGIN' => 'primary',
            'LOGOUT' => 'secondary'
        ];
        
        return $colors[$accion] ?? 'secondary';
    }
    
    // Placeholders para métricas de rendimiento
    private function calcularRotacionInventario() {
        return 2.3; // Rotaciones por mes
    }
    
    private function calcularEficienciaUsuarios() {
        return 85.7; // Porcentaje de eficiencia
    }
    
    private function calcularTiempoRespuesta() {
        return 1.2; // Segundos promedio
    }
    
    private function calcularDisponibilidad() {
        return 99.5; // Porcentaje de uptime
    }
    
    private function calcularUsoStorage() {
        return 23.4; // Porcentaje de uso
    }
    
    private function calcularCrecimientoInventario() {
        return 12.5; // Porcentaje de crecimiento mensual
    }
    
    /**
     * Calcula el valor total del inventario
     * Simulación basada en precio promedio por utensilio
     */
    private function calcularValorInventario() {
        // Simulación del valor de inventario
        $sql = "SELECT COUNT(*) as total_utensilios, SUM(cantidad_total) as total_items FROM utensilios WHERE activo = 1";
        $result = Database::getInstance()->selectOne($sql);
        
        $totalUtensilios = (int)($result['total_utensilios'] ?? 0);
        $totalItems = (int)($result['total_items'] ?? 0);
        
        // Precio promedio estimado por utensilio (simulación)
        $precioPromedio = 25.50; // Pesos
        
        return $totalItems * $precioPromedio;
    }
}