<?php

/**
 * Servicio de Inventario
 * Maneja toda la lógica de negocio relacionada con utensilios
 */
class InventarioService {
    private $utensilioRepo;
    private $movimientoRepo;
    private $authService;
    
    public function __construct() {
        $this->utensilioRepo = new UtensilioRepository();
        $this->movimientoRepo = new MovimientoRepository();
        $this->authService = new AuthService();
    }

    /**
     * Obtiene todos los utensilios con información adicional
     */
    public function obtenerTodos($filtros = []) {
        $this->authService->verificarPermiso('utensilios_ver');
        
        if (empty($filtros)) {
            return $this->utensilioRepo->getActivos();
        }

        return $this->filtrarUtensilios($filtros);
    }

    /**
     * Obtiene un utensilio por ID con validaciones
     */
    public function obtenerPorId($id) {
        $this->authService->verificarPermiso('utensilios_ver');
        
        if (!is_numeric($id) || $id <= 0) {
            throw new Exception('ID de utensilio inválido');
        }

        $utensilio = $this->utensilioRepo->getById($id);
        
        if (!$utensilio || !$utensilio->activo) {
            throw new Exception('Utensilio no encontrado');
        }

        return $utensilio;
    }

    /**
     * Crea un nuevo utensilio
     */
    public function crear($datos) {
        $this->authService->verificarPermiso('utensilios_crear');
        
        // Crear modelo para validación
        $utensilio = new Utensilio($datos);
        $errores = $utensilio->validar();
        
        if (!empty($errores)) {
            throw new Exception('Datos inválidos: ' . implode(', ', $errores));
        }

        // Verificar ubicación
        $this->validarUbicacion($datos['ubicacion']);
        
        try {
            $id = $this->utensilioRepo->crear($datos);
            
            // Registrar auditoría
            $this->registrarAuditoria('CREATE', $id, $datos);
            
            // Si hay cantidad inicial, crear movimiento de entrada
            if ($datos['cantidad_total'] > 0) {
                $this->registrarMovimientoInicial($id, $datos['cantidad_total']);
            }
            
            return $id;
            
        } catch (Exception $e) {
            throw new Exception('Error al crear utensilio: ' . $e->getMessage());
        }
    }

    /**
     * Actualiza un utensilio existente
     */
    public function actualizar($id, $datos) {
        $this->authService->verificarPermiso('utensilios_editar');
        
        $utensilioActual = $this->obtenerPorId($id);
        
        // Crear modelo temporal para validación
        $datosCompletos = array_merge($utensilioActual->toArray(), $datos);
        $utensilio = new Utensilio($datosCompletos);
        $errores = $utensilio->validar();
        
        if (!empty($errores)) {
            throw new Exception('Datos inválidos: ' . implode(', ', $errores));
        }

        try {
            $resultado = $this->utensilioRepo->actualizar($id, $datos);
            
            if ($resultado) {
                // Registrar auditoría
                $this->registrarAuditoria('UPDATE', $id, $datos, $utensilioActual->toArray());
            }
            
            return $resultado;
            
        } catch (Exception $e) {
            throw new Exception('Error al actualizar utensilio: ' . $e->getMessage());
        }
    }

    /**
     * Elimina un utensilio (marca como inactivo)
     */
    public function eliminar($id) {
        $this->authService->verificarPermiso('utensilios_eliminar');
        
        $utensilio = $this->obtenerPorId($id);
        
        // Verificar que no tenga cantidad disponible
        if ($utensilio->cantidad_disponible > 0) {
            throw new Exception('No se puede eliminar un utensilio con cantidad disponible');
        }

        try {
            $resultado = $this->utensilioRepo->eliminar($id);
            
            if ($resultado) {
                // Registrar auditoría
                $this->registrarAuditoria('DELETE', $id, null, $utensilio->toArray());
            }
            
            return $resultado;
            
        } catch (Exception $e) {
            throw new Exception('Error al eliminar utensilio: ' . $e->getMessage());
        }
    }

    /**
     * Obtiene utensilios con stock bajo
     */
    public function obtenerConStockBajo() {
        $this->authService->verificarPermiso('utensilios_ver');
        return $this->utensilioRepo->getConStockBajo();
    }

    /**
     * Obtiene utensilios que requieren atención
     */
    public function obtenerQueRequierenAtencion() {
        $this->authService->verificarPermiso('utensilios_ver');
        return $this->utensilioRepo->getQueRequierenAtencion();
    }

    /**
     * Obtiene estadísticas del inventario
     */
    public function obtenerEstadisticas() {
        $this->authService->verificarPermiso('utensilios_ver');
        
        $estadisticas = $this->utensilioRepo->getEstadisticas();
        $estadisticasPorTipo = $this->utensilioRepo->getEstadisticasPorTipo();
        $valorTotal = $this->utensilioRepo->getValorTotal();
        
        return [
            'generales' => $estadisticas,
            'por_tipo' => $estadisticasPorTipo,
            'valor_total' => $valorTotal,
            'alertas' => $this->generarAlertas()
        ];
    }

    /**
     * Busca utensilios por múltiples criterios
     */
    public function buscar($termino, $filtros = []) {
        $this->authService->verificarPermiso('utensilios_ver');
        
        if (!empty($termino)) {
            return $this->utensilioRepo->buscar($termino);
        }
        
        return $this->filtrarUtensilios($filtros);
    }

    /**
     * Obtiene ubicaciones disponibles
     */
    public function obtenerUbicaciones() {
        $this->authService->verificarPermiso('utensilios_ver');
        return $this->utensilioRepo->getUbicaciones();
    }

    /**
     * Valida disponibilidad para movimiento de salida
     */
    public function validarDisponibilidad($idUtensilio, $cantidad) {
        $utensilio = $this->obtenerPorId($idUtensilio);
        
        if ($cantidad > $utensilio->cantidad_disponible) {
            throw new Exception("Solo hay {$utensilio->cantidad_disponible} unidades disponibles");
        }
        
        if (!$utensilio->estaDisponible()) {
            throw new Exception('El utensilio no está disponible para uso');
        }
        
        return true;
    }

    /**
     * Filtra utensilios por criterios específicos
     */
    private function filtrarUtensilios($filtros) {
        $where = "activo = 1";
        $params = [];
        
        if (!empty($filtros['tipo'])) {
            $where .= " AND tipo = ?";
            $params[] = $filtros['tipo'];
        }
        
        if (!empty($filtros['estado'])) {
            $where .= " AND estado = ?";
            $params[] = $filtros['estado'];
        }
        
        if (!empty($filtros['ubicacion'])) {
            $where .= " AND ubicacion = ?";
            $params[] = $filtros['ubicacion'];
        }
        
        if (isset($filtros['stock_bajo']) && $filtros['stock_bajo']) {
            $where .= " AND cantidad_disponible <= cantidad_minima";
        }
        
        return $this->utensilioRepo->findWhere($where, $params, 'nombre', 'ASC');
    }

    /**
     * Valida una ubicación
     */
    private function validarUbicacion($ubicacion) {
        if (empty($ubicacion)) {
            throw new Exception('La ubicación es requerida');
        }
        
        if (strlen($ubicacion) > 100) {
            throw new Exception('La ubicación no puede exceder 100 caracteres');
        }
        
        return true;
    }

    /**
     * Registra el movimiento inicial de un utensilio
     */
    private function registrarMovimientoInicial($idUtensilio, $cantidad) {
        $usuario = $this->authService->getUsuarioActual();
        
        $movimientoService = new MovimientoService();
        $movimientoService->registrarEntrada($idUtensilio, $cantidad, 'Inventario inicial', 'Registro inicial en el sistema', $usuario->id);
    }

    /**
     * Genera alertas del sistema
     */
    private function generarAlertas() {
        $alertas = [];
        
        // Alertas de stock bajo
        $stockBajo = $this->utensilioRepo->getConStockBajo();
        foreach ($stockBajo as $utensilio) {
            $alertas[] = [
                'tipo' => 'stock_bajo',
                'mensaje' => "Stock bajo: {$utensilio->nombre} ({$utensilio->cantidad_disponible} disponibles)",
                'nivel' => 'warning',
                'utensilio_id' => $utensilio->id
            ];
        }
        
        // Alertas de estado
        $estadosMalos = $this->utensilioRepo->getByEstado('malo');
        foreach ($estadosMalos as $utensilio) {
            $alertas[] = [
                'tipo' => 'estado_malo',
                'mensaje' => "Utensilio en mal estado: {$utensilio->nombre}",
                'nivel' => 'danger',
                'utensilio_id' => $utensilio->id
            ];
        }
        
        return $alertas;
    }

    /**
     * Registra auditoría de cambios
     */
    private function registrarAuditoria($accion, $idUtensilio, $datosNuevos = null, $datosAnteriores = null) {
        try {
            $usuario = $this->authService->getUsuarioActual();
            
            // Aquí se registraría en la tabla de auditoría
            error_log("Auditoría - Acción: {$accion}, Utensilio ID: {$idUtensilio}, Usuario: {$usuario->nombre}");
            
        } catch (Exception $e) {
            error_log("Error registrando auditoría: " . $e->getMessage());
        }
    }
}