<?php

/**
 * Servicio de Movimientos
 * Maneja toda la lógica de negocio para entradas y salidas de inventario
 */
class MovimientoService {
    private $movimientoRepo;
    private $utensilioRepo;
    private $authService;
    
    public function __construct() {
        $this->movimientoRepo = new MovimientoRepository();
        $this->utensilioRepo = new UtensilioRepository();
        $this->authService = new AuthService();
    }

    /**
     * Registra una entrada de inventario
     */
    public function registrarEntrada($idUtensilio, $cantidad, $motivo, $detalle = '', $usuarioId = null) {
        $this->authService->verificarPermiso('movimientos_crear');
        
        // Validaciones básicas
        $this->validarDatos($idUtensilio, $cantidad, $motivo);
        
        // Obtener usuario actual si no se especifica
        if ($usuarioId === null) {
            $usuario = $this->authService->getUsuarioActual();
            $usuarioId = $usuario->id;
        }
        
        $datos = [
            'id_utensilio' => $idUtensilio,
            'tipo_movimiento' => 'entrada',
            'cantidad' => $cantidad,
            'motivo' => $motivo,
            'detalle_motivo' => $detalle,
            'usuario_id' => $usuarioId
        ];

        try {
            // Registrar movimiento (el repositorio maneja la transacción)
            $movimientoId = $this->movimientoRepo->registrarMovimiento($datos);
            
            // Verificar y generar alertas si es necesario
            $this->verificarAlertas($idUtensilio);
            
            // Registrar auditoría
            $this->registrarAuditoria('ENTRADA', $movimientoId, $datos);
            
            return $movimientoId;
            
        } catch (Exception $e) {
            throw new Exception('Error al registrar entrada: ' . $e->getMessage());
        }
    }

    /**
     * Registra una salida de inventario
     */
    public function registrarSalida($idUtensilio, $cantidad, $motivo, $detalle = '', $usuarioId = null) {
        $this->authService->verificarPermiso('movimientos_crear');
        
        // Validaciones básicas
        $this->validarDatos($idUtensilio, $cantidad, $motivo);
        
        // Validar disponibilidad
        $utensilio = $this->utensilioRepo->getById($idUtensilio);
        if (!$utensilio) {
            throw new Exception('Utensilio no encontrado');
        }
        
        if ($cantidad > $utensilio->cantidad_disponible) {
            throw new Exception("Solo hay {$utensilio->cantidad_disponible} unidades disponibles");
        }
        
        // Obtener usuario actual si no se especifica
        if ($usuarioId === null) {
            $usuario = $this->authService->getUsuarioActual();
            $usuarioId = $usuario->id;
        }
        
        $datos = [
            'id_utensilio' => $idUtensilio,
            'tipo_movimiento' => 'salida',
            'cantidad' => $cantidad,
            'motivo' => $motivo,
            'detalle_motivo' => $detalle,
            'usuario_id' => $usuarioId
        ];

        try {
            // Registrar movimiento
            $movimientoId = $this->movimientoRepo->registrarMovimiento($datos);
            
            // Verificar y generar alertas si es necesario
            $this->verificarAlertas($idUtensilio);
            
            // Registrar auditoría
            $this->registrarAuditoria('SALIDA', $movimientoId, $datos);
            
            return $movimientoId;
            
        } catch (Exception $e) {
            throw new Exception('Error al registrar salida: ' . $e->getMessage());
        }
    }

    /**
     * Obtiene el historial de movimientos
     */
    public function obtenerHistorial($filtros = [], $limite = 50) {
        $this->authService->verificarPermiso('movimientos_ver');
        
        if (empty($filtros)) {
            return $this->movimientoRepo->getConDetalles($limite);
        }
        
        return $this->movimientoRepo->buscar($filtros);
    }

    /**
     * Obtiene movimientos de un utensilio específico
     */
    public function obtenerHistorialUtensilio($idUtensilio, $limite = 20) {
        $this->authService->verificarPermiso('movimientos_ver');
        
        if (!is_numeric($idUtensilio) || $idUtensilio <= 0) {
            throw new Exception('ID de utensilio inválido');
        }
        
        return $this->movimientoRepo->getByUtensilio($idUtensilio, $limite);
    }

    /**
     * Obtiene movimientos recientes
     */
    public function obtenerRecientes($limite = 10) {
        $this->authService->verificarPermiso('movimientos_ver');
        return $this->movimientoRepo->getRecientes($limite);
    }

    /**
     * Obtiene estadísticas de movimientos
     */
    public function obtenerEstadisticas($dias = 30) {
        $this->authService->verificarPermiso('movimientos_ver');
        
        $estadisticas = $this->movimientoRepo->getEstadisticas($dias);
        $actividadDiaria = $this->movimientoRepo->getActividadDiaria(7);
        $motivosComunes = $this->movimientoRepo->getMotivosMasComunes(null, 5);
        
        return [
            'generales' => $estadisticas,
            'actividad_diaria' => $actividadDiaria,
            'motivos_comunes' => $motivosComunes,
            'resumen' => $this->generarResumenEstadisticas($estadisticas)
        ];
    }

    /**
     * Busca movimientos por criterios
     */
    public function buscar($filtros) {
        $this->authService->verificarPermiso('movimientos_ver');
        return $this->movimientoRepo->buscar($filtros);
    }

    /**
     * Obtiene un movimiento específico por ID
     */
    public function obtenerPorId($id) {
        $this->authService->verificarPermiso('movimientos_ver');
        
        if (!is_numeric($id) || $id <= 0) {
            throw new Exception('ID de movimiento inválido');
        }

        $movimiento = $this->movimientoRepo->getById($id);
        
        if (!$movimiento) {
            throw new Exception('Movimiento no encontrado');
        }

        return $movimiento;
    }

    /**
     * Obtiene movimientos por rango de fechas
     */
    public function obtenerPorFechas($fechaInicio, $fechaFin) {
        $this->authService->verificarPermiso('movimientos_ver');
        
        // Validar fechas
        if (!$this->validarFecha($fechaInicio) || !$this->validarFecha($fechaFin)) {
            throw new Exception('Formato de fecha inválido (usar YYYY-MM-DD)');
        }
        
        if ($fechaInicio > $fechaFin) {
            throw new Exception('La fecha de inicio no puede ser mayor a la fecha de fin');
        }
        
        return $this->movimientoRepo->getByRangoFecha($fechaInicio, $fechaFin);
    }

    /**
     * Corrige un movimiento (solo administradores)
     */
    public function corregirMovimiento($idMovimiento, $observacion) {
        if (!$this->authService->esAdministrador()) {
            throw new Exception('Solo los administradores pueden corregir movimientos');
        }
        
        // Esta función requeriría lógica adicional para revertir el movimiento
        // y crear uno nuevo con la corrección
        throw new Exception('Función de corrección no implementada aún');
    }

    /**
     * Obtiene motivos sugeridos según el tipo de movimiento
     */
    public function obtenerMotivosSugeridos($tipoMovimiento) {
        if ($tipoMovimiento === 'entrada') {
            return Movimiento::getMotivosEntrada();
        } elseif ($tipoMovimiento === 'salida') {
            return Movimiento::getMotivosSalida();
        }
        
        return [];
    }

    /**
     * Valida los datos básicos del movimiento
     */
    private function validarDatos($idUtensilio, $cantidad, $motivo) {
        if (!is_numeric($idUtensilio) || $idUtensilio <= 0) {
            throw new Exception('ID de utensilio inválido');
        }
        
        if (!is_numeric($cantidad) || $cantidad <= 0) {
            throw new Exception('La cantidad debe ser un número mayor a 0');
        }
        
        if (empty($motivo)) {
            throw new Exception('El motivo es requerido');
        }
        
        if (strlen($motivo) > 200) {
            throw new Exception('El motivo no puede exceder 200 caracteres');
        }
    }

    /**
     * Valida formato de fecha
     */
    private function validarFecha($fecha) {
        $d = DateTime::createFromFormat('Y-m-d', $fecha);
        return $d && $d->format('Y-m-d') === $fecha;
    }

    /**
     * Verifica y genera alertas después de un movimiento
     */
    private function verificarAlertas($idUtensilio) {
        $utensilio = $this->utensilioRepo->getById($idUtensilio);
        
        if (!$utensilio) return;
        
        // Alerta de stock bajo
        if ($utensilio->tieneStockBajo()) {
            $this->generarAlerta([
                'tipo' => 'stock_bajo',
                'id_utensilio' => $idUtensilio,
                'mensaje' => "Stock bajo: {$utensilio->nombre} ({$utensilio->cantidad_disponible} disponibles)",
                'nivel' => 'warning'
            ]);
        }
        
        // Alerta de stock agotado
        if ($utensilio->cantidad_disponible == 0) {
            $this->generarAlerta([
                'tipo' => 'stock_agotado',
                'id_utensilio' => $idUtensilio,
                'mensaje' => "Stock agotado: {$utensilio->nombre}",
                'nivel' => 'critical'
            ]);
        }
    }

    /**
     * Genera una alerta en el sistema
     */
    private function generarAlerta($datos) {
        try {
            // Aquí se insertaría en la tabla de alertas
            error_log("Alerta generada: " . json_encode($datos));
        } catch (Exception $e) {
            error_log("Error generando alerta: " . $e->getMessage());
        }
    }

    /**
     * Genera resumen de estadísticas
     */
    private function generarResumenEstadisticas($estadisticas) {
        $balanceItems = $estadisticas['items_ingresados'] - $estadisticas['items_retirados'];
        $porcentajeEntradas = $estadisticas['total_movimientos'] > 0 
            ? round(($estadisticas['entradas'] / $estadisticas['total_movimientos']) * 100, 2)
            : 0;
        
        return [
            'balance_items' => $balanceItems,
            'porcentaje_entradas' => $porcentajeEntradas,
            'porcentaje_salidas' => 100 - $porcentajeEntradas,
            'promedio_movimientos_dia' => round($estadisticas['total_movimientos'] / 30, 2)
        ];
    }

    /**
     * Registra auditoría de movimientos
     */
    private function registrarAuditoria($tipo, $movimientoId, $datos) {
        try {
            $usuario = $this->authService->getUsuarioActual();
            
            error_log("Auditoría Movimiento - Tipo: {$tipo}, ID: {$movimientoId}, Usuario: {$usuario->nombre}");
            
        } catch (Exception $e) {
            error_log("Error registrando auditoría de movimiento: " . $e->getMessage());
        }
    }

    /**
     * Exporta movimientos a formato array para reportes
     */
    public function exportarParaReporte($filtros = []) {
        $this->authService->verificarPermiso('reportes_ver');
        
        $movimientos = $this->buscar($filtros);
        $datos = [];
        
        foreach ($movimientos as $movimiento) {
            $datos[] = [
                'fecha' => $movimiento->getFechaFormateada(),
                'utensilio' => $movimiento->nombre_utensilio ?? 'N/A',
                'tipo' => ucfirst($movimiento->tipo_movimiento),
                'cantidad' => $movimiento->getImpacto(),
                'motivo' => $movimiento->motivo,
                'usuario' => $movimiento->nombre_usuario ?? 'N/A',
                'observaciones' => $movimiento->observaciones
            ];
        }
        
        return $datos;
    }
}