<?php

// No require directo - se carga con autoloader

/**
 * Servicio para generar reportes del sistema
 */
class ReporteService {
    
    private $authService;
    private $movimientoRepo;
    private $utensilioRepo;
    private $usuarioRepo;
    
    public function __construct() {
        $this->authService = new AuthService();
        $this->movimientoRepo = new MovimientoRepository();
        $this->utensilioRepo = new UtensilioRepository();
        $this->usuarioRepo = new UsuarioRepository();
    }
    
    /**
     * Genera reporte de inventario actual
     */
    public function generarReporteInventario($formato = 'array') {
        $this->authService->verificarPermiso('reportes_ver');
        
        $utensilios = $this->utensilioRepo->getConDetalles();
        $data = [];
        
        $totalItems = 0;
        $valorTotal = 0;
        $stockBajo = 0;
        $stockAgotado = 0;
        
        foreach ($utensilios as $utensilio) {
            $valorItem = $utensilio->cantidad_disponible * ($utensilio->precio ?? 0);
            $totalItems += $utensilio->cantidad_disponible;
            $valorTotal += $valorItem;
            
            if ($utensilio->cantidad_disponible == 0) {
                $stockAgotado++;
            } elseif ($utensilio->tieneStockBajo()) {
                $stockBajo++;
            }
            
            $data[] = [
                'nombre' => $utensilio->nombre,
                'tipo' => $utensilio->tipo,
                'cantidad_total' => $utensilio->cantidad_total,
                'cantidad_disponible' => $utensilio->cantidad_disponible,
                'cantidad_en_uso' => $utensilio->cantidad_total - $utensilio->cantidad_disponible,
                'estado' => $utensilio->estado,
                'ubicacion' => $utensilio->ubicacion ?? 'No especificada',
                'precio_unitario' => $utensilio->precio ?? 0,
                'valor_inventario' => $valorItem,
                'stock_minimo' => $utensilio->stock_minimo ?? 0,
                'estado_stock' => $this->getEstadoStock($utensilio),
                'ultima_actualizacion' => $utensilio->updated_at
            ];
        }
        
        $resumen = [
            'fecha_generacion' => date('Y-m-d H:i:s'),
            'total_utensilios' => count($data),
            'total_items' => $totalItems,
            'valor_total_inventario' => $valorTotal,
            'alertas' => [
                'stock_bajo' => $stockBajo,
                'stock_agotado' => $stockAgotado
            ]
        ];
        
        return [
            'resumen' => $resumen,
            'detalles' => $data
        ];
    }
    
    /**
     * Genera reporte de movimientos por período
     */
    public function generarReporteMovimientos($fechaInicio, $fechaFin, $filtros = []) {
        $this->authService->verificarPermiso('reportes_ver');
        
        $filtrosBusqueda = array_merge($filtros, [
            'fecha_inicio' => $fechaInicio,
            'fecha_fin' => $fechaFin
        ]);
        
        $movimientos = $this->movimientoRepo->buscar($filtrosBusqueda);
        $data = [];
        
        $totalEntradas = 0;
        $totalSalidas = 0;
        $itemsIngresados = 0;
        $itemsRetirados = 0;
        $usuariosActivos = [];
        $utensiliosAfectados = [];
        
        foreach ($movimientos as $movimiento) {
            if ($movimiento->tipo_movimiento === 'entrada') {
                $totalEntradas++;
                $itemsIngresados += $movimiento->cantidad;
            } else {
                $totalSalidas++;
                $itemsRetirados += $movimiento->cantidad;
            }
            
            $usuariosActivos[$movimiento->usuario_id] = true;
            $utensiliosAfectados[$movimiento->id_utensilio] = true;
            
            $data[] = [
                'fecha' => $movimiento->fecha,
                'tipo' => $movimiento->tipo_movimiento,
                'utensilio' => $movimiento->nombre_utensilio ?? 'N/A',
                'cantidad' => $movimiento->cantidad,
                'impacto' => $movimiento->getImpacto(),
                'motivo' => $movimiento->motivo,
                'usuario' => $movimiento->nombre_usuario ?? 'N/A',
                'observaciones' => $movimiento->observaciones
            ];
        }
        
        $resumen = [
            'periodo' => [
                'inicio' => $fechaInicio,
                'fin' => $fechaFin
            ],
            'totales' => [
                'movimientos' => count($data),
                'entradas' => $totalEntradas,
                'salidas' => $totalSalidas,
                'items_ingresados' => $itemsIngresados,
                'items_retirados' => $itemsRetirados,
                'balance_neto' => $itemsIngresados - $itemsRetirados
            ],
            'actividad' => [
                'usuarios_activos' => count($usuariosActivos),
                'utensilios_afectados' => count($utensiliosAfectados),
                'promedio_movimientos_dia' => round(count($data) / max(1, $this->getDiasEntreFechas($fechaInicio, $fechaFin)), 2)
            ]
        ];
        
        return [
            'resumen' => $resumen,
            'detalles' => $data
        ];
    }
    
    /**
     * Genera reporte de actividad por usuario
     */
    public function generarReporteUsuarios($fechaInicio = null, $fechaFin = null) {
        $this->authService->verificarPermiso('reportes_ver');
        
        $filtros = [];
        if ($fechaInicio && $fechaFin) {
            $filtros['fecha_inicio'] = $fechaInicio;
            $filtros['fecha_fin'] = $fechaFin;
        }
        
        $usuarios = $this->usuarioRepo->getAll();
        $data = [];
        
        foreach ($usuarios as $usuario) {
            $movimientosUsuario = $this->movimientoRepo->getByUsuario($usuario->id, 1000);
            
            $entradas = 0;
            $salidas = 0;
            $itemsMovidos = 0;
            $ultimaActividad = null;
            
            foreach ($movimientosUsuario as $mov) {
                if ($fechaInicio && $fechaFin) {
                    if ($mov->fecha < $fechaInicio || $mov->fecha > $fechaFin) continue;
                }
                
                if ($mov->tipo_movimiento === 'entrada') {
                    $entradas++;
                } else {
                    $salidas++;
                }
                
                $itemsMovidos += $mov->cantidad;
                
                if (!$ultimaActividad || $mov->fecha > $ultimaActividad) {
                    $ultimaActividad = $mov->fecha;
                }
            }
            
            $data[] = [
                'usuario' => $usuario->nombre,
                'email' => $usuario->email,
                'rol' => $usuario->rol,
                'total_movimientos' => $entradas + $salidas,
                'entradas' => $entradas,
                'salidas' => $salidas,
                'items_movidos' => $itemsMovidos,
                'ultima_actividad' => $ultimaActividad,
                'estado' => $usuario->activo ? 'Activo' : 'Inactivo'
            ];
        }
        
        // Ordenar por total de movimientos
        usort($data, function($a, $b) {
            return $b['total_movimientos'] - $a['total_movimientos'];
        });
        
        $resumen = [
            'periodo' => $fechaInicio && $fechaFin ? ['inicio' => $fechaInicio, 'fin' => $fechaFin] : 'Todo el período',
            'total_usuarios' => count($data),
            'usuarios_activos' => count(array_filter($data, function($u) { return $u['estado'] === 'Activo'; })),
            'total_movimientos' => array_sum(array_column($data, 'total_movimientos')),
            'usuario_mas_activo' => $data[0] ?? null
        ];
        
        return [
            'resumen' => $resumen,
            'detalles' => $data
        ];
    }
    
    /**
     * Genera reporte de alertas y notificaciones
     */
    public function generarReporteAlertas() {
        $this->authService->verificarPermiso('reportes_ver');
        
        $utensilios = $this->utensilioRepo->getAll();
        $alertas = [];
        
        foreach ($utensilios as $utensilio) {
            if ($utensilio->cantidad_disponible == 0) {
                $alertas[] = [
                    'tipo' => 'CRITICA',
                    'nivel' => 'danger',
                    'utensilio' => $utensilio->nombre,
                    'cantidad_actual' => $utensilio->cantidad_disponible,
                    'stock_minimo' => $utensilio->stock_minimo ?? 0,
                    'mensaje' => 'Stock agotado',
                    'recomendacion' => 'Reabastecer inmediatamente'
                ];
            } elseif ($utensilio->tieneStockBajo()) {
                $alertas[] = [
                    'tipo' => 'ADVERTENCIA',
                    'nivel' => 'warning',
                    'utensilio' => $utensilio->nombre,
                    'cantidad_actual' => $utensilio->cantidad_disponible,
                    'stock_minimo' => $utensilio->stock_minimo ?? 0,
                    'mensaje' => 'Stock bajo',
                    'recomendacion' => 'Considerar reabastecimiento'
                ];
            }
            
            // Verificar inactividad (sin movimientos recientes)
            $movimientosRecientes = $this->movimientoRepo->getByUtensilio($utensilio->id, 1);
            if (empty($movimientosRecientes)) {
                $alertas[] = [
                    'tipo' => 'INFO',
                    'nivel' => 'info',
                    'utensilio' => $utensilio->nombre,
                    'cantidad_actual' => $utensilio->cantidad_disponible,
                    'stock_minimo' => $utensilio->stock_minimo ?? 0,
                    'mensaje' => 'Sin actividad reciente',
                    'recomendacion' => 'Revisar si sigue siendo necesario'
                ];
            } else {
                $ultimoMovimiento = $movimientosRecientes[0];
                $diasSinActividad = $this->getDiasEntreFechas($ultimoMovimiento->fecha, date('Y-m-d'));
                
                if ($diasSinActividad > 30) {
                    $alertas[] = [
                        'tipo' => 'INFO',
                        'nivel' => 'info',
                        'utensilio' => $utensilio->nombre,
                        'cantidad_actual' => $utensilio->cantidad_disponible,
                        'stock_minimo' => $utensilio->stock_minimo ?? 0,
                        'mensaje' => "Sin actividad por {$diasSinActividad} días",
                        'recomendacion' => 'Evaluar frecuencia de uso'
                    ];
                }
            }
        }
        
        // Clasificar alertas por prioridad
        $criticas = array_filter($alertas, function($a) { return $a['tipo'] === 'CRITICA'; });
        $advertencias = array_filter($alertas, function($a) { return $a['tipo'] === 'ADVERTENCIA'; });
        $informativas = array_filter($alertas, function($a) { return $a['tipo'] === 'INFO'; });
        
        $resumen = [
            'fecha_generacion' => date('Y-m-d H:i:s'),
            'total_alertas' => count($alertas),
            'por_tipo' => [
                'criticas' => count($criticas),
                'advertencias' => count($advertencias),
                'informativas' => count($informativas)
            ]
        ];
        
        return [
            'resumen' => $resumen,
            'alertas' => $alertas
        ];
    }
    
    /**
     * Genera reporte ejecutivo (resumen general)
     */
    public function generarReporteEjecutivo($periodo = 'mes') {
        $this->authService->verificarPermiso('reportes_ver');
        
        // Calcular fechas según el período
        $fechaFin = date('Y-m-d');
        switch ($periodo) {
            case 'semana':
                $fechaInicio = date('Y-m-d', strtotime('-7 days'));
                break;
            case 'mes':
                $fechaInicio = date('Y-m-d', strtotime('-30 days'));
                break;
            case 'trimestre':
                $fechaInicio = date('Y-m-d', strtotime('-90 days'));
                break;
            case 'año':
                $fechaInicio = date('Y-m-d', strtotime('-365 days'));
                break;
            default:
                $fechaInicio = date('Y-m-d', strtotime('-30 days'));
        }
        
        // Obtener datos de diferentes fuentes
        $reporteInventario = $this->generarReporteInventario();
        $reporteMovimientos = $this->generarReporteMovimientos($fechaInicio, $fechaFin);
        $reporteUsuarios = $this->generarReporteUsuarios($fechaInicio, $fechaFin);
        $reporteAlertas = $this->generarReporteAlertas();
        
        return [
            'periodo' => ucfirst($periodo),
            'fechas' => ['inicio' => $fechaInicio, 'fin' => $fechaFin],
            'inventario' => $reporteInventario['resumen'],
            'movimientos' => $reporteMovimientos['resumen'],
            'usuarios' => $reporteUsuarios['resumen'],
            'alertas' => $reporteAlertas['resumen'],
            'kpis' => [
                'rotacion_inventario' => $this->calcularRotacionInventario($fechaInicio, $fechaFin),
                'eficiencia_usuarios' => $this->calcularEficienciaUsuarios($fechaInicio, $fechaFin),
                'disponibilidad_stock' => $this->calcularDisponibilidadStock(),
                'tendencia_movimientos' => $this->calcularTendenciaMovimientos($fechaInicio, $fechaFin)
            ]
        ];
    }
    
    /**
     * Exporta reporte a formato CSV
     */
    public function exportarCSV($data, $filename) {
        $this->authService->verificarPermiso('reportes_exportar');
        
        $filepath = REPORTS_PATH . '/' . $filename . '_' . date('Y-m-d_H-i-s') . '.csv';
        
        // Crear directorio si no existe
        if (!is_dir(REPORTS_PATH)) {
            mkdir(REPORTS_PATH, 0755, true);
        }
        
        $file = fopen($filepath, 'w');
        
        if (empty($data)) {
            fclose($file);
            return $filepath;
        }
        
        // Escribir encabezados
        fputcsv($file, array_keys($data[0]));
        
        // Escribir datos
        foreach ($data as $row) {
            fputcsv($file, $row);
        }
        
        fclose($file);
        return $filepath;
    }
    
    // Métodos auxiliares
    private function getEstadoStock($utensilio) {
        if ($utensilio->cantidad_disponible == 0) {
            return 'AGOTADO';
        } elseif ($utensilio->tieneStockBajo()) {
            return 'STOCK_BAJO';
        } else {
            return 'NORMAL';
        }
    }
    
    private function getDiasEntreFechas($fecha1, $fecha2) {
        $d1 = new DateTime($fecha1);
        $d2 = new DateTime($fecha2);
        return $d2->diff($d1)->days;
    }
    
    private function calcularRotacionInventario($fechaInicio, $fechaFin) {
        // Implementar cálculo de rotación de inventario
        return 0; // Placeholder
    }
    
    private function calcularEficienciaUsuarios($fechaInicio, $fechaFin) {
        // Implementar cálculo de eficiencia de usuarios
        return 0; // Placeholder
    }
    
    private function calcularDisponibilidadStock() {
        // Implementar cálculo de disponibilidad de stock
        return 0; // Placeholder
    }
    
    private function calcularTendenciaMovimientos($fechaInicio, $fechaFin) {
        // Implementar cálculo de tendencia de movimientos
        return 'ESTABLE'; // Placeholder
    }
}