<?php

require_once 'BaseRepository.php';

/**
 * Repositorio para la gestión de auditoría
 */
class AuditoriaRepository extends BaseRepository {
    
    public function __construct() {
        parent::__construct();
        $this->table = 'auditoria';
        $this->model = 'stdClass'; // Usando stdClass para simplicidad
    }

    /**
     * Registra un evento de auditoría
     */
    public function registrar($datos) {
        // Campos automáticos
        $datos['fecha'] = date('Y-m-d H:i:s');
        $datos['ip_address'] = $_SERVER['REMOTE_ADDR'] ?? null;
        $datos['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? null;
        
        return $this->insert($datos);
    }

    /**
     * Obtiene registros de auditoría con filtros
     */
    public function obtenerConFiltros($filtros = []) {
        $where = "1=1";
        $params = [];
        
        if (!empty($filtros['tabla'])) {
            $where .= " AND tabla_afectada = ?";
            $params[] = $filtros['tabla'];
        }
        
        if (!empty($filtros['accion'])) {
            $where .= " AND accion = ?";
            $params[] = $filtros['accion'];
        }
        
        if (!empty($filtros['usuario_id'])) {
            $where .= " AND usuario_id = ?";
            $params[] = $filtros['usuario_id'];
        }
        
        if (!empty($filtros['fecha_desde'])) {
            $where .= " AND DATE(fecha) >= ?";
            $params[] = $filtros['fecha_desde'];
        }
        
        if (!empty($filtros['fecha_hasta'])) {
            $where .= " AND DATE(fecha) <= ?";
            $params[] = $filtros['fecha_hasta'];
        }
        
        $sql = "SELECT 
                    a.*,
                    u.nombre as nombre_usuario
                FROM {$this->table} a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                WHERE {$where}
                ORDER BY a.fecha DESC";
        
        return $this->db->select($sql, $params);
    }

    /**
     * Obtiene actividad por usuario
     */
    public function getActividadPorUsuario($dias = 30) {
        $sql = "SELECT 
                    u.nombre,
                    COUNT(*) as total_acciones,
                    MAX(a.fecha) as ultima_actividad
                FROM {$this->table} a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                WHERE a.fecha >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY a.usuario_id, u.nombre
                ORDER BY total_acciones DESC";
        
        return $this->db->select($sql, [$dias]);
    }

    /**
     * Obtiene estadísticas de auditoría
     */
    public function getEstadisticas($dias = 30) {
        $sql = "SELECT 
                    COUNT(*) as total_eventos,
                    COUNT(DISTINCT usuario_id) as usuarios_activos,
                    COUNT(DISTINCT tabla_afectada) as tablas_afectadas,
                    SUM(CASE WHEN accion = 'INSERT' THEN 1 ELSE 0 END) as inserciones,
                    SUM(CASE WHEN accion = 'UPDATE' THEN 1 ELSE 0 END) as actualizaciones,
                    SUM(CASE WHEN accion = 'DELETE' THEN 1 ELSE 0 END) as eliminaciones
                FROM {$this->table}
                WHERE fecha >= DATE_SUB(NOW(), INTERVAL ? DAY)";
        
        return $this->db->selectOne($sql, [$dias]);
    }

    /**
     * Elimina registros antiguos de auditoría
     */
    public function eliminarRegistrosAntiguos($diasRetener = 365) {
        $sql = "DELETE FROM {$this->table} WHERE fecha < DATE_SUB(NOW(), INTERVAL ? DAY)";
        return $this->db->execute($sql, [$diasRetener]);
    }
}