<?php

/**
 * Repositorio base con funciones comunes
 */
abstract class BaseRepository {
    protected $db;
    protected $table;
    protected $model;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los registros
     */
    public function getAll($orderBy = 'id', $order = 'ASC') {
        $sql = "SELECT * FROM {$this->table} ORDER BY {$orderBy} {$order}";
        $data = $this->db->select($sql);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene un registro por ID
     */
    public function getById($id) {
        $sql = "SELECT * FROM {$this->table} WHERE id = ?";
        $data = $this->db->selectOne($sql, [$id]);
        return $data ? new $this->model($data) : null;
    }

    /**
     * Obtiene registros con paginación
     */
    public function getPaginated($page = 1, $limit = 10, $orderBy = 'id', $order = 'ASC') {
        $offset = ($page - 1) * $limit;
        $sql = "SELECT * FROM {$this->table} ORDER BY {$orderBy} {$order} LIMIT ? OFFSET ?";
        $data = $this->db->select($sql, [$limit, $offset]);
        
        // Obtener total de registros
        $total = $this->count();
        
        return [
            'data' => $this->mapToModels($data),
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ];
    }

    /**
     * Cuenta el total de registros
     */
    public function count($where = '', $params = []) {
        $sql = "SELECT COUNT(*) as total FROM {$this->table}";
        if (!empty($where)) {
            $sql .= " WHERE {$where}";
        }
        $result = $this->db->selectOne($sql, $params);
        return $result['total'];
    }

    /**
     * Inserta un nuevo registro
     */
    public function insert($data) {
        $fields = array_keys($data);
        $placeholders = array_fill(0, count($fields), '?');
        
        $sql = "INSERT INTO {$this->table} (" . implode(', ', $fields) . ") 
                VALUES (" . implode(', ', $placeholders) . ")";
        
        return $this->db->insert($sql, array_values($data));
    }

    /**
     * Actualiza un registro por ID
     */
    public function update($id, $data) {
        $fields = array_keys($data);
        $setClause = implode(' = ?, ', $fields) . ' = ?';
        
        $sql = "UPDATE {$this->table} SET {$setClause} WHERE id = ?";
        $params = array_merge(array_values($data), [$id]);
        
        return $this->db->execute($sql, $params);
    }

    /**
     * Elimina un registro por ID
     */
    public function delete($id) {
        $sql = "DELETE FROM {$this->table} WHERE id = ?";
        return $this->db->execute($sql, [$id]);
    }

    /**
     * Elimina registros que cumplan una condición
     */
    public function deleteWhere($where, $params = []) {
        $sql = "DELETE FROM {$this->table} WHERE {$where}";
        return $this->db->execute($sql, $params);
    }

    /**
     * Busca registros que cumplan una condición
     */
    public function findWhere($where, $params = [], $orderBy = 'id', $order = 'ASC') {
        $sql = "SELECT * FROM {$this->table} WHERE {$where} ORDER BY {$orderBy} {$order}";
        $data = $this->db->select($sql, $params);
        return $this->mapToModels($data);
    }

    /**
     * Busca un registro que cumpla una condición
     */
    public function findOneWhere($where, $params = []) {
        $sql = "SELECT * FROM {$this->table} WHERE {$where} LIMIT 1";
        $data = $this->db->selectOne($sql, $params);
        return $data ? new $this->model($data) : null;
    }

    /**
     * Verifica si existe un registro que cumpla una condición
     */
    public function exists($where, $params = []) {
        $sql = "SELECT 1 FROM {$this->table} WHERE {$where} LIMIT 1";
        $result = $this->db->selectOne($sql, $params);
        return !empty($result);
    }

    /**
     * Convierte array de datos a array de modelos
     */
    protected function mapToModels($data) {
        $models = [];
        foreach ($data as $row) {
            $models[] = new $this->model($row);
        }
        return $models;
    }

    /**
     * Inicia una transacción
     */
    public function beginTransaction() {
        return $this->db->beginTransaction();
    }

    /**
     * Confirma una transacción
     */
    public function commit() {
        return $this->db->commit();
    }

    /**
     * Revierte una transacción
     */
    public function rollback() {
        return $this->db->rollback();
    }

    /**
     * Ejecuta una consulta SQL personalizada
     */
    protected function query($sql, $params = []) {
        return $this->db->query($sql, $params);
    }

    /**
     * Ejecuta una consulta SELECT personalizada
     */
    protected function select($sql, $params = []) {
        return $this->db->select($sql, $params);
    }

    /**
     * Ejecuta una consulta SELECT que retorna un solo resultado
     */
    protected function selectOne($sql, $params = []) {
        return $this->db->selectOne($sql, $params);
    }
}