<?php

// No require directo - se carga con autoloader

/**
 * Clase de conexión a la base de datos
 * Patrón Singleton para mantener una sola conexión
 * Incluye monitoreo de rendimiento y optimizaciones
 */
class Database {
    private static $instance = null;
    private $connection;
    private $host;
    private $dbname;
    private $username;
    private $password;
    private $charset;
    private $monitor;

    private function __construct() {
        $this->host = DB_HOST;
        $this->dbname = DB_NAME;
        $this->username = DB_USER;
        $this->password = DB_PASS;
        $this->charset = DB_CHARSET;
        $this->monitor = PerformanceMonitor::getInstance();
        
        $this->connect();
    }

    /**
     * Obtiene la instancia única de la conexión
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Establece la conexión a la base de datos
     */
    private function connect() {
        try {
            $dsn = "mysql:host={$this->host};dbname={$this->dbname};charset={$this->charset}";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES {$this->charset}"
            ];

            $this->connection = new PDO($dsn, $this->username, $this->password, $options);
            
            if (DEBUG_MODE) {
                error_log("Conexión a base de datos establecida exitosamente");
            }
        } catch (PDOException $e) {
            error_log("Error de conexión a la base de datos: " . $e->getMessage());
            throw new Exception("Error de conexión a la base de datos");
        }
    }

    /**
     * Obtiene la conexión PDO
     */
    public function getConnection() {
        // Verificar si la conexión sigue activa
        if ($this->connection === null) {
            $this->connect();
        }
        
        return $this->connection;
    }

    /**
     * Ejecuta una consulta preparada con monitoreo de rendimiento
     */
    public function query($sql, $params = []) {
        $startTime = microtime(true);
        
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            
            $executionTime = microtime(true) - $startTime;
            $rowCount = $stmt->rowCount();
            
            // Registrar consulta para análisis de rendimiento
            $this->monitor->logQuery($sql, $params, $executionTime, $rowCount);
            
            return $stmt;
        } catch (PDOException $e) {
            $executionTime = microtime(true) - $startTime;
            $this->monitor->logQuery($sql, $params, $executionTime, 0);
            
            error_log("Error en consulta SQL: " . $e->getMessage() . " | SQL: " . $sql);
            throw new Exception("Error en la consulta: " . $e->getMessage());
        }
    }
    
    /**
     * Obtiene estadísticas de rendimiento de la base de datos
     */
    public function getPerformanceStats() {
        return $this->monitor->getPerformanceStats();
    }
    
    /**
     * Genera reporte de rendimiento
     */
    public function getPerformanceReport() {
        return $this->monitor->generateReport();
    }

    /**
     * Ejecuta una consulta SELECT y retorna todos los resultados
     */
    public function select($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchAll();
    }

    /**
     * Ejecuta una consulta SELECT y retorna un solo resultado
     */
    public function selectOne($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetch();
    }

    /**
     * Ejecuta una consulta INSERT y retorna el ID insertado
     */
    public function insert($sql, $params = []) {
        $this->query($sql, $params);
        return $this->connection->lastInsertId();
    }

    /**
     * Ejecuta una consulta UPDATE o DELETE y retorna el número de filas afectadas
     */
    public function execute($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->rowCount();
    }

    /**
     * Inicia una transacción
     */
    public function beginTransaction() {
        return $this->connection->beginTransaction();
    }

    /**
     * Confirma una transacción
     */
    public function commit() {
        return $this->connection->commit();
    }

    /**
     * Revierte una transacción
     */
    public function rollback() {
        return $this->connection->rollback();
    }

    /**
     * Verifica si hay una transacción activa
     */
    public function inTransaction() {
        return $this->connection->inTransaction();
    }

    /**
     * Escapa un string para uso seguro en SQL
     */
    public function quote($string) {
        return $this->connection->quote($string);
    }

    /**
     * Obtiene información sobre la última consulta
     */
    public function getLastQuery() {
        return $this->connection->lastInsertId();
    }

    /**
     * Cierra la conexión
     */
    public function close() {
        $this->connection = null;
    }

    /**
     * Evitar clonación de la instancia
     */
    private function __clone() {}

    /**
     * Evitar deserialización de la instancia
     */
    public function __wakeup() {
        throw new Exception("No se puede deserializar la instancia de Database");
    }
}