<?php

require_once 'BaseRepository.php';

/**
 * Repositorio para la gestión de movimientos
 */
class MovimientoRepository extends BaseRepository {
    
    public function __construct() {
        parent::__construct();
        $this->table = 'movimientos';
        $this->model = 'Movimiento';
    }

    /**
     * Obtiene movimientos con información relacionada
     */
    public function getConDetalles($limite = 50, $offset = 0) {
        $sql = "SELECT 
                    m.*,
                    u.nombre as nombre_utensilio,
                    u.tipo as tipo_utensilio,
                    us.nombre as nombre_usuario
                FROM {$this->table} m
                LEFT JOIN utensilios u ON m.id_utensilio = u.id
                LEFT JOIN usuarios us ON m.usuario_id = us.id
                ORDER BY m.fecha DESC
                LIMIT ? OFFSET ?";
        
        $data = $this->db->select($sql, [$limite, $offset]);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene movimientos de un utensilio específico
     */
    public function getByUtensilio($idUtensilio, $limite = 20) {
        $sql = "SELECT 
                    m.*,
                    u.nombre as nombre_utensilio,
                    us.nombre as nombre_usuario
                FROM {$this->table} m
                LEFT JOIN utensilios u ON m.id_utensilio = u.id
                LEFT JOIN usuarios us ON m.usuario_id = us.id
                WHERE m.id_utensilio = ?
                ORDER BY m.fecha DESC
                LIMIT ?";
        
        $data = $this->db->select($sql, [$idUtensilio, $limite]);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene movimientos de un usuario específico
     */
    public function getByUsuario($idUsuario, $limite = 20) {
        return $this->findWhere('usuario_id = ?', [$idUsuario], 'fecha', 'DESC');
    }

    /**
     * Obtiene movimientos por tipo
     */
    public function getByTipo($tipoMovimiento, $limite = 50) {
        $sql = "SELECT 
                    m.*,
                    u.nombre as nombre_utensilio,
                    us.nombre as nombre_usuario
                FROM {$this->table} m
                LEFT JOIN utensilios u ON m.id_utensilio = u.id
                LEFT JOIN usuarios us ON m.usuario_id = us.id
                WHERE m.tipo_movimiento = ?
                ORDER BY m.fecha DESC
                LIMIT ?";
        
        $data = $this->db->select($sql, [$tipoMovimiento, $limite]);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene movimientos en un rango de fechas
     */
    public function getByRangoFecha($fechaInicio, $fechaFin) {
        $sql = "SELECT 
                    m.*,
                    u.nombre as nombre_utensilio,
                    u.tipo as tipo_utensilio,
                    us.nombre as nombre_usuario
                FROM {$this->table} m
                LEFT JOIN utensilios u ON m.id_utensilio = u.id
                LEFT JOIN usuarios us ON m.usuario_id = us.id
                WHERE DATE(m.fecha) BETWEEN ? AND ?
                ORDER BY m.fecha DESC";
        
        $data = $this->db->select($sql, [$fechaInicio, $fechaFin]);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene estadísticas de movimientos
     */
    public function getEstadisticas($dias = 30) {
        $sql = "SELECT 
                    COUNT(*) as total_movimientos,
                    SUM(CASE WHEN tipo_movimiento = 'entrada' THEN 1 ELSE 0 END) as entradas,
                    SUM(CASE WHEN tipo_movimiento = 'salida' THEN 1 ELSE 0 END) as salidas,
                    SUM(CASE WHEN tipo_movimiento = 'entrada' THEN cantidad ELSE 0 END) as items_ingresados,
                    SUM(CASE WHEN tipo_movimiento = 'salida' THEN cantidad ELSE 0 END) as items_retirados,
                    COUNT(DISTINCT id_utensilio) as utensilios_afectados,
                    COUNT(DISTINCT usuario_id) as usuarios_activos
                FROM {$this->table} 
                WHERE fecha >= DATE_SUB(NOW(), INTERVAL ? DAY)";
        
        return $this->db->selectOne($sql, [$dias]);
    }

    /**
     * Obtiene motivos más comunes
     */
    public function getMotivosMasComunes($tipo = null, $limite = 10) {
        $where = "1=1";
        $params = [];
        
        if ($tipo) {
            $where .= " AND tipo_movimiento = ?";
            $params[] = $tipo;
        }
        
        $sql = "SELECT 
                    motivo,
                    COUNT(*) as frecuencia,
                    tipo_movimiento
                FROM {$this->table} 
                WHERE {$where}
                GROUP BY motivo, tipo_movimiento
                ORDER BY frecuencia DESC
                LIMIT ?";
        
        $params[] = $limite;
        return $this->db->select($sql, $params);
    }

    /**
     * Obtiene movimientos recientes
     */
    public function getRecientes($limite = 10) {
        return $this->getConDetalles($limite, 0);
    }

    /**
     * Obtiene actividad diaria
     */
    public function getActividadDiaria($dias = 7) {
        $sql = "SELECT 
                    DATE(fecha) as fecha,
                    COUNT(*) as total_movimientos,
                    SUM(CASE WHEN tipo_movimiento = 'entrada' THEN cantidad ELSE 0 END) as entradas,
                    SUM(CASE WHEN tipo_movimiento = 'salida' THEN cantidad ELSE 0 END) as salidas
                FROM {$this->table} 
                WHERE fecha >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY DATE(fecha)
                ORDER BY fecha DESC";
        
        return $this->db->select($sql, [$dias]);
    }

    /**
     * Registra un nuevo movimiento con validaciones
     */
    public function registrarMovimiento($datos) {
        // Validar que el utensilio existe
        $utensilioRepo = new UtensilioRepository();
        $utensilio = $utensilioRepo->getById($datos['id_utensilio']);
        
        if (!$utensilio) {
            throw new Exception('El utensilio especificado no existe');
        }

        // Calcular cantidades
        $datos['cantidad_anterior'] = $utensilio->cantidad_disponible;
        
        if ($datos['tipo_movimiento'] === 'entrada') {
            $datos['cantidad_nueva'] = $utensilio->cantidad_disponible + $datos['cantidad'];
        } else {
            if ($datos['cantidad'] > $utensilio->cantidad_disponible) {
                throw new Exception('No hay suficiente cantidad disponible');
            }
            $datos['cantidad_nueva'] = $utensilio->cantidad_disponible - $datos['cantidad'];
        }

        // Agregar fecha si no está presente
        if (!isset($datos['fecha'])) {
            $datos['fecha'] = date('Y-m-d H:i:s');
        }

        // Iniciar transacción
        $this->beginTransaction();
        
        try {
            // Insertar movimiento
            $movimientoId = $this->insert($datos);
            
            // Actualizar cantidad del utensilio
            $utensilioRepo->actualizarCantidadDisponible(
                $datos['id_utensilio'], 
                $datos['cantidad_nueva']
            );
            
            // Si es entrada, también actualizar cantidad total
            if ($datos['tipo_movimiento'] === 'entrada') {
                $utensilioRepo->update($datos['id_utensilio'], [
                    'cantidad_total' => $utensilio->cantidad_total + $datos['cantidad']
                ]);
            }
            
            $this->commit();
            return $movimientoId;
            
        } catch (Exception $e) {
            $this->rollback();
            throw $e;
        }
    }

    /**
     * Busca movimientos por múltiples criterios
     */
    public function buscar($filtros = []) {
        $where = "1=1";
        $params = [];
        
        if (!empty($filtros['utensilio_id'])) {
            $where .= " AND m.id_utensilio = ?";
            $params[] = $filtros['utensilio_id'];
        }
        
        if (!empty($filtros['tipo_movimiento'])) {
            $where .= " AND m.tipo_movimiento = ?";
            $params[] = $filtros['tipo_movimiento'];
        }
        
        if (!empty($filtros['motivo'])) {
            $where .= " AND m.motivo LIKE ?";
            $params[] = "%{$filtros['motivo']}%";
        }
        
        if (!empty($filtros['usuario_id'])) {
            $where .= " AND m.usuario_id = ?";
            $params[] = $filtros['usuario_id'];
        }
        
        if (!empty($filtros['fecha_desde'])) {
            $where .= " AND DATE(m.fecha) >= ?";
            $params[] = $filtros['fecha_desde'];
        }
        
        if (!empty($filtros['fecha_hasta'])) {
            $where .= " AND DATE(m.fecha) <= ?";
            $params[] = $filtros['fecha_hasta'];
        }
        
        $sql = "SELECT 
                    m.*,
                    u.nombre as nombre_utensilio,
                    u.tipo as tipo_utensilio,
                    us.nombre as nombre_usuario
                FROM {$this->table} m
                LEFT JOIN utensilios u ON m.id_utensilio = u.id
                LEFT JOIN usuarios us ON m.usuario_id = us.id
                WHERE {$where}
                ORDER BY m.fecha DESC";
        
        $data = $this->db->select($sql, $params);
        return $this->mapToModels($data);
    }
}