<?php

require_once 'BaseRepository.php';

/**
 * Repositorio para la gestión de usuarios
 */
class UsuarioRepository extends BaseRepository {
    
    public function __construct() {
        parent::__construct();
        $this->table = 'usuarios';
        $this->model = 'Usuario';
    }

    /**
     * Busca un usuario por email
     */
    public function getByEmail($email) {
        return $this->findOneWhere('email = ?', [$email]);
    }

    /**
     * Verifica si existe un email (para validar duplicados)
     */
    public function emailExists($email, $excludeId = null) {
        $where = 'email = ?';
        $params = [$email];
        
        if ($excludeId) {
            $where .= ' AND id != ?';
            $params[] = $excludeId;
        }
        
        return $this->exists($where, $params);
    }

    /**
     * Obtiene usuarios activos
     */
    public function getActivos() {
        return $this->findWhere('activo = 1', [], 'nombre', 'ASC');
    }

    /**
     * Obtiene usuarios por rol
     */
    public function getByRol($rol) {
        return $this->findWhere('rol = ? AND activo = 1', [$rol], 'nombre', 'ASC');
    }

    /**
     * Actualiza el último acceso del usuario
     */
    public function actualizarUltimoAcceso($id) {
        $sql = "UPDATE {$this->table} SET ultimo_acceso = NOW() WHERE id = ?";
        return $this->db->execute($sql, [$id]);
    }

    /**
     * Cambia la contraseña de un usuario
     */
    public function cambiarPassword($id, $nuevaPasswordHash) {
        return $this->update($id, ['password_hash' => $nuevaPasswordHash]);
    }

    /**
     * Activa o desactiva un usuario
     */
    public function cambiarEstado($id, $activo) {
        return $this->update($id, ['activo' => $activo ? 1 : 0]);
    }

    /**
     * Busca usuarios por nombre o email
     */
    public function buscar($termino, $limite = 10) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE (nombre LIKE ? OR email LIKE ?) AND activo = 1 
                ORDER BY nombre ASC 
                LIMIT ?";
        $params = ["%{$termino}%", "%{$termino}%", $limite];
        $data = $this->db->select($sql, $params);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene estadísticas de usuarios
     */
    public function getEstadisticas() {
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN activo = 1 THEN 1 ELSE 0 END) as activos,
                    SUM(CASE WHEN activo = 0 THEN 1 ELSE 0 END) as inactivos,
                    SUM(CASE WHEN rol = 'administrador' THEN 1 ELSE 0 END) as administradores,
                    SUM(CASE WHEN rol = 'encargado' THEN 1 ELSE 0 END) as encargados,
                    SUM(CASE WHEN rol = 'auditor' THEN 1 ELSE 0 END) as auditores,
                    SUM(CASE WHEN ultimo_acceso >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as activos_mes
                FROM {$this->table}";
        
        return $this->db->selectOne($sql);
    }

    /**
     * Obtiene usuarios con último acceso reciente
     */
    public function getUsuariosRecientes($dias = 7, $limite = 10) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE ultimo_acceso >= DATE_SUB(NOW(), INTERVAL ? DAY) 
                ORDER BY ultimo_acceso DESC 
                LIMIT ?";
        $data = $this->db->select($sql, [$dias, $limite]);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene usuarios inactivos por mucho tiempo
     */
    public function getUsuariosInactivos($dias = 30) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE activo = 1 AND (
                    ultimo_acceso IS NULL OR 
                    ultimo_acceso < DATE_SUB(NOW(), INTERVAL ? DAY)
                )
                ORDER BY ultimo_acceso ASC";
        $data = $this->db->select($sql, [$dias]);
        return $this->mapToModels($data);
    }

    /**
     * Crea un nuevo usuario con validaciones
     */
    public function crear($datos) {
        // Verificar que no existe el email
        if ($this->emailExists($datos['email'])) {
            throw new Exception('El email ya está registrado en el sistema');
        }

        // Hashear la contraseña
        $datos['password_hash'] = password_hash($datos['password'], PASSWORD_HASH_ALGO);
        unset($datos['password']); // No guardar la contraseña en texto plano

        // Campos automáticos
        $datos['fecha_creacion'] = date('Y-m-d H:i:s');
        $datos['activo'] = 1;

        return $this->insert($datos);
    }

    /**
     * Actualiza un usuario con validaciones
     */
    public function actualizar($id, $datos) {
        // Verificar que no existe el email en otro usuario
        if (isset($datos['email']) && $this->emailExists($datos['email'], $id)) {
            throw new Exception('El email ya está registrado en otro usuario');
        }

        // Si se proporciona nueva contraseña, hashearla
        if (isset($datos['password']) && !empty($datos['password'])) {
            $datos['password_hash'] = password_hash($datos['password'], PASSWORD_HASH_ALGO);
            unset($datos['password']);
        }

        return $this->update($id, $datos);
    }

    /**
     * Verifica las credenciales de login
     */
    public function verificarCredenciales($email, $password) {
        $usuario = $this->getByEmail($email);
        
        if (!$usuario || !$usuario->activo) {
            return false;
        }

        if (password_verify($password, $usuario->password_hash)) {
            // Actualizar último acceso
            $this->actualizarUltimoAcceso($usuario->id);
            return $usuario;
        }

        return false;
    }
}