<?php

require_once 'BaseRepository.php';

/**
 * Repositorio para la gestión de utensilios
 */
class UtensilioRepository extends BaseRepository {
    
    public function __construct() {
        parent::__construct();
        $this->table = 'utensilios';
        $this->model = 'Utensilio';
    }

    /**
     * Obtiene utensilios activos
     */
    public function getActivos() {
        return $this->findWhere('activo = 1', [], 'nombre', 'ASC');
    }

    /**
     * Obtiene utensilios por tipo
     */
    public function getByTipo($tipo) {
        return $this->findWhere('tipo = ? AND activo = 1', [$tipo], 'nombre', 'ASC');
    }

    /**
     * Obtiene utensilios por ubicación
     */
    public function getByUbicacion($ubicacion) {
        return $this->findWhere('ubicacion = ? AND activo = 1', [$ubicacion], 'nombre', 'ASC');
    }

    /**
     * Obtiene utensilios con stock bajo
     */
    public function getConStockBajo() {
        return $this->findWhere('cantidad_disponible <= cantidad_minima AND activo = 1', [], 'cantidad_disponible', 'ASC');
    }

    /**
     * Obtiene utensilios por estado
     */
    public function getByEstado($estado) {
        return $this->findWhere('estado = ? AND activo = 1', [$estado], 'nombre', 'ASC');
    }

    /**
     * Busca utensilios por nombre o descripción
     */
    public function buscar($termino, $limite = 20) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE (nombre LIKE ? OR descripcion LIKE ?) AND activo = 1 
                ORDER BY nombre ASC 
                LIMIT ?";
        $params = ["%{$termino}%", "%{$termino}%", $limite];
        $data = $this->db->select($sql, $params);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene estadísticas generales de utensilios
     */
    public function getEstadisticas() {
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN activo = 1 THEN 1 ELSE 0 END) as activos,
                    SUM(cantidad_total) as total_items,
                    SUM(cantidad_disponible) as items_disponibles,
                    SUM(CASE WHEN cantidad_disponible <= cantidad_minima THEN 1 ELSE 0 END) as con_stock_bajo,
                    SUM(CASE WHEN estado = 'excelente' THEN 1 ELSE 0 END) as excelente,
                    SUM(CASE WHEN estado = 'bueno' THEN 1 ELSE 0 END) as bueno,
                    SUM(CASE WHEN estado = 'regular' THEN 1 ELSE 0 END) as regular,
                    SUM(CASE WHEN estado = 'malo' THEN 1 ELSE 0 END) as malo,
                    SUM(CASE WHEN estado = 'en_revision' THEN 1 ELSE 0 END) as en_revision,
                    SUM(CASE WHEN estado = 'fuera_servicio' THEN 1 ELSE 0 END) as fuera_servicio
                FROM {$this->table} 
                WHERE activo = 1";
        
        return $this->db->selectOne($sql);
    }

    /**
     * Obtiene estadísticas por tipo
     */
    public function getEstadisticasPorTipo() {
        $sql = "SELECT 
                    tipo,
                    COUNT(*) as cantidad,
                    SUM(cantidad_total) as total_items,
                    SUM(cantidad_disponible) as items_disponibles,
                    AVG(precio_unitario) as precio_promedio
                FROM {$this->table} 
                WHERE activo = 1 
                GROUP BY tipo 
                ORDER BY cantidad DESC";
        
        return $this->db->select($sql);
    }

    /**
     * Obtiene utensilios más utilizados (basado en movimientos)
     */
    public function getMasUtilizados($limite = 10, $dias = 30) {
        $sql = "SELECT 
                    u.*,
                    COUNT(m.id) as total_movimientos,
                    SUM(CASE WHEN m.tipo_movimiento = 'salida' THEN m.cantidad ELSE 0 END) as total_salidas
                FROM {$this->table} u
                LEFT JOIN movimientos m ON u.id = m.id_utensilio 
                    AND m.fecha >= DATE_SUB(NOW(), INTERVAL ? DAY)
                WHERE u.activo = 1
                GROUP BY u.id
                ORDER BY total_salidas DESC
                LIMIT ?";
        
        $data = $this->db->select($sql, [$dias, $limite]);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene todas las ubicaciones disponibles
     */
    public function getUbicaciones() {
        $sql = "SELECT DISTINCT ubicacion FROM {$this->table} WHERE activo = 1 ORDER BY ubicacion";
        $data = $this->db->select($sql);
        return array_column($data, 'ubicacion');
    }

    /**
     * Actualiza la cantidad disponible de un utensilio
     */
    public function actualizarCantidadDisponible($id, $nuevaCantidad) {
        return $this->update($id, ['cantidad_disponible' => $nuevaCantidad]);
    }

    /**
     * Incrementa la cantidad total y disponible
     */
    public function incrementarCantidad($id, $cantidad) {
        $sql = "UPDATE {$this->table} 
                SET cantidad_total = cantidad_total + ?, 
                    cantidad_disponible = cantidad_disponible + ? 
                WHERE id = ?";
        return $this->db->execute($sql, [$cantidad, $cantidad, $id]);
    }

    /**
     * Decrementa la cantidad disponible
     */
    public function decrementarCantidad($id, $cantidad) {
        $sql = "UPDATE {$this->table} 
                SET cantidad_disponible = cantidad_disponible - ? 
                WHERE id = ? AND cantidad_disponible >= ?";
        $result = $this->db->execute($sql, [$cantidad, $id, $cantidad]);
        
        if ($result === 0) {
            throw new Exception('No hay suficiente cantidad disponible para realizar la operación');
        }
        
        return $result;
    }

    /**
     * Obtiene utensilios que requieren atención
     */
    public function getQueRequierenAtencion() {
        $sql = "SELECT * FROM {$this->table} 
                WHERE activo = 1 AND (
                    cantidad_disponible <= cantidad_minima OR
                    estado IN ('malo', 'en_revision', 'fuera_servicio')
                )
                ORDER BY 
                    CASE 
                        WHEN estado = 'fuera_servicio' THEN 1
                        WHEN estado = 'malo' THEN 2
                        WHEN estado = 'en_revision' THEN 3
                        WHEN cantidad_disponible <= cantidad_minima THEN 4
                        ELSE 5
                    END";
        
        $data = $this->db->select($sql);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene utensilios agregados recientemente
     */
    public function getRecientes($dias = 7, $limite = 10) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE fecha_registro >= DATE_SUB(NOW(), INTERVAL ? DAY) AND activo = 1
                ORDER BY fecha_registro DESC 
                LIMIT ?";
        $data = $this->db->select($sql, [$dias, $limite]);
        return $this->mapToModels($data);
    }

    /**
     * Obtiene valor total del inventario
     */
    public function getValorTotal() {
        $sql = "SELECT SUM(cantidad_total * precio_unitario) as valor_total FROM {$this->table} WHERE activo = 1";
        $result = $this->db->selectOne($sql);
        return $result['valor_total'] ?? 0;
    }

    /**
     * Crea un nuevo utensilio con validaciones
     */
    public function crear($datos) {
        // Validaciones específicas
        if ($datos['cantidad_disponible'] > $datos['cantidad_total']) {
            throw new Exception('La cantidad disponible no puede ser mayor a la cantidad total');
        }

        // Campos automáticos
        $datos['fecha_registro'] = date('Y-m-d H:i:s');
        $datos['activo'] = 1;

        return $this->insert($datos);
    }

    /**
     * Actualiza un utensilio con validaciones
     */
    public function actualizar($id, $datos) {
        // Validaciones específicas
        if (isset($datos['cantidad_disponible']) && isset($datos['cantidad_total'])) {
            if ($datos['cantidad_disponible'] > $datos['cantidad_total']) {
                throw new Exception('La cantidad disponible no puede ser mayor a la cantidad total');
            }
        }

        return $this->update($id, $datos);
    }

    /**
     * Elimina un utensilio (lo marca como inactivo)
     */
    public function eliminar($id) {
        return $this->update($id, ['activo' => 0]);
    }
}