<?php

/**
 * Modelo Movimiento
 * Representa un movimiento de inventario (entrada o salida)
 */
class Movimiento {
    public $id;
    public $id_utensilio;
    public $tipo_movimiento;
    public $cantidad;
    public $cantidad_anterior;
    public $cantidad_nueva;
    public $motivo;
    public $detalle_motivo;
    public $usuario_id;
    public $fecha;
    public $observaciones;
    public $referencia_externa;
    
    // Propiedades calculadas (joins)
    public $nombre_utensilio;
    public $nombre_usuario;
    public $tipo_utensilio;

    // Constantes para tipos de movimiento
    const TIPO_ENTRADA = 'entrada';
    const TIPO_SALIDA = 'salida';

    // Constantes para motivos comunes
    const MOTIVO_COMPRA = 'Compra';
    const MOTIVO_DONACION = 'Donación';
    const MOTIVO_INVENTARIO_INICIAL = 'Inventario inicial';
    const MOTIVO_USO = 'Uso';
    const MOTIVO_PRESTAMO = 'Préstamo';
    const MOTIVO_DANO = 'Daño';
    const MOTIVO_PERDIDA = 'Pérdida';
    const MOTIVO_ROBO = 'Robo';
    const MOTIVO_LIMPIEZA = 'Limpieza';
    const MOTIVO_REPARACION = 'Reparación';
    const MOTIVO_MANTENIMIENTO = 'Mantenimiento';
    const MOTIVO_REVISION = 'Revisión';

    public function __construct($data = []) {
        if (!empty($data)) {
            $this->hydrate($data);
        }
    }

    /**
     * Llena el objeto con datos del array
     */
    public function hydrate($data) {
        foreach ($data as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    /**
     * Convierte el objeto a array
     */
    public function toArray() {
        return [
            'id' => $this->id,
            'id_utensilio' => $this->id_utensilio,
            'tipo_movimiento' => $this->tipo_movimiento,
            'cantidad' => $this->cantidad,
            'cantidad_anterior' => $this->cantidad_anterior,
            'cantidad_nueva' => $this->cantidad_nueva,
            'motivo' => $this->motivo,
            'detalle_motivo' => $this->detalle_motivo,
            'usuario_id' => $this->usuario_id,
            'fecha' => $this->fecha,
            'observaciones' => $this->observaciones,
            'referencia_externa' => $this->referencia_externa
        ];
    }

    /**
     * Obtiene todos los tipos de movimiento
     */
    public static function getTiposMovimiento() {
        return [
            self::TIPO_ENTRADA => 'Entrada',
            self::TIPO_SALIDA => 'Salida'
        ];
    }

    /**
     * Obtiene motivos comunes para entradas
     */
    public static function getMotivosEntrada() {
        return [
            self::MOTIVO_COMPRA,
            self::MOTIVO_DONACION,
            self::MOTIVO_INVENTARIO_INICIAL,
            'Devolución',
            'Reparación completada',
            'Otro'
        ];
    }

    /**
     * Obtiene motivos comunes para salidas
     */
    public static function getMotivosSalida() {
        return [
            self::MOTIVO_USO,
            self::MOTIVO_PRESTAMO,
            self::MOTIVO_DANO,
            self::MOTIVO_PERDIDA,
            self::MOTIVO_ROBO,
            self::MOTIVO_LIMPIEZA,
            self::MOTIVO_REPARACION,
            self::MOTIVO_MANTENIMIENTO,
            self::MOTIVO_REVISION,
            'Otro'
        ];
    }

    /**
     * Verifica si es un movimiento de entrada
     */
    public function esEntrada() {
        return $this->tipo_movimiento === self::TIPO_ENTRADA;
    }

    /**
     * Verifica si es un movimiento de salida
     */
    public function esSalida() {
        return $this->tipo_movimiento === self::TIPO_SALIDA;
    }

    /**
     * Obtiene la clase CSS para el tipo de movimiento
     */
    public function getClaseTipo() {
        return $this->esEntrada() ? 'success' : 'danger';
    }

    /**
     * Obtiene el icono para el tipo de movimiento
     */
    public function getIconoTipo() {
        return $this->esEntrada() ? 'fas fa-arrow-up' : 'fas fa-arrow-down';
    }

    /**
     * Formatea la fecha del movimiento
     */
    public function getFechaFormateada() {
        if (empty($this->fecha)) return '';
        return date('d/m/Y H:i', strtotime($this->fecha));
    }

    /**
     * Calcula el impacto del movimiento
     */
    public function getImpacto() {
        if ($this->esEntrada()) {
            return "+{$this->cantidad}";
        } else {
            return "-{$this->cantidad}";
        }
    }

    /**
     * Valida los datos del movimiento
     */
    public function validar() {
        $errores = [];

        if (empty($this->id_utensilio) || !is_numeric($this->id_utensilio)) {
            $errores[] = 'El utensilio es requerido';
        }

        if (empty($this->tipo_movimiento) || !in_array($this->tipo_movimiento, array_keys(self::getTiposMovimiento()))) {
            $errores[] = 'El tipo de movimiento debe ser válido';
        }

        if (!is_numeric($this->cantidad) || $this->cantidad <= 0) {
            $errores[] = 'La cantidad debe ser un número mayor a 0';
        }

        if (empty($this->motivo)) {
            $errores[] = 'El motivo es requerido';
        }

        if (empty($this->usuario_id) || !is_numeric($this->usuario_id)) {
            $errores[] = 'El usuario es requerido';
        }

        return $errores;
    }

    /**
     * Valida si el movimiento es válido para un utensilio específico
     */
    public function validarConUtensilio($utensilio) {
        $errores = [];

        if ($this->esSalida() && $this->cantidad > $utensilio->cantidad_disponible) {
            $errores[] = 'No hay suficiente cantidad disponible para la salida solicitada';
        }

        if ($this->cantidad_nueva < 0) {
            $errores[] = 'La cantidad resultante no puede ser negativa';
        }

        return $errores;
    }
}