<?php

/**
 * Modelo Usuario
 * Representa un usuario del sistema
 */
class Usuario {
    public $id;
    public $nombre;
    public $email;
    public $password_hash;
    public $rol;
    public $activo;
    public $ultimo_acceso;
    public $fecha_creacion;
    public $fecha_actualizacion;

    // Constantes para roles
    const ROL_ADMINISTRADOR = 'administrador';
    const ROL_ENCARGADO = 'encargado';
    const ROL_AUDITOR = 'auditor';

    public function __construct($data = []) {
        if (!empty($data)) {
            $this->hydrate($data);
        }
    }

    /**
     * Llena el objeto con datos del array
     */
    public function hydrate($data) {
        foreach ($data as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    /**
     * Convierte el objeto a array
     */
    public function toArray() {
        return [
            'id' => $this->id,
            'nombre' => $this->nombre,
            'email' => $this->email,
            'rol' => $this->rol,
            'activo' => $this->activo,
            'ultimo_acceso' => $this->ultimo_acceso,
            'fecha_creacion' => $this->fecha_creacion,
            'fecha_actualizacion' => $this->fecha_actualizacion
        ];
    }

    /**
     * Obtiene todos los roles disponibles
     */
    public static function getRoles() {
        return [
            self::ROL_ADMINISTRADOR => 'Administrador',
            self::ROL_ENCARGADO => 'Encargado de Cocina',
            self::ROL_AUDITOR => 'Auditor'
        ];
    }

    /**
     * Verifica si el usuario tiene un rol específico
     */
    public function tieneRol($rol) {
        return $this->rol === $rol;
    }

    /**
     * Verifica si el usuario es administrador
     */
    public function esAdministrador() {
        return $this->tieneRol(self::ROL_ADMINISTRADOR);
    }

    /**
     * Verifica si el usuario es encargado
     */
    public function esEncargado() {
        return $this->tieneRol(self::ROL_ENCARGADO);
    }

    /**
     * Verifica si el usuario es auditor
     */
    public function esAuditor() {
        return $this->tieneRol(self::ROL_AUDITOR);
    }

    /**
     * Valida los datos del usuario
     */
    public function validar() {
        $errores = [];

        if (empty($this->nombre)) {
            $errores[] = 'El nombre es requerido';
        }

        if (empty($this->email)) {
            $errores[] = 'El email es requerido';
        } elseif (!filter_var($this->email, FILTER_VALIDATE_EMAIL)) {
            $errores[] = 'El email no tiene un formato válido';
        }

        if (empty($this->rol) || !in_array($this->rol, array_keys(self::getRoles()))) {
            $errores[] = 'El rol debe ser válido';
        }

        return $errores;
    }
}