<?php

/**
 * Modelo Utensilio
 * Representa un utensilio de cocina
 */
class Utensilio {
    public $id;
    public $nombre;
    public $tipo;
    public $descripcion;
    public $cantidad_total;
    public $cantidad_disponible;
    public $cantidad_minima;
    public $estado;
    public $ubicacion;
    public $precio_unitario;
    public $fecha_registro;
    public $fecha_actualizacion;
    public $observaciones;
    public $activo;

    // Constantes para tipos
    const TIPO_CUCHILLO = 'cuchillo';
    const TIPO_SARTEN = 'sarten';
    const TIPO_OLLA = 'olla';
    const TIPO_UTENSILIO_MENOR = 'utensilio_menor';
    const TIPO_ELECTRODOMESTICO = 'electrodomestico';
    const TIPO_OTRO = 'otro';

    // Constantes para estados
    const ESTADO_EXCELENTE = 'excelente';
    const ESTADO_BUENO = 'bueno';
    const ESTADO_REGULAR = 'regular';
    const ESTADO_MALO = 'malo';
    const ESTADO_EN_REVISION = 'en_revision';
    const ESTADO_FUERA_SERVICIO = 'fuera_servicio';

    public function __construct($data = []) {
        if (!empty($data)) {
            $this->hydrate($data);
        }
    }

    /**
     * Llena el objeto con datos del array
     */
    public function hydrate($data) {
        foreach ($data as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    /**
     * Convierte el objeto a array
     */
    public function toArray() {
        return [
            'id' => $this->id,
            'nombre' => $this->nombre,
            'tipo' => $this->tipo,
            'descripcion' => $this->descripcion,
            'cantidad_total' => $this->cantidad_total,
            'cantidad_disponible' => $this->cantidad_disponible,
            'cantidad_minima' => $this->cantidad_minima,
            'estado' => $this->estado,
            'ubicacion' => $this->ubicacion,
            'precio_unitario' => $this->precio_unitario,
            'fecha_registro' => $this->fecha_registro,
            'fecha_actualizacion' => $this->fecha_actualizacion,
            'observaciones' => $this->observaciones,
            'activo' => $this->activo
        ];
    }

    /**
     * Obtiene todos los tipos disponibles
     */
    public static function getTipos() {
        return [
            self::TIPO_CUCHILLO => 'Cuchillo',
            self::TIPO_SARTEN => 'Sartén',
            self::TIPO_OLLA => 'Olla',
            self::TIPO_UTENSILIO_MENOR => 'Utensilio Menor',
            self::TIPO_ELECTRODOMESTICO => 'Electrodoméstico',
            self::TIPO_OTRO => 'Otro'
        ];
    }

    /**
     * Obtiene todos los estados disponibles
     */
    public static function getEstados() {
        return [
            self::ESTADO_EXCELENTE => 'Excelente',
            self::ESTADO_BUENO => 'Bueno',
            self::ESTADO_REGULAR => 'Regular',
            self::ESTADO_MALO => 'Malo',
            self::ESTADO_EN_REVISION => 'En Revisión',
            self::ESTADO_FUERA_SERVICIO => 'Fuera de Servicio'
        ];
    }

    /**
     * Verifica si el utensilio tiene stock bajo
     */
    public function tieneStockBajo() {
        return $this->cantidad_disponible <= $this->cantidad_minima;
    }

    /**
     * Verifica si el utensilio está disponible para uso
     */
    public function estaDisponible() {
        return $this->activo && 
               $this->cantidad_disponible > 0 && 
               !in_array($this->estado, [self::ESTADO_FUERA_SERVICIO, self::ESTADO_EN_REVISION]);
    }

    /**
     * Obtiene la clase CSS para el estado
     */
    public function getClaseEstado() {
        $clases = [
            self::ESTADO_EXCELENTE => 'success',
            self::ESTADO_BUENO => 'info',
            self::ESTADO_REGULAR => 'warning',
            self::ESTADO_MALO => 'danger',
            self::ESTADO_EN_REVISION => 'warning',
            self::ESTADO_FUERA_SERVICIO => 'danger'
        ];
        
        return $clases[$this->estado] ?? 'secondary';
    }

    /**
     * Calcula el porcentaje de disponibilidad
     */
    public function getPorcentajeDisponibilidad() {
        if ($this->cantidad_total == 0) return 0;
        return round(($this->cantidad_disponible / $this->cantidad_total) * 100, 2);
    }

    /**
     * Valida los datos del utensilio
     */
    public function validar() {
        $errores = [];

        if (empty($this->nombre)) {
            $errores[] = 'El nombre es requerido';
        }

        if (empty($this->tipo) || !in_array($this->tipo, array_keys(self::getTipos()))) {
            $errores[] = 'El tipo debe ser válido';
        }

        if (!is_numeric($this->cantidad_total) || $this->cantidad_total < 0) {
            $errores[] = 'La cantidad total debe ser un número válido';
        }

        if (!is_numeric($this->cantidad_disponible) || $this->cantidad_disponible < 0) {
            $errores[] = 'La cantidad disponible debe ser un número válido';
        }

        if ($this->cantidad_disponible > $this->cantidad_total) {
            $errores[] = 'La cantidad disponible no puede ser mayor a la cantidad total';
        }

        if (!is_numeric($this->cantidad_minima) || $this->cantidad_minima < 0) {
            $errores[] = 'La cantidad mínima debe ser un número válido';
        }

        if (empty($this->estado) || !in_array($this->estado, array_keys(self::getEstados()))) {
            $errores[] = 'El estado debe ser válido';
        }

        if (empty($this->ubicacion)) {
            $errores[] = 'La ubicación es requerida';
        }

        if (!is_numeric($this->precio_unitario) || $this->precio_unitario < 0) {
            $errores[] = 'El precio unitario debe ser un número válido';
        }

        return $errores;
    }
}