<?php

/**
 * Sistema de caché simple para mejorar el rendimiento
 * Implementa caché en memoria y archivo
 */
class CacheManager {
    
    private static $instance = null;
    private $cache = [];
    private $cacheDir;
    private $defaultTtl = 300; // 5 minutos por defecto
    
    private function __construct() {
        $this->cacheDir = dirname(__DIR__) . '/cache/';
        $this->ensureCacheDirectory();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Obtiene un valor del caché
     */
    public function get($key) {
        // Primero verificar caché en memoria
        if (isset($this->cache[$key])) {
            $item = $this->cache[$key];
            if ($item['expires'] > time()) {
                return $item['data'];
            }
            unset($this->cache[$key]);
        }
        
        // Verificar caché en archivo
        $filePath = $this->getCacheFilePath($key);
        if (file_exists($filePath)) {
            $data = unserialize(file_get_contents($filePath));
            if ($data['expires'] > time()) {
                // Cargar de vuelta en memoria
                $this->cache[$key] = $data;
                return $data['data'];
            }
            // Cache expirado, eliminar archivo
            unlink($filePath);
        }
        
        return null;
    }
    
    /**
     * Guarda un valor en el caché
     */
    public function set($key, $data, $ttl = null) {
        $ttl = $ttl ?? $this->defaultTtl;
        $expires = time() + $ttl;
        
        $cacheItem = [
            'data' => $data,
            'expires' => $expires,
            'created' => time()
        ];
        
        // Guardar en memoria
        $this->cache[$key] = $cacheItem;
        
        // Guardar en archivo para persistencia
        $filePath = $this->getCacheFilePath($key);
        file_put_contents($filePath, serialize($cacheItem), LOCK_EX);
        
        return true;
    }
    
    /**
     * Elimina un valor específico del caché
     */
    public function delete($key) {
        // Eliminar de memoria
        unset($this->cache[$key]);
        
        // Eliminar archivo
        $filePath = $this->getCacheFilePath($key);
        if (file_exists($filePath)) {
            unlink($filePath);
        }
        
        return true;
    }
    
    /**
     * Limpia todo el caché
     */
    public function clear() {
        // Limpiar memoria
        $this->cache = [];
        
        // Limpiar archivos
        $files = glob($this->cacheDir . 'cache_*.ser');
        foreach ($files as $file) {
            unlink($file);
        }
        
        return true;
    }
    
    /**
     * Obtiene estadísticas del caché
     */
    public function getStats() {
        $memoryItems = count($this->cache);
        $fileItems = count(glob($this->cacheDir . 'cache_*.ser'));
        $cacheSize = $this->getCacheSize();
        
        return [
            'memory_items' => $memoryItems,
            'file_items' => $fileItems,
            'total_size_mb' => round($cacheSize / 1024 / 1024, 2),
            'cache_dir' => $this->cacheDir
        ];
    }
    
    /**
     * Función helper para cachear resultados de funciones
     */
    public function remember($key, $callback, $ttl = null) {
        $value = $this->get($key);
        
        if ($value === null) {
            $value = $callback();
            $this->set($key, $value, $ttl);
        }
        
        return $value;
    }
    
    /**
     * Limpia cache expirado
     */
    public function cleanup() {
        $cleaned = 0;
        
        // Limpiar memoria
        foreach ($this->cache as $key => $item) {
            if ($item['expires'] <= time()) {
                unset($this->cache[$key]);
                $cleaned++;
            }
        }
        
        // Limpiar archivos
        $files = glob($this->cacheDir . 'cache_*.ser');
        foreach ($files as $file) {
            $data = unserialize(file_get_contents($file));
            if ($data['expires'] <= time()) {
                unlink($file);
                $cleaned++;
            }
        }
        
        return $cleaned;
    }
    
    /**
     * Genera una clave de cache válida
     */
    public function generateKey($prefix, $params = []) {
        $key = $prefix;
        if (!empty($params)) {
            $key .= '_' . md5(serialize($params));
        }
        return $key;
    }
    
    /**
     * Invalida cache por patrón
     */
    public function invalidatePattern($pattern) {
        $deleted = 0;
        
        // Invalidar en memoria
        foreach (array_keys($this->cache) as $key) {
            if (fnmatch($pattern, $key)) {
                unset($this->cache[$key]);
                $deleted++;
            }
        }
        
        // Invalidar archivos
        $files = glob($this->cacheDir . 'cache_*.ser');
        foreach ($files as $file) {
            $keyFromFile = $this->getKeyFromFilePath($file);
            if (fnmatch($pattern, $keyFromFile)) {
                unlink($file);
                $deleted++;
            }
        }
        
        return $deleted;
    }
    
    /**
     * Asegura que el directorio de cache existe
     */
    private function ensureCacheDirectory() {
        if (!is_dir($this->cacheDir)) {
            mkdir($this->cacheDir, 0777, true);
        }
    }
    
    /**
     * Obtiene la ruta del archivo de cache
     */
    private function getCacheFilePath($key) {
        $safeKey = preg_replace('/[^a-zA-Z0-9_-]/', '_', $key);
        return $this->cacheDir . 'cache_' . $safeKey . '.ser';
    }
    
    /**
     * Obtiene la clave desde la ruta del archivo
     */
    private function getKeyFromFilePath($filePath) {
        $filename = basename($filePath);
        return str_replace(['cache_', '.ser'], '', $filename);
    }
    
    /**
     * Calcula el tamaño total del cache
     */
    private function getCacheSize() {
        $size = 0;
        $files = glob($this->cacheDir . 'cache_*.ser');
        foreach ($files as $file) {
            $size += filesize($file);
        }
        return $size;
    }
    
    /**
     * Configurar TTL por defecto
     */
    public function setDefaultTtl($ttl) {
        $this->defaultTtl = $ttl;
    }
    
    /**
     * Verificar si una clave existe en cache
     */
    public function has($key) {
        return $this->get($key) !== null;
    }
}
?>