<?php

/**
 * Monitor y optimizador de rendimiento del sistema
 * Analiza consultas SQL, uso de caché y rendimiento general
 */
class PerformanceMonitor {
    
    private static $instance = null;
    private $queryLog = [];
    private $startTime;
    private $memoryStart;
    private $slowQueryThreshold = 1.0; // 1 segundo
    
    private function __construct() {
        $this->startTime = microtime(true);
        $this->memoryStart = memory_get_usage();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Registra una consulta SQL para análisis
     */
    public function logQuery($sql, $params = [], $executionTime = null, $rowCount = null) {
        $this->queryLog[] = [
            'sql' => $sql,
            'params' => $params,
            'execution_time' => $executionTime,
            'row_count' => $rowCount,
            'timestamp' => microtime(true),
            'memory_usage' => memory_get_usage(),
            'is_slow' => $executionTime > $this->slowQueryThreshold
        ];
    }
    
    /**
     * Obtiene estadísticas de rendimiento
     */
    public function getPerformanceStats() {
        $totalTime = microtime(true) - $this->startTime;
        $memoryUsed = memory_get_usage() - $this->memoryStart;
        $peakMemory = memory_get_peak_usage();
        
        $totalQueries = count($this->queryLog);
        $slowQueries = array_filter($this->queryLog, function($query) {
            return $query['is_slow'];
        });
        
        $totalQueryTime = array_sum(array_column($this->queryLog, 'execution_time'));
        $avgQueryTime = $totalQueries > 0 ? $totalQueryTime / $totalQueries : 0;
        
        return [
            'execution' => [
                'total_time' => round($totalTime, 3),
                'memory_used_mb' => round($memoryUsed / 1024 / 1024, 2),
                'peak_memory_mb' => round($peakMemory / 1024 / 1024, 2)
            ],
            'queries' => [
                'total_count' => $totalQueries,
                'slow_count' => count($slowQueries),
                'total_time' => round($totalQueryTime, 3),
                'average_time' => round($avgQueryTime, 3),
                'slowest_query' => $this->getSlowestQuery()
            ],
            'cache' => CacheManager::getInstance()->getStats()
        ];
    }
    
    /**
     * Obtiene la consulta más lenta
     */
    private function getSlowestQuery() {
        if (empty($this->queryLog)) {
            return null;
        }
        
        $slowest = array_reduce($this->queryLog, function($carry, $query) {
            return ($query['execution_time'] > ($carry['execution_time'] ?? 0)) ? $query : $carry;
        }, []);
        
        return [
            'sql' => substr($slowest['sql'], 0, 100) . '...',
            'execution_time' => $slowest['execution_time'],
            'row_count' => $slowest['row_count']
        ];
    }
    
    /**
     * Obtiene consultas lentas
     */
    public function getSlowQueries() {
        return array_filter($this->queryLog, function($query) {
            return $query['is_slow'];
        });
    }
    
    /**
     * Obtiene recomendaciones de optimización
     */
    public function getOptimizationRecommendations() {
        $recommendations = [];
        $stats = $this->getPerformanceStats();
        
        // Análisis de memoria
        if ($stats['execution']['memory_used_mb'] > 50) {
            $recommendations[] = [
                'type' => 'memory',
                'severity' => 'high',
                'message' => 'Alto uso de memoria detectado. Considere implementar paginación.',
                'value' => $stats['execution']['memory_used_mb'] . ' MB'
            ];
        }
        
        // Análisis de consultas
        if ($stats['queries']['slow_count'] > 0) {
            $recommendations[] = [
                'type' => 'queries',
                'severity' => 'medium',
                'message' => 'Se detectaron consultas lentas. Revise los índices de base de datos.',
                'value' => $stats['queries']['slow_count'] . ' consultas lentas'
            ];
        }
        
        // Análisis de tiempo total
        if ($stats['execution']['total_time'] > 5) {
            $recommendations[] = [
                'type' => 'performance',
                'severity' => 'medium',
                'message' => 'Tiempo de ejecución elevado. Considere implementar más caché.',
                'value' => $stats['execution']['total_time'] . ' segundos'
            ];
        }
        
        // Análisis de caché
        if ($stats['cache']['memory_items'] < 5) {
            $recommendations[] = [
                'type' => 'cache',
                'severity' => 'low',
                'message' => 'Bajo uso de caché. Implemente caché en más operaciones.',
                'value' => $stats['cache']['memory_items'] . ' items en caché'
            ];
        }
        
        return $recommendations;
    }
    
    /**
     * Genera un reporte de rendimiento
     */
    public function generateReport() {
        $stats = $this->getPerformanceStats();
        $recommendations = $this->getOptimizationRecommendations();
        $slowQueries = $this->getSlowQueries();
        
        return [
            'timestamp' => date('Y-m-d H:i:s'),
            'statistics' => $stats,
            'recommendations' => $recommendations,
            'slow_queries' => array_slice($slowQueries, 0, 5), // Top 5 consultas lentas
            'summary' => $this->generateSummary($stats, $recommendations)
        ];
    }
    
    /**
     * Genera un resumen del rendimiento
     */
    private function generateSummary($stats, $recommendations) {
        $criticalIssues = count(array_filter($recommendations, function($r) {
            return $r['severity'] === 'high';
        }));
        
        $status = 'excellent';
        if ($criticalIssues > 0) {
            $status = 'critical';
        } elseif ($stats['queries']['slow_count'] > 3) {
            $status = 'needs_attention';
        } elseif ($stats['execution']['total_time'] > 2) {
            $status = 'good';
        }
        
        return [
            'status' => $status,
            'score' => $this->calculatePerformanceScore($stats),
            'critical_issues' => $criticalIssues,
            'total_recommendations' => count($recommendations)
        ];
    }
    
    /**
     * Calcula un puntaje de rendimiento (0-100)
     */
    private function calculatePerformanceScore($stats) {
        $score = 100;
        
        // Penalizar por tiempo de ejecución
        if ($stats['execution']['total_time'] > 1) {
            $score -= min(30, $stats['execution']['total_time'] * 10);
        }
        
        // Penalizar por uso de memoria
        if ($stats['execution']['memory_used_mb'] > 20) {
            $score -= min(25, $stats['execution']['memory_used_mb']);
        }
        
        // Penalizar por consultas lentas
        if ($stats['queries']['slow_count'] > 0) {
            $score -= min(20, $stats['queries']['slow_count'] * 5);
        }
        
        // Bonificar por uso de caché
        if ($stats['cache']['memory_items'] > 10) {
            $score += min(10, $stats['cache']['memory_items']);
        }
        
        return max(0, min(100, round($score)));
    }
    
    /**
     * Limpia los logs de consultas
     */
    public function clearLogs() {
        $this->queryLog = [];
    }
    
    /**
     * Configura el umbral para consultas lentas
     */
    public function setSlowQueryThreshold($threshold) {
        $this->slowQueryThreshold = $threshold;
    }
    
    /**
     * Inicia el monitoreo de una operación específica
     */
    public function startOperation($name) {
        return [
            'name' => $name,
            'start_time' => microtime(true),
            'start_memory' => memory_get_usage()
        ];
    }
    
    /**
     * Finaliza el monitoreo de una operación
     */
    public function endOperation($operation) {
        $endTime = microtime(true);
        $endMemory = memory_get_usage();
        
        return [
            'name' => $operation['name'],
            'execution_time' => round($endTime - $operation['start_time'], 3),
            'memory_used' => round(($endMemory - $operation['start_memory']) / 1024 / 1024, 2),
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
}
?>