<?php

/**
 * Sistema de seguridad avanzado
 * Maneja validación, sanitización, CSRF, rate limiting y logging de seguridad
 */
class SecurityManager {
    
    private static $instance = null;
    private $securityLog = [];
    private $rateLimits = [];
    private $blockedIPs = [];
    
    // Configuración de seguridad
    private $maxLoginAttempts = 5;
    private $lockoutDuration = 900; // 15 minutos
    private $rateLimitWindow = 60; // 1 minuto
    private $maxRequestsPerMinute = 100;
    
    private function __construct() {
        $this->initializeSecuritySettings();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Inicializa configuraciones de seguridad
     */
    private function initializeSecuritySettings() {
        // Configurar headers de seguridad
        $this->setSecurityHeaders();
        
        // Verificar HTTPS en producción
        $this->enforceHTTPS();
        
        // Limpiar datos de entrada
        $this->sanitizeGlobalInput();
    }
    
    /**
     * Establece headers de seguridad
     */
    private function setSecurityHeaders() {
        // Prevenir XSS
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: DENY');
        header('X-XSS-Protection: 1; mode=block');
        header('Referrer-Policy: strict-origin-when-cross-origin');
        
        // Content Security Policy
        $csp = "default-src 'self'; " .
               "script-src 'self' 'unsafe-inline' cdn.jsdelivr.net cdnjs.cloudflare.com; " .
               "style-src 'self' 'unsafe-inline' cdn.jsdelivr.net cdnjs.cloudflare.com; " .
               "font-src 'self' cdnjs.cloudflare.com; " .
               "img-src 'self' data: https:; " .
               "connect-src 'self'";
        
        header("Content-Security-Policy: $csp");
    }
    
    /**
     * Fuerza HTTPS en producción
     */
    private function enforceHTTPS() {
        if (!$this->isHTTPS() && $this->isProduction()) {
            $redirectURL = 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
            header("Location: $redirectURL");
            exit();
        }
    }
    
    /**
     * Sanitiza datos de entrada globales
     */
    private function sanitizeGlobalInput() {
        $_GET = $this->sanitizeArray($_GET);
        $_POST = $this->sanitizeArray($_POST);
        $_COOKIE = $this->sanitizeArray($_COOKIE);
    }
    
    /**
     * Sanitiza un array de datos
     */
    public function sanitizeArray($data) {
        if (!is_array($data)) {
            return $this->sanitizeInput($data);
        }
        
        $sanitized = [];
        foreach ($data as $key => $value) {
            $sanitizedKey = $this->sanitizeInput($key);
            $sanitized[$sanitizedKey] = is_array($value) ? 
                $this->sanitizeArray($value) : 
                $this->sanitizeInput($value);
        }
        
        return $sanitized;
    }
    
    /**
     * Sanitiza una entrada individual
     */
    public function sanitizeInput($input) {
        if (is_string($input)) {
            // Eliminar caracteres de control
            $input = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $input);
            
            // Trim y normalizar espacios
            $input = trim($input);
            $input = preg_replace('/\s+/', ' ', $input);
            
            // Filtrar caracteres peligrosos básicos
            $input = str_replace(['<script', '</script', 'javascript:', 'vbscript:', 'onload='], '', $input);
        }
        
        return $input;
    }
    
    /**
     * Valida entrada según tipo
     */
    public function validateInput($input, $type, $options = []) {
        switch ($type) {
            case 'email':
                return filter_var($input, FILTER_VALIDATE_EMAIL) !== false;
                
            case 'integer':
                $min = $options['min'] ?? null;
                $max = $options['max'] ?? null;
                $flags = 0;
                if ($min !== null || $max !== null) {
                    $flags = FILTER_FLAG_ALLOW_OCTAL | FILTER_FLAG_ALLOW_HEX;
                }
                $result = filter_var($input, FILTER_VALIDATE_INT, $flags);
                if ($result === false) return false;
                if ($min !== null && $result < $min) return false;
                if ($max !== null && $result > $max) return false;
                return true;
                
            case 'float':
                return filter_var($input, FILTER_VALIDATE_FLOAT) !== false;
                
            case 'url':
                return filter_var($input, FILTER_VALIDATE_URL) !== false;
                
            case 'alphanumeric':
                return preg_match('/^[a-zA-Z0-9]+$/', $input);
                
            case 'alpha':
                return preg_match('/^[a-zA-Z\s]+$/', $input);
                
            case 'text':
                $maxLength = $options['max_length'] ?? 1000;
                return strlen($input) <= $maxLength && !$this->containsMaliciousContent($input);
                
            case 'password':
                $minLength = $options['min_length'] ?? 8;
                return strlen($input) >= $minLength && $this->isStrongPassword($input);
                
            default:
                return true;
        }
    }
    
    /**
     * Verifica si el contenido es malicioso
     */
    private function containsMaliciousContent($input) {
        $patterns = [
            '/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/mi',
            '/javascript:/i',
            '/vbscript:/i',
            '/onload\s*=/i',
            '/onerror\s*=/i',
            '/onclick\s*=/i',
            '/<iframe/i',
            '/<object/i',
            '/<embed/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Verifica si una contraseña es fuerte
     */
    private function isStrongPassword($password) {
        // Al menos 8 caracteres, una mayúscula, una minúscula, un número
        return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).{8,}$/', $password);
    }
    
    /**
     * Genera token CSRF
     */
    public function generateCSRFToken() {
        if (!isset($_SESSION)) {
            session_start();
        }
        
        $token = bin2hex(random_bytes(32));
        $_SESSION['csrf_token'] = $token;
        $_SESSION['csrf_token_time'] = time();
        
        return $token;
    }
    
    /**
     * Verifica token CSRF
     */
    public function verifyCSRFToken($token) {
        if (!isset($_SESSION)) {
            session_start();
        }
        
        // Verificar que existe el token en sesión
        if (!isset($_SESSION['csrf_token']) || !isset($_SESSION['csrf_token_time'])) {
            return false;
        }
        
        // Verificar expiración (30 minutos)
        if (time() - $_SESSION['csrf_token_time'] > 1800) {
            unset($_SESSION['csrf_token'], $_SESSION['csrf_token_time']);
            return false;
        }
        
        // Verificar que coinciden
        return hash_equals($_SESSION['csrf_token'], $token);
    }
    
    /**
     * Rate limiting por IP
     */
    public function checkRateLimit($ip = null) {
        $ip = $ip ?? $this->getClientIP();
        $currentTime = time();
        
        // Limpiar ventanas expiradas
        $this->cleanupRateLimits($currentTime);
        
        // Verificar IP bloqueada
        if (isset($this->blockedIPs[$ip]) && $this->blockedIPs[$ip] > $currentTime) {
            $this->logSecurityEvent('rate_limit_exceeded', ['ip' => $ip]);
            return false;
        }
        
        // Contar requests en la ventana actual
        if (!isset($this->rateLimits[$ip])) {
            $this->rateLimits[$ip] = [];
        }
        
        $this->rateLimits[$ip][] = $currentTime;
        $requestsInWindow = count(array_filter($this->rateLimits[$ip], function($timestamp) use ($currentTime) {
            return ($currentTime - $timestamp) < $this->rateLimitWindow;
        }));
        
        // Verificar límite
        if ($requestsInWindow > $this->maxRequestsPerMinute) {
            $this->blockedIPs[$ip] = $currentTime + $this->lockoutDuration;
            $this->logSecurityEvent('rate_limit_blocked', ['ip' => $ip, 'requests' => $requestsInWindow]);
            return false;
        }
        
        return true;
    }
    
    /**
     * Registra intento de login
     */
    public function recordLoginAttempt($username, $success, $ip = null) {
        $ip = $ip ?? $this->getClientIP();
        $key = "login_attempts_{$ip}_{$username}";
        
        if (!isset($_SESSION)) {
            session_start();
        }
        
        if (!isset($_SESSION[$key])) {
            $_SESSION[$key] = [];
        }
        
        $_SESSION[$key][] = [
            'timestamp' => time(),
            'success' => $success,
            'ip' => $ip
        ];
        
        // Limpiar intentos antiguos (más de 1 hora)
        $_SESSION[$key] = array_filter($_SESSION[$key], function($attempt) {
            return (time() - $attempt['timestamp']) < 3600;
        });
        
        // Verificar si debe bloquear
        $failedAttempts = count(array_filter($_SESSION[$key], function($attempt) {
            return !$attempt['success'] && (time() - $attempt['timestamp']) < $this->lockoutDuration;
        }));
        
        if ($failedAttempts >= $this->maxLoginAttempts) {
            $this->logSecurityEvent('account_locked', [
                'username' => $username,
                'ip' => $ip,
                'failed_attempts' => $failedAttempts
            ]);
            return false;
        }
        
        if (!$success) {
            $this->logSecurityEvent('login_failed', [
                'username' => $username,
                'ip' => $ip,
                'attempt_number' => $failedAttempts + 1
            ]);
        } else {
            $this->logSecurityEvent('login_success', [
                'username' => $username,
                'ip' => $ip
            ]);
        }
        
        return true;
    }
    
    /**
     * Verifica si un usuario/IP está bloqueado
     */
    public function isBlocked($username, $ip = null) {
        $ip = $ip ?? $this->getClientIP();
        $key = "login_attempts_{$ip}_{$username}";
        
        if (!isset($_SESSION)) {
            session_start();
        }
        
        if (!isset($_SESSION[$key])) {
            return false;
        }
        
        $failedAttempts = count(array_filter($_SESSION[$key], function($attempt) {
            return !$attempt['success'] && (time() - $attempt['timestamp']) < $this->lockoutDuration;
        }));
        
        return $failedAttempts >= $this->maxLoginAttempts;
    }
    
    /**
     * Registra evento de seguridad
     */
    public function logSecurityEvent($event, $details = []) {
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'event' => $event,
            'ip' => $this->getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'details' => $details
        ];
        
        $this->securityLog[] = $logEntry;
        
        // Escribir a archivo de log
        $logFile = dirname(__DIR__) . '/logs/security-' . date('Y-m-d') . '.log';
        $logDir = dirname($logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0777, true);
        }
        
        $logLine = json_encode($logEntry) . PHP_EOL;
        file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Obtiene IP del cliente
     */
    private function getClientIP() {
        $headers = [
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_CLIENT_IP',            // Proxy
            'HTTP_X_FORWARDED_FOR',      // Load balancer/proxy
            'HTTP_X_FORWARDED',          // Proxy
            'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
            'HTTP_FORWARDED_FOR',        // Proxy
            'HTTP_FORWARDED',           // Proxy
            'REMOTE_ADDR'               // Standard
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ip = trim($ips[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Limpia rate limits expirados
     */
    private function cleanupRateLimits($currentTime) {
        foreach ($this->rateLimits as $ip => &$timestamps) {
            $timestamps = array_filter($timestamps, function($timestamp) use ($currentTime) {
                return ($currentTime - $timestamp) < $this->rateLimitWindow;
            });
            
            if (empty($timestamps)) {
                unset($this->rateLimits[$ip]);
            }
        }
        
        // Limpiar IPs bloqueadas expiradas
        foreach ($this->blockedIPs as $ip => $expiry) {
            if ($expiry <= $currentTime) {
                unset($this->blockedIPs[$ip]);
            }
        }
    }
    
    /**
     * Verifica si está en HTTPS
     */
    private function isHTTPS() {
        return !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';
    }
    
    /**
     * Verifica si está en producción
     */
    private function isProduction() {
        return (defined('ENVIRONMENT') && ENVIRONMENT === 'production') || 
               (!empty($_SERVER['SERVER_NAME']) && $_SERVER['SERVER_NAME'] !== 'localhost');
    }
    
    /**
     * Obtiene estadísticas de seguridad
     */
    public function getSecurityStats() {
        $recentEvents = array_filter($this->securityLog, function($event) {
            return (time() - strtotime($event['timestamp'])) < 3600; // Última hora
        });
        
        $eventCounts = array_count_values(array_column($recentEvents, 'event'));
        
        return [
            'recent_events' => count($recentEvents),
            'blocked_ips' => count($this->blockedIPs),
            'event_types' => $eventCounts,
            'security_level' => $this->calculateSecurityLevel($eventCounts)
        ];
    }
    
    /**
     * Calcula nivel de seguridad
     */
    private function calculateSecurityLevel($eventCounts) {
        $criticalEvents = ($eventCounts['rate_limit_exceeded'] ?? 0) + 
                         ($eventCounts['account_locked'] ?? 0) + 
                         ($eventCounts['login_failed'] ?? 0);
        
        if ($criticalEvents > 10) return 'high_risk';
        if ($criticalEvents > 5) return 'medium_risk';
        if ($criticalEvents > 0) return 'low_risk';
        return 'secure';
    }
}
?>