<?php
/**
 * PÁGINA PARA AJUSTAR STOCK DE INVENTARIO
 */

// Verificar autenticación
if (!isset($_SESSION['usuario_id'])) {
    header('Location: /app-web-Enciso/public/login');
    exit;
}


// Obtener ID del utensilio
$utensilio_id = $_GET['id'] ?? null;

if (!$utensilio_id) {
    $_SESSION['error'] = 'ID de utensilio requerido';
    header('Location: /app-web-Enciso/public/inventario');
    exit;
}

try {
    $pdo = new PDO("mysql:host=localhost;dbname=server_inventario_utensilios;charset=utf8mb4", 'server_admin', 'Admin2025!');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Obtener datos del utensilio
    $stmt = $pdo->prepare("SELECT * FROM utensilios WHERE id = ? AND activo = 1");
    $stmt->execute([$utensilio_id]);
    $utensilio = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$utensilio) {
        $_SESSION['error'] = 'Utensilio no encontrado';
        header('Location: /app-web-Enciso/public/inventario');
        exit;
    }
    
    // Procesar formulario de ajuste
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $nueva_cantidad_total = (int)($_POST['nueva_cantidad_total'] ?? 0);
        $tipo_ajuste = $_POST['tipo_ajuste'] ?? '';
        $motivo = trim($_POST['motivo'] ?? '');
        
        // Validaciones
        if ($nueva_cantidad_total < 0) {
            throw new Exception('La cantidad no puede ser negativa');
        }
        
        if (empty($tipo_ajuste)) {
            throw new Exception('❌ OBLIGATORIO: Debe seleccionar un tipo de ajuste');
        }
        
        // Validación del motivo - OBLIGATORIO con mínimo de caracteres
        if (empty($motivo)) {
            throw new Exception('El motivo es obligatorio para mantener la trazabilidad');
        }
        
        if (strlen($motivo) < 10) {
            throw new Exception('El motivo debe tener al menos 10 caracteres');
        }
        
        // Calcular diferencia y nueva cantidad disponible
        $cantidad_actual = $utensilio['cantidad_total'];
        $diferencia = $nueva_cantidad_total - $cantidad_actual;
        $nueva_cantidad_disponible = $utensilio['cantidad_disponible'] + $diferencia;
        
        // Validar que cantidad disponible no sea negativa
        if ($nueva_cantidad_disponible < 0) {
            $max_reduccion = $utensilio['cantidad_disponible'];
            throw new Exception("No se puede reducir más de {$max_reduccion} unidades. Hay {$utensilio['cantidad_en_uso']} unidades en uso que no se pueden eliminar.");
        }
        
        // Iniciar transacción
        $pdo->beginTransaction();
        
        try {
            // Actualizar cantidades del utensilio
            $stmt = $pdo->prepare("UPDATE utensilios SET cantidad_total = ?, cantidad_disponible = ?, fecha_actualizacion = NOW() WHERE id = ?");
            $stmt->execute([$nueva_cantidad_total, $nueva_cantidad_disponible, $utensilio_id]);
            
            // Registrar movimiento para auditoría
            $movimiento_tipo = ($diferencia > 0) ? 'entrada' : 'salida';
            $cantidad_movimiento = abs($diferencia);
            
            if ($cantidad_movimiento > 0) {
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_inventario 
                    (utensilio_id, tipo, cantidad, motivo, usuario_id, fecha, observaciones) 
                    VALUES (?, ?, ?, ?, ?, NOW(), ?)
                ");
                
                $observaciones = "Ajuste de inventario: {$tipo_ajuste}. De {$cantidad_actual} a {$nueva_cantidad_total} unidades.";
                
                $stmt->execute([
                    $utensilio_id,
                    $movimiento_tipo,
                    $cantidad_movimiento,
                    $motivo,
                    $_SESSION['usuario_id'],
                    $observaciones
                ]);
            }
            
            $pdo->commit();
            
            $_SESSION['success'] = "Stock ajustado exitosamente. {$tipo_ajuste}: {$cantidad_actual} → {$nueva_cantidad_total} unidades";
            header('Location: /app-web-Enciso/public/inventario');
            exit;
            
        } catch (Exception $e) {
            $pdo->rollback();
            throw $e;
        }
    }
    
} catch (Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ajustar Stock - Sistema CAYEJERO</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../../assets/css/dashboard.css?v=20251210003" rel="stylesheet">
    <link href="../../assets/css/utensilios.css?v=20251210003" rel="stylesheet">
    <link href="../../assets/css/form-utensilio.css?v=20251210003" rel="stylesheet">
    <style>
        /* CAYEJERO - Forzar tema oscuro */
        body, html { background: #0a0a0a !important; }
        .container-fluid { background: #0a0a0a !important; }
        
        /* Navbar */
        .navbar { background: linear-gradient(135deg, #1a1a1a 0%, #2d2d2d 100%) !important; border-bottom: 3px solid #FF8C00 !important; min-height: 80px !important; }
        .navbar-logo { height: 70px !important; filter: drop-shadow(0 0 10px rgba(255, 140, 0, 0.5)); }
        .navbar .nav-link { color: #FFD700 !important; font-weight: 600 !important; transition: all 0.3s; }
        .navbar .nav-link:hover { color: #FF8C00 !important; transform: translateY(-2px); }
        .navbar .nav-link i { color: #FF8C00 !important; }
        
        /* Cards */
        .card { background: #1a1a1a !important; border: 2px solid rgba(255, 140, 0, 0.3) !important; }
        .card-body { background: #1a1a1a !important; }
        .card-header { background: linear-gradient(135deg, rgba(255, 140, 0, 0.3), rgba(255, 215, 0, 0.3)) !important; color: #FFD700 !important; border-bottom: 2px solid #FF8C00 !important; }
        
        /* Textos */
        h1, h2, h3, h4, h5, h6 { color: #FFD700 !important; font-weight: 800 !important; text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.8); }
        label, .form-label { color: #FFD700 !important; font-weight: 600 !important; }
        p, span, small, .text-muted { color: #e0e0e0 !important; }
        
        /* Formularios */
        .form-control, .form-select { background: rgba(45, 45, 45, 0.8) !important; border: 2px solid rgba(255, 140, 0, 0.4) !important; color: #fff !important; }
        .form-control:focus, .form-select:focus { background: rgba(255, 140, 0, 0.15) !important; border-color: #FF8C00 !important; color: #fff !important; }
        
        /* Opciones del select */
        select option { background: #1a1a1a !important; color: #FFD700 !important; font-weight: 600; }
        select option:checked { background: #FF8C00 !important; color: #000 !important; }
        
        /* Alertas */
        .alert { background: rgba(26, 26, 26, 0.9) !important; border: 2px solid rgba(255, 140, 0, 0.3) !important; color: #fff !important; }
        
        /* Botones */
        .btn-primary { background: linear-gradient(135deg, #FF8C00, #FFD700) !important; border: none !important; color: #000 !important; font-weight: 700 !important; }
        .btn-secondary { background: #5a6268 !important; color: #fff !important; }
        .badge { font-weight: 700 !important; }
        
        /* Estilos específicos para ajustar stock */
        .stock-info {
            background: linear-gradient(135deg, rgba(255, 140, 0, 0.3) 0%, rgba(255, 215, 0, 0.3) 100%) !important;
            color: #FFD700 !important;
            border: 2px solid #FF8C00 !important;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 25px;
        }
        
        .stock-badge {
            font-size: 1.2em;
            padding: 0.5em 1em;
            background: linear-gradient(135deg, #FF8C00, #FFD700) !important;
            color: #000 !important;
        }
        
        /* Estilos para tipos de ajuste */
        .tipo-ajuste-card {
            background: rgba(26, 26, 26, 0.8) !important;
            border: 2px solid rgba(255, 140, 0, 0.3) !important;
            color: #FFD700 !important;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .tipo-ajuste-card:hover {
            border-color: #FF8C00 !important;
            box-shadow: 0 4px 15px rgba(255, 140, 0, 0.4) !important;
            transform: translateY(-2px);
        }
        
        .tipo-ajuste-card.selected {
            border-color: #FF8C00 !important;
            background: linear-gradient(135deg, rgba(255, 140, 0, 0.2), rgba(255, 215, 0, 0.2)) !important;
        }
        
        /* Validación en tiempo real */
        .form-control.valid {
            border-color: #198754;
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 8 8'%3e%3cpath fill='%23198754' d='m2.3 6.73.94-.94 1.93 1.93 4.27-4.27.94.94-5.21 5.21-2.87-2.87z'/%3e%3c/svg%3e");
            background-repeat: no-repeat;
            background-position: right calc(.375em + .1875rem) center;
            background-size: calc(.75em + .375rem) calc(.75em + .375rem);
        }
        
        .form-control.invalid {
            border-color: #dc3545;
        }
        
        /* Estilos para la calculadora de diferencias */
        .diferencia-display {
            font-family: 'Courier New', monospace;
            font-size: 1.1em;
            padding: 15px;
            border-radius: 10px;
            margin: 15px 0;
            font-weight: 700;
        }
        
        .diferencia-positiva {
            background-color: rgba(40, 167, 69, 0.2) !important;
            border: 2px solid #28a745 !important;
            color: #4caf50 !important;
        }
        
        .diferencia-negativa {
            background-color: rgba(220, 53, 69, 0.2) !important;
            border: 2px solid #dc3545 !important;
            color: #ff6b6b !important;
        }
        
        .diferencia-neutral {
            background-color: rgba(255, 140, 0, 0.2) !important;
            border: 2px solid #FF8C00 !important;
            color: #FFD700 !important;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <?php include __DIR__ . '/../partials/navbar-simple.php'; ?>
    
    <div class="container-fluid py-4">
        <div class="row justify-content-center">
            <div class="col-md-10">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2><i class="fas fa-calculator text-primary"></i> Ajustar Stock de Inventario</h2>
                        <small class="text-muted">
                            <i class="fas fa-shield-alt"></i> 
                            Todos los ajustes quedan registrados para auditoría y trazabilidad completa
                        </small>
                    </div>
                    <div class="btn-group">
                        <a href="/app-web-Enciso/public/inventario/editar?id=<?php echo $utensilio_id; ?>" class="btn btn-warning">
                            <i class="fas fa-edit"></i> Editar Info
                        </a>
                        <a href="/app-web-Enciso/public/inventario" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> Volver
                        </a>
                    </div>
                </div>

                <!-- Mensajes -->
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Información del Utensilio -->
                <div class="stock-info">
                    <div class="row align-items-center">
                        <div class="col-md-6">
                            <h4><i class="fas fa-utensils me-2"></i><?php echo htmlspecialchars($utensilio['nombre']); ?></h4>
                            <p class="mb-2"><?php echo htmlspecialchars($utensilio['descripcion'] ?? 'Sin descripción'); ?></p>
                            <small><i class="fas fa-tag me-1"></i><?php echo htmlspecialchars($utensilio['categoria']); ?></small>
                        </div>
                        <div class="col-md-6">
                            <div class="row text-center">
                                <div class="col-4">
                                    <div class="stock-badge badge bg-light text-dark">
                                        <div style="font-size: 1.5em;"><?php echo $utensilio['cantidad_total']; ?></div>
                                        <small>Total</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Formulario de Ajuste -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-cogs"></i> Realizar Ajuste de Stock
                        </h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" id="formAjusteStock" novalidate>
                            <!-- Tipo de Ajuste -->
                            <div class="mb-4">
                                <label class="form-label fw-bold">Tipo de Ajuste *</label>
                                <div class="row">
                                    <div class="col-md-3 mb-2">
                                        <div class="card tipo-ajuste-card h-100" onclick="seleccionarTipo('inventario_fisico')">
                                            <div class="card-body text-center">
                                                <i class="fas fa-clipboard-check fa-2x text-info mb-2"></i>
                                                <h6>Inventario Físico</h6>
                                                <small class="text-muted">Corrección por conteo</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-2">
                                        <div class="card tipo-ajuste-card h-100" onclick="seleccionarTipo('compra')">
                                            <div class="card-body text-center">
                                                <i class="fas fa-shopping-cart fa-2x text-success mb-2"></i>
                                                <h6>Compra/Entrada</h6>
                                                <small class="text-muted">Nueva adquisición</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-2">
                                        <div class="card tipo-ajuste-card h-100" onclick="seleccionarTipo('perdida')">
                                            <div class="card-body text-center">
                                                <i class="fas fa-exclamation-triangle fa-2x text-warning mb-2"></i>
                                                <h6>Pérdida/Rotura</h6>
                                                <small class="text-muted">Reducción no registrada</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-2">
                                        <div class="card tipo-ajuste-card h-100" onclick="seleccionarTipo('correccion')">
                                            <div class="card-body text-center">
                                                <i class="fas fa-tools fa-2x text-secondary mb-2"></i>
                                                <h6>Corrección</h6>
                                                <small class="text-muted">Ajuste general</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <input type="hidden" id="tipo_ajuste" name="tipo_ajuste" required>
                            </div>

                            <!-- Nueva Cantidad -->
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="nueva_cantidad_total" class="form-label fw-bold">
                                        Nueva Cantidad Total *
                                    </label>
                                    <input type="number" 
                                           class="form-control form-control-lg" 
                                           id="nueva_cantidad_total" 
                                           name="nueva_cantidad_total" 
                                           value="<?php echo $utensilio['cantidad_total']; ?>"
                                           min="0" 
                                           required
                                           oninput="calcularDiferencia()">
                                    <div class="form-text">
                                        Cantidad actual: <strong><?php echo $utensilio['cantidad_total']; ?></strong> unidades
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">Cálculo de Diferencia</label>
                                    <div id="diferencia-display" class="diferencia-display diferencia-neutral">
                                        <div class="d-flex justify-content-between">
                                            <span>Cantidad actual:</span>
                                            <span><strong><?php echo $utensilio['cantidad_total']; ?></strong></span>
                                        </div>
                                        <div class="d-flex justify-content-between">
                                            <span>Nueva cantidad:</span>
                                            <span id="nueva-cantidad-display"><strong><?php echo $utensilio['cantidad_total']; ?></strong></span>
                                        </div>
                                        <hr class="my-2">
                                        <div class="d-flex justify-content-between">
                                            <span>Diferencia:</span>
                                            <span id="diferencia-valor"><strong>0</strong></span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Motivo -->
                            <div class="mb-4">
                                <label for="motivo" class="form-label fw-bold">
                                    Motivo del Ajuste *
                                </label>
                                <textarea class="form-control" 
                                          id="motivo" 
                                          name="motivo" 
                                          rows="4" 
                                          placeholder="Describa el motivo del ajuste de inventario (mínimo 10 caracteres)..."
                                          required
                                          oninput="validarMotivo()"></textarea>
                                <div class="d-flex justify-content-between">
                                    <div class="form-text">
                                        <i class="fas fa-info-circle"></i> 
                                        Requerido para mantener trazabilidad
                                    </div>
                                    <div id="contador-caracteres" class="form-text">
                                        <span id="caracteres-actuales">0</span> / 10 mínimo
                                    </div>
                                </div>
                            </div>

                            <!-- Botones -->
                            <div class="row">
                                <div class="col-12">
                                    <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                        <a href="/app-web-Enciso/public/inventario" class="btn btn-secondary me-md-2">
                                            <i class="fas fa-times"></i> Cancelar
                                        </a>
                                        <button type="submit" class="btn btn-secondary" id="btn-ajustar" disabled>
                                            <i class="fas fa-lock"></i> Complete todos los campos obligatorios
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        let tipoSeleccionado = false;
        let motivoValido = false;
        const cantidadActual = <?php echo $utensilio['cantidad_total']; ?>;
        
        // Seleccionar tipo de ajuste
        function seleccionarTipo(tipo) {
            // Remover selección anterior
            document.querySelectorAll('.tipo-ajuste-card').forEach(card => {
                card.classList.remove('selected');
            });
            
            // Seleccionar nuevo tipo
            event.currentTarget.classList.add('selected');
            document.getElementById('tipo_ajuste').value = tipo;
            tipoSeleccionado = true;
            
            // Actualizar placeholders según el tipo
            const motivo = document.getElementById('motivo');
            switch(tipo) {
                case 'inventario_fisico':
                    motivo.placeholder = 'Ej: Inventario físico mensual realizado el [fecha]. Diferencia encontrada debido a...';
                    break;
                case 'compra':
                    motivo.placeholder = 'Ej: Compra a proveedor [nombre] según factura #[número]. Fecha de recepción...';
                    break;
                case 'perdida':
                    motivo.placeholder = 'Ej: Rotura durante el servicio del día [fecha]. Incidente reportado por...';
                    break;
                case 'correccion':
                    motivo.placeholder = 'Ej: Corrección de error en registro anterior. Motivo del error...';
                    break;
            }
            
            validarFormulario();
        }
        
        // Calcular diferencia en tiempo real
        function calcularDiferencia() {
            const nuevaCantidad = parseInt(document.getElementById('nueva_cantidad_total').value) || 0;
            const diferencia = nuevaCantidad - cantidadActual;
            
            // Actualizar display
            document.getElementById('nueva-cantidad-display').innerHTML = `<strong>${nuevaCantidad}</strong>`;
            
            const diferenciaValor = document.getElementById('diferencia-valor');
            const diferenciaDisplay = document.getElementById('diferencia-display');
            
            if (diferencia > 0) {
                diferenciaValor.innerHTML = `<strong>+${diferencia}</strong> (Entrada)`;
                diferenciaDisplay.className = 'diferencia-display diferencia-positiva';
            } else if (diferencia < 0) {
                diferenciaValor.innerHTML = `<strong>${diferencia}</strong> (Salida)`;
                diferenciaDisplay.className = 'diferencia-display diferencia-negativa';
            } else {
                diferenciaValor.innerHTML = `<strong>0</strong> (Sin cambio)`;
                diferenciaDisplay.className = 'diferencia-display diferencia-neutral';
            }
            
            validarFormulario();
        }
        
        // Validar motivo - Solo obligatorio y mínimo de caracteres
        function validarMotivo() {
            const motivo = document.getElementById('motivo');
            const contador = document.getElementById('caracteres-actuales');
            const longitud = motivo.value.trim().length;
            
            contador.textContent = longitud;
            
            if (longitud >= 10) {
                motivo.classList.remove('invalid');
                motivo.classList.add('valid');
                motivoValido = true;
            } else {
                motivo.classList.remove('valid');
                motivo.classList.add('invalid');
                motivoValido = false;
            }
            
            validarFormulario();
        }
        

        
        // Validar formulario completo
        function validarFormulario() {
            const btnAjustar = document.getElementById('btn-ajustar');
            
            if (tipoSeleccionado && motivoValido) {
                btnAjustar.disabled = false;
                btnAjustar.classList.remove('btn-secondary');
                btnAjustar.classList.add('btn-primary');
                btnAjustar.innerHTML = '<i class="fas fa-save"></i> Ejecutar Ajuste de Stock';
            } else {
                btnAjustar.disabled = true;
                btnAjustar.classList.remove('btn-primary');
                btnAjustar.classList.add('btn-secondary');
                btnAjustar.innerHTML = '<i class="fas fa-lock"></i> Complete los campos obligatorios';
            }
        }
        
        // Confirmación antes de enviar
        document.getElementById('formAjusteStock').addEventListener('submit', function(e) {
            const nuevaCantidad = document.getElementById('nueva_cantidad_total').value;
            const diferencia = parseInt(nuevaCantidad) - cantidadActual;
            const tipo = document.getElementById('tipo_ajuste').value;
            const motivo = document.getElementById('motivo').value.trim();
            
            // Validación final simple
            if (!tipo) {
                alert('Debe seleccionar un tipo de ajuste');
                e.preventDefault();
                return;
            }
            
            if (!motivo || motivo.length < 10) {
                alert('El motivo es obligatorio y debe tener al menos 10 caracteres');
                e.preventDefault();
                return;
            }
            
            // Confirmación
            let mensaje = `¿Confirma el ajuste de stock?\n\n`;
            mensaje += `Utensilio: <?php echo htmlspecialchars($utensilio['nombre']); ?>\n`;
            mensaje += `Cantidad actual: ${cantidadActual}\n`;
            mensaje += `Nueva cantidad: ${nuevaCantidad}\n`;
            mensaje += `Diferencia: ${diferencia > 0 ? '+' : ''}${diferencia}\n`;
            mensaje += `Tipo: ${tipo}\n\n`;
            mensaje += `Esta acción quedará registrada en el historial.`;
            
            if (!confirm(mensaje)) {
                e.preventDefault();
            }
        });
        
        // Inicializar
        document.addEventListener('DOMContentLoaded', function() {
            calcularDiferencia();
        });
    </script>
</body>
</html>


