<?php
/**
 * PÁGINA PARA REGISTRAR MOVIMIENTOS DE INVENTARIO
 */

// Verificar autenticación
if (!isset($_SESSION['usuario_id'])) {
    header('Location: /app-web-Enciso/public/login');
    exit;
}

// Obtener ID del utensilio
$utensilio_id = $_GET['id'] ?? null;
if (!$utensilio_id) {
    $_SESSION['error'] = 'ID de utensilio no válido';
    header('Location: /app-web-Enciso/public/inventario');
    exit;
}

try {
    $pdo = new PDO("mysql:host=localhost;dbname=server_inventario_utensilios;charset=utf8mb4", 'server_admin', 'Admin2025!');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Obtener datos del utensilio
    $stmt = $pdo->prepare("SELECT * FROM utensilios WHERE id = ?");
    $stmt->execute([$utensilio_id]);
    $utensilio = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$utensilio) {
        $_SESSION['error'] = 'Utensilio no encontrado';
        header('Location: /app-web-Enciso/public/inventario');
        exit;
    }
    
    // Crear tabla de movimientos si no existe
    $pdo->exec("CREATE TABLE IF NOT EXISTS movimientos_inventario (
        id INT AUTO_INCREMENT PRIMARY KEY,
        utensilio_id INT NOT NULL,
        tipo_movimiento ENUM('entrada', 'salida') NOT NULL,
        cantidad INT NOT NULL,
        motivo VARCHAR(255),
        usuario_id INT,
        fecha_movimiento TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (utensilio_id) REFERENCES utensilios(id)
    )");
    
    // Procesar formulario si es POST
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $tipo_movimiento = $_POST['tipo_movimiento'] ?? '';
        $cantidad = (int)($_POST['cantidad'] ?? 0);
        $motivo = $_POST['motivo'] ?? '';
        
        if (empty($tipo_movimiento) || $cantidad <= 0) {
            throw new Exception('Por favor complete todos los campos requeridos');
        }
        
        // Validar que el motivo sea obligatorio para TODOS los movimientos
        if (in_array($tipo_movimiento, ['entrada', 'salida'])) {
            if (empty($motivo)) {
                $tipo_texto = match($tipo_movimiento) {
                    'entrada' => 'entrada al inventario',
                    'salida' => 'salida del inventario',
                    default => $tipo_movimiento
                };
                throw new Exception("El motivo es obligatorio para la {$tipo_texto}. Por favor describe la razón del movimiento para mantener la trazabilidad del inventario.");
            }
            
            // Validar longitud mínima del motivo (al menos 10 caracteres)
            if (strlen(trim($motivo)) < 10) {
                throw new Exception("El motivo debe ser más descriptivo (mínimo 10 caracteres). Por favor proporciona más detalles sobre la razón del movimiento.");
            }
        }
        
        // Validar movimientos según tipo
        if ($tipo_movimiento === 'salida') {
            if ($cantidad > $utensilio['cantidad_total']) {
                throw new Exception('No hay suficiente cantidad en el inventario. Disponible: ' . $utensilio['cantidad_total']);
            }
        }
        
        // Iniciar transacción
        $pdo->beginTransaction();
        
        try {
            // Registrar movimiento
            $stmt = $pdo->prepare("INSERT INTO movimientos_inventario (utensilio_id, tipo_movimiento, cantidad, motivo, usuario_id) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$utensilio_id, $tipo_movimiento, $cantidad, $motivo, $_SESSION['usuario_id']]);
            
            // Actualizar cantidades en utensilio
            switch ($tipo_movimiento) {
                case 'entrada':
                    $stmt = $pdo->prepare("UPDATE utensilios SET cantidad_total = cantidad_total + ? WHERE id = ?");
                    $stmt->execute([$cantidad, $utensilio_id]);
                    break;
                    
                case 'salida':
                    $stmt = $pdo->prepare("UPDATE utensilios SET cantidad_total = cantidad_total - ? WHERE id = ?");
                    $stmt->execute([$cantidad, $utensilio_id]);
                    break;
            }
            
            $pdo->commit();
            $_SESSION['success'] = 'Movimiento registrado exitosamente';
            header('Location: /app-web-Enciso/public/inventario');
            exit;
            
        } catch (Exception $e) {
            $pdo->rollback();
            throw $e;
        }
    }
    
    // Obtener últimos movimientos
    $stmt = $pdo->prepare("SELECT m.*, u.nombre as usuario_nombre FROM movimientos_inventario m 
                           LEFT JOIN usuarios u ON m.usuario_id = u.id 
                           WHERE m.utensilio_id = ? ORDER BY m.fecha_movimiento DESC LIMIT 10");
    $stmt->execute([$utensilio_id]);
    $movimientos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Movimiento de Inventario - Sistema CAYEJERO</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../../assets/css/dashboard.css?v=20251210003" rel="stylesheet">
    <link href="../../assets/css/utensilios.css?v=20251210003" rel="stylesheet">
    <link href="../../assets/css/form-utensilio.css?v=20251210003" rel="stylesheet">
    <style>
        /* CAYEJERO - Forzar tema oscuro */
        body, html { background: #0a0a0a !important; }
        .container-fluid { background: #0a0a0a !important; }
        
        /* Navbar */
        .navbar { background: linear-gradient(135deg, #1a1a1a 0%, #2d2d2d 100%) !important; border-bottom: 3px solid #FF8C00 !important; min-height: 80px !important; }
        .navbar-logo { height: 70px !important; filter: drop-shadow(0 0 10px rgba(255, 140, 0, 0.5)); }
        .navbar .nav-link { color: #FFD700 !important; font-weight: 600 !important; transition: all 0.3s; }
        .navbar .nav-link:hover { color: #FF8C00 !important; transform: translateY(-2px); }
        .navbar .nav-link i { color: #FF8C00 !important; }
        
        /* Cards */
        .card { background: #1a1a1a !important; border: 2px solid rgba(255, 140, 0, 0.3) !important; }
        .card-body { background: #1a1a1a !important; }
        .card-header { background: linear-gradient(135deg, rgba(255, 140, 0, 0.3), rgba(255, 215, 0, 0.3)) !important; color: #FFD700 !important; border-bottom: 2px solid #FF8C00 !important; }
        
        /* Textos */
        h1, h2, h3, h4, h5, h6 { color: #FFD700 !important; font-weight: 800 !important; text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.8); }
        label, .form-label { color: #FFD700 !important; font-weight: 600 !important; }
        p, span, small, .text-muted { color: #e0e0e0 !important; }
        
        /* Formularios */
        .form-control, .form-select, textarea { background: rgba(45, 45, 45, 0.8) !important; border: 2px solid rgba(255, 140, 0, 0.4) !important; color: #fff !important; }
        .form-control:focus, .form-select:focus, textarea:focus { background: rgba(255, 140, 0, 0.15) !important; border-color: #FF8C00 !important; color: #fff !important; }
        
        /* Opciones del select - Tema CAYEJERO */
        .form-select option {
            background: #1a1a1a !important;
            color: #FFD700 !important;
            font-weight: 600;
            padding: 10px;
        }
        
        .form-select option:hover,
        .form-select option:checked,
        .form-select option:focus {
            background: linear-gradient(135deg, #FF8C00, #FFD700) !important;
            color: #000 !important;
        }
        
        .form-select option:disabled {
            color: #888 !important;
            background: #2d2d2d !important;
        }
        
        /* Alertas */
        .alert { background: rgba(26, 26, 26, 0.9) !important; border: 2px solid rgba(255, 140, 0, 0.3) !important; color: #fff !important; }
        
        /* Botones */
        .btn-primary { background: linear-gradient(135deg, #FF8C00, #FFD700) !important; border: none !important; color: #000 !important; font-weight: 700 !important; }
        .btn-secondary { background: #5a6268 !important; color: #fff !important; }
        .btn-success { background: linear-gradient(135deg, #28a745, #20c997) !important; border: none !important; }
        .btn-danger { background: linear-gradient(135deg, #dc3545, #c82333) !important; border: none !important; }
        .badge { font-weight: 700 !important; }
        
        /* Estilos específicos para movimientos */
        .motivo-required {
            color: #ff6b6b !important;
            font-weight: bold;
        }
        
        .motivo-help {
            background: rgba(255, 193, 7, 0.2) !important;
            border: 2px solid #ffc107 !important;
            border-radius: 0.375rem;
            padding: 0.5rem 0.75rem;
            margin-top: 0.25rem;
            color: #FFD700 !important;
        }
        
        .border-warning {
            border-color: #ffc107 !important;
            box-shadow: 0 0 0 0.2rem rgba(255, 193, 7, 0.25) !important;
        }
        
        .border-success {
            border-color: #198754 !important;
            box-shadow: 0 0 0 0.2rem rgba(25, 135, 84, 0.25) !important;
        }
        
        .border-danger {
            border-color: #dc3545 !important;
            box-shadow: 0 0 0 0.2rem rgba(220, 53, 69, 0.25) !important;
        }
        
        /* Animación para campos requeridos */
        .required-field {
            animation: pulse-warning 2s infinite;
        }
        
        @keyframes pulse-warning {
            0% { box-shadow: 0 0 0 0 rgba(255, 193, 7, 0.4); }
            50% { box-shadow: 0 0 0 10px rgba(255, 193, 7, 0); }
            100% { box-shadow: 0 0 0 0 rgba(255, 193, 7, 0); }
        }
        
        /* Mejora visual para tipos de movimiento */
        #tipo_movimiento option:disabled {
            color: #6c757d;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <?php include __DIR__ . '/../partials/navbar-simple.php'; ?>
    
    <div class="container-fluid py-4">
        <div class="row">
            <!-- Formulario de movimiento -->
            <div class="col-md-8">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="fas fa-exchange-alt text-info"></i> Registrar Movimiento</h2>
                    <a href="/app-web-Enciso/public/inventario" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Volver al Inventario
                    </a>
                </div>

                <!-- Información del utensilio -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-utensils"></i> <?php echo htmlspecialchars($utensilio['nombre']); ?></h5>
                    </div>
                    <div class="card-body">
                        <div class="row text-center">
                            <div class="col-md-4">
                                <div class="badge bg-primary fs-6 mb-1">Cantidad Total</div>
                                <div class="h4"><?php echo $utensilio['cantidad_total']; ?></div>
                            </div>
                            <div class="col-md-4">
                                <div class="badge bg-info fs-6 mb-1">Código</div>
                                <div class="h6"><?php echo htmlspecialchars($utensilio['codigo'] ?? 'N/A'); ?></div>
                            </div>
                            <div class="col-md-4">
                                <div class="badge bg-<?php echo ($utensilio['cantidad_total'] ?? 0) <= ($utensilio['minimo'] ?? 0) ? 'danger' : 'success'; ?> fs-6 mb-1">Estado</div>
                                <div class="h6">
                                    <span class="badge bg-<?php echo ($utensilio['cantidad_total'] ?? 0) <= ($utensilio['minimo'] ?? 0) ? 'danger' : 'success'; ?>">
                                        <?php echo ($utensilio['cantidad_total'] ?? 0) <= ($utensilio['minimo'] ?? 0) ? 'Stock Bajo' : 'Disponible'; ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Mensajes -->
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Formulario -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-clipboard-list"></i> Nuevo Movimiento</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="tipo_movimiento" class="form-label">Tipo de Movimiento *</label>
                                    <select class="form-control" id="tipo_movimiento" name="tipo_movimiento" required>
                                        <option value="">Seleccione el tipo</option>
                                        <option value="entrada" <?php echo ($_POST['tipo_movimiento'] ?? '') === 'entrada' ? 'selected' : ''; ?>>
                                            📥 Entrada (Agregar al inventario)
                                        </option>
                                        <option value="salida" <?php echo ($_POST['tipo_movimiento'] ?? '') === 'salida' ? 'selected' : ''; ?>>
                                            📤 Salida (Pérdida o retiro del inventario)
                                        </option>
                                    </select>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="cantidad" class="form-label">Cantidad *</label>
                                    <input type="number" class="form-control" id="cantidad" name="cantidad" 
                                           value="<?php echo htmlspecialchars($_POST['cantidad'] ?? ''); ?>" min="1" required>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="motivo" class="form-label">
                                    Motivo 
                                    <span class="text-danger motivo-required" style="display: none;">*</span>
                                </label>
                                <textarea class="form-control" id="motivo" name="motivo" rows="3" 
                                          placeholder="Describe el motivo del movimiento..."><?php echo htmlspecialchars($_POST['motivo'] ?? ''); ?></textarea>
                                <div class="form-text">
                                    <span class="motivo-help text-muted" style="display: none;">
                                        <i class="fas fa-exclamation-triangle text-warning"></i>
                                        <strong>Obligatorio</strong> para todos los movimientos (mínimo 10 caracteres)
                                    </span>
                                    <div class="d-flex justify-content-between mt-1">
                                        <span></span>
                                        <small class="text-muted">
                                            <span id="char-count">0</span> caracteres
                                            <span id="char-status" class="ms-1"></span>
                                        </small>
                                    </div>
                                </div>
                            </div>

                            <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                <a href="/app-web-Enciso/public/inventario" class="btn btn-secondary me-md-2">
                                    <i class="fas fa-times"></i> Cancelar
                                </a>
                                <button type="submit" class="btn btn-info">
                                    <i class="fas fa-save"></i> Registrar Movimiento
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Historial de movimientos -->
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0"><i class="fas fa-history"></i> Últimos Movimientos</h6>
                    </div>
                    <div class="card-body">
                        <?php if (empty($movimientos)): ?>
                            <div class="text-center text-muted">
                                <i class="fas fa-inbox fa-2x mb-2"></i>
                                <p>No hay movimientos registrados</p>
                            </div>
                        <?php else: ?>
                            <div class="timeline">
                                <?php foreach ($movimientos as $mov): ?>
                                    <div class="mb-3 p-2 border-start border-3 border-<?php 
                                        echo $mov['tipo_movimiento'] === 'entrada' ? 'success' : 'danger'; 
                                    ?>">
                                        <div class="d-flex justify-content-between align-items-start">
                                            <div>
                                                <span class="badge bg-<?php 
                                                    echo $mov['tipo_movimiento'] === 'entrada' ? 'success' : 'danger'; 
                                                ?>">
                                                    <?php echo ucfirst($mov['tipo_movimiento']); ?>
                                                </span>
                                                <strong class="ms-2"><?php echo $mov['cantidad']; ?></strong>
                                            </div>
                                            <small class="text-muted">
                                                <?php echo date('d/m H:i', strtotime($mov['fecha_movimiento'])); ?>
                                            </small>
                                        </div>
                                        <?php if ($mov['motivo']): ?>
                                            <small class="text-muted d-block mt-1">
                                                <?php echo htmlspecialchars($mov['motivo']); ?>
                                            </small>
                                        <?php endif; ?>
                                        <?php if ($mov['usuario_nombre']): ?>
                                            <small class="text-muted d-block">
                                                Por: <?php echo htmlspecialchars($mov['usuario_nombre']); ?>
                                            </small>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Validación dinámica del motivo según el tipo de movimiento
        const tipoMovimientoSelect = document.getElementById('tipo_movimiento');
        const motivoTextarea = document.getElementById('motivo');
        const motivoRequired = document.querySelector('.motivo-required');
        const motivoHelp = document.querySelector('.motivo-help');
        const form = document.querySelector('form');
        
        // Tipos que requieren motivo obligatorio (TODOS)
        const tiposConMotivoObligatorio = ['entrada', 'salida'];
        
        function updateMotivoRequirement() {
            const tipoSeleccionado = tipoMovimientoSelect.value;
            
            if (tipoSeleccionado && tiposConMotivoObligatorio.includes(tipoSeleccionado)) {
                // Hacer obligatorio (todos los tipos)
                motivoTextarea.required = true;
                motivoRequired.style.display = 'inline';
                motivoHelp.style.display = 'block';
                motivoTextarea.classList.add('border-warning');
                motivoTextarea.placeholder = 'Motivo obligatorio para mantener trazabilidad del inventario...';
            } else if (!tipoSeleccionado) {
                // Sin tipo seleccionado
                motivoTextarea.required = false;
                motivoRequired.style.display = 'none';
                motivoHelp.style.display = 'none';
                motivoTextarea.classList.remove('border-warning');
                motivoTextarea.placeholder = 'Primero selecciona el tipo de movimiento...';
            } else {
                // Fallback (no debería llegar aquí)
                motivoTextarea.required = true;
                motivoRequired.style.display = 'inline';
                motivoHelp.style.display = 'block';
                motivoTextarea.placeholder = 'Motivo obligatorio...';
            }
        }
        
        // Ejecutar al cambiar el tipo de movimiento
        tipoMovimientoSelect.addEventListener('change', updateMotivoRequirement);
        
        // Ejecutar al cargar la página
        updateMotivoRequirement();
        
        // Validación adicional al enviar formulario
        form.addEventListener('submit', function(e) {
            const tipoSeleccionado = tipoMovimientoSelect.value;
            const motivoTexto = motivoTextarea.value.trim();
            
            if (tiposConMotivoObligatorio.includes(tipoSeleccionado)) {
                if (motivoTexto === '') {
                    e.preventDefault();
                    
                    // Mostrar alerta
                    alert('⚠️ El motivo es obligatorio para TODOS los movimientos de inventario.\n\nPor favor describe la razón del movimiento para mantener la trazabilidad completa.');
                    
                    // Enfocar el campo
                    motivoTextarea.focus();
                    motivoTextarea.classList.add('border-danger');
                    
                    return false;
                } else if (motivoTexto.length < 10) {
                    e.preventDefault();
                    
                    // Mostrar alerta para longitud mínima
                    alert('📝 El motivo debe ser más descriptivo (mínimo 10 caracteres).\n\nActual: ' + motivoTexto.length + ' caracteres.\nPor favor proporciona más detalles.');
                    
                    // Enfocar el campo
                    motivoTextarea.focus();
                    motivoTextarea.classList.add('border-warning');
                    
                    return false;
                }
            }
        });
        
        // Contador de caracteres y feedback visual
        const charCount = document.getElementById('char-count');
        const charStatus = document.getElementById('char-status');
        
        function updateCharCount() {
            const length = motivoTextarea.value.trim().length;
            const tipoSeleccionado = tipoMovimientoSelect.value;
            
            charCount.textContent = length;
            
            if (tiposConMotivoObligatorio.includes(tipoSeleccionado)) {
                if (length === 0) {
                    charStatus.innerHTML = '<i class="fas fa-exclamation-triangle text-danger"></i>';
                    motivoTextarea.classList.remove('border-success', 'border-warning');
                    motivoTextarea.classList.add('border-danger');
                } else if (length < 10) {
                    charStatus.innerHTML = '<i class="fas fa-clock text-warning"></i> Muy corto';
                    motivoTextarea.classList.remove('border-success', 'border-danger');
                    motivoTextarea.classList.add('border-warning');
                } else {
                    charStatus.innerHTML = '<i class="fas fa-check-circle text-success"></i> Suficiente';
                    motivoTextarea.classList.remove('border-warning', 'border-danger');
                    motivoTextarea.classList.add('border-success');
                }
            } else {
                charStatus.innerHTML = '';
                motivoTextarea.classList.remove('border-warning', 'border-danger', 'border-success');
            }
        }
        
        // Actualizar contador al escribir
        motivoTextarea.addEventListener('input', updateCharCount);
        
        // Actualizar contador al cambiar tipo
        tipoMovimientoSelect.addEventListener('change', updateCharCount);
        
        // Información adicional según el tipo de movimiento
        tipoMovimientoSelect.addEventListener('change', function() {
            const tipo = this.value;
            let placeholder = '';
            
            switch(tipo) {
                case 'entrada':
                    placeholder = 'Ej: Compra nueva, reposición de stock, devolución de proveedor, donación... (OBLIGATORIO - mínimo 10 caracteres)';
                    break;
                case 'salida':
                    placeholder = 'Ej: Rotura, pérdida, transferencia a otra ubicación, descarte, extravío... (OBLIGATORIO - mínimo 10 caracteres)';
                    break;
                default:
                    placeholder = 'Describe el motivo del movimiento... (OBLIGATORIO - mínimo 10 caracteres)';
            }
            
            motivoTextarea.placeholder = placeholder;
        });
    </script>
</body>
</html>


