<?php
// Verificar autenticación
$authService = new AuthService();
if (!$authService->estaAutenticado()) {
    header('Location: ' . BASE_URL . '/login');
    exit;
}

$usuario = $authService->getUsuarioActual();

// Obtener servicios
$movimientoService = new MovimientoService();
$inventarioService = new InventarioService();

// Obtener filtros de la URL
$filtros = [
    'tipo' => $_GET['tipo'] ?? '',
    'utensilio' => $_GET['utensilio'] ?? '',
    'usuario' => $_GET['usuario'] ?? '',
    'fecha_inicio' => $_GET['fecha_inicio'] ?? '',
    'fecha_fin' => $_GET['fecha_fin'] ?? '',
    'busqueda' => $_GET['busqueda'] ?? ''
];

// Paginación
$pagina = (int)($_GET['pagina'] ?? 1);
$limite = 20;

try {
    // Obtener movimientos con filtros
    $movimientos = $movimientoService->obtenerHistorial($filtros, $limite, ($pagina - 1) * $limite);
    
    // Obtener datos para filtros
    $utensilios = $inventarioService->obtenerTodos(['solo_nombres' => true]);
    
    // Obtener estadísticas rápidas
    $estadisticas = $movimientoService->obtenerEstadisticas(7); // Últimos 7 días
    
} catch (Exception $e) {
    $error = $e->getMessage();
    $movimientos = [];
    $utensilios = [];
    $estadisticas = [];
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Movimientos de Inventario - <?php echo APP_NAME; ?></title>
    <meta name="csrf-token" content="<?php echo $authService->getCSRFToken(); ?>">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="<?php echo ASSETS_URL; ?>/css/movimientos.css" rel="stylesheet">
</head>
<body>
    <!-- Navbar -->
    <?php include VIEWS_PATH . '/partials/navbar.php'; ?>

    <div class="container-fluid py-4">
        
        <!-- Breadcrumb -->
        <nav aria-label="breadcrumb" class="mb-4">
            <ol class="breadcrumb">
                <li class="breadcrumb-item">
                    <a href="<?php echo BASE_URL; ?>/dashboard">
                        <i class="fas fa-home"></i> Dashboard
                    </a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">
                    Movimientos de Inventario
                </li>
            </ol>
        </nav>

        <!-- Encabezado con acciones -->
        <div class="row mb-4">
            <div class="col-md-8">
                <h1 class="h3 mb-2">
                    <i class="fas fa-exchange-alt text-primary me-2"></i>
                    Movimientos de Inventario
                </h1>
                <p class="text-muted">Gestión de entradas y salidas de utensilios</p>
            </div>
            <div class="col-md-4 text-md-end">
                <?php if ($usuario->puedeGestionar('movimientos')): ?>
                    <div class="btn-group" role="group">
                        <button type="button" class="btn btn-success" onclick="abrirModalMovimiento('entrada')">
                            <i class="fas fa-plus me-2"></i>
                            Registrar Entrada
                        </button>
                        <button type="button" class="btn btn-danger" onclick="abrirModalMovimiento('salida')">
                            <i class="fas fa-minus me-2"></i>
                            Registrar Salida
                        </button>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Estadísticas rápidas -->
        <?php if (!empty($estadisticas)): ?>
            <div class="row mb-4">
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card stats-card border-0 shadow-sm">
                        <div class="card-body text-center">
                            <div class="stats-icon bg-success bg-opacity-10 text-success mb-2">
                                <i class="fas fa-arrow-up"></i>
                            </div>
                            <h5 class="stats-number text-success"><?php echo $estadisticas['entradas_semana'] ?? 0; ?></h5>
                            <p class="stats-label text-muted mb-0">Entradas esta semana</p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card stats-card border-0 shadow-sm">
                        <div class="card-body text-center">
                            <div class="stats-icon bg-danger bg-opacity-10 text-danger mb-2">
                                <i class="fas fa-arrow-down"></i>
                            </div>
                            <h5 class="stats-number text-danger"><?php echo $estadisticas['salidas_semana'] ?? 0; ?></h5>
                            <p class="stats-label text-muted mb-0">Salidas esta semana</p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card stats-card border-0 shadow-sm">
                        <div class="card-body text-center">
                            <div class="stats-icon bg-primary bg-opacity-10 text-primary mb-2">
                                <i class="fas fa-exchange-alt"></i>
                            </div>
                            <h5 class="stats-number text-primary"><?php echo ($estadisticas['entradas_semana'] ?? 0) + ($estadisticas['salidas_semana'] ?? 0); ?></h5>
                            <p class="stats-label text-muted mb-0">Total movimientos</p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card stats-card border-0 shadow-sm">
                        <div class="card-body text-center">
                            <div class="stats-icon bg-warning bg-opacity-10 text-warning mb-2">
                                <i class="fas fa-utensils"></i>
                            </div>
                            <h5 class="stats-number text-warning"><?php echo $estadisticas['utensilios_movidos'] ?? 0; ?></h5>
                            <p class="stats-label text-muted mb-0">Utensilios involucrados</p>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Filtros -->
        <div class="card mb-4">
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h6 class="mb-0">
                        <i class="fas fa-filter me-2"></i>
                        Filtros de búsqueda
                    </h6>
                    <button type="button" class="btn btn-sm btn-outline-secondary" id="btnLimpiarFiltros">
                        <i class="fas fa-times me-1"></i>
                        Limpiar
                    </button>
                </div>
            </div>
            <div class="card-body">
                <form method="GET" id="formFiltros" class="row g-3">
                    <div class="col-md-3">
                        <label for="busqueda" class="form-label">Búsqueda general</label>
                        <div class="input-group">
                            <span class="input-group-text">
                                <i class="fas fa-search"></i>
                            </span>
                            <input type="text" 
                                   class="form-control" 
                                   id="busqueda" 
                                   name="busqueda" 
                                   value="<?php echo htmlspecialchars($filtros['busqueda']); ?>"
                                   placeholder="Buscar en observaciones...">
                        </div>
                    </div>
                    
                    <div class="col-md-2">
                        <label for="tipo" class="form-label">Tipo</label>
                        <select class="form-select" id="tipo" name="tipo">
                            <option value="">Todos los tipos</option>
                            <option value="entrada" <?php echo $filtros['tipo'] === 'entrada' ? 'selected' : ''; ?>>
                                Entradas
                            </option>
                            <option value="salida" <?php echo $filtros['tipo'] === 'salida' ? 'selected' : ''; ?>>
                                Salidas
                            </option>
                        </select>
                    </div>
                    
                    <div class="col-md-3">
                        <label for="utensilio" class="form-label">Utensilio</label>
                        <select class="form-select" id="utensilio" name="utensilio">
                            <option value="">Todos los utensilios</option>
                            <?php foreach ($utensilios as $ut): ?>
                                <option value="<?php echo $ut->id; ?>" 
                                        <?php echo $filtros['utensilio'] == $ut->id ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($ut->nombre); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-2">
                        <label for="fecha_inicio" class="form-label">Fecha desde</label>
                        <input type="date" 
                               class="form-control" 
                               id="fecha_inicio" 
                               name="fecha_inicio" 
                               value="<?php echo htmlspecialchars($filtros['fecha_inicio']); ?>">
                    </div>
                    
                    <div class="col-md-2">
                        <label for="fecha_fin" class="form-label">Fecha hasta</label>
                        <input type="date" 
                               class="form-control" 
                               id="fecha_fin" 
                               name="fecha_fin" 
                               value="<?php echo htmlspecialchars($filtros['fecha_fin']); ?>">
                    </div>
                </form>
            </div>
        </div>

        <!-- Lista de movimientos -->
        <div class="card">
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h6 class="mb-0">
                        <i class="fas fa-list me-2"></i>
                        Historial de Movimientos
                    </h6>
                    <div class="btn-group btn-group-sm">
                        <button type="button" class="btn btn-outline-primary" onclick="exportarMovimientos()">
                            <i class="fas fa-download"></i>
                        </button>
                        <button type="button" class="btn btn-outline-secondary" onclick="actualizarLista()">
                            <i class="fas fa-sync"></i>
                        </button>
                    </div>
                </div>
            </div>
            <div class="card-body p-0">
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger m-3">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php elseif (empty($movimientos)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">No hay movimientos registrados</h5>
                        <p class="text-muted">Los movimientos de inventario aparecerán aquí</p>
                        <?php if ($usuario->puedeGestionar('movimientos')): ?>
                            <button type="button" class="btn btn-primary mt-2" onclick="abrirModalMovimiento('entrada')">
                                <i class="fas fa-plus me-2"></i>
                                Registrar primer movimiento
                            </button>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th width="120">Fecha/Hora</th>
                                    <th width="80">Tipo</th>
                                    <th>Utensilio</th>
                                    <th width="80">Cantidad</th>
                                    <th>Motivo</th>
                                    <th width="120">Usuario</th>
                                    <th width="100">Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($movimientos as $mov): ?>
                                    <tr>
                                        <td>
                                            <small class="text-muted">
                                                <?php echo date('d/m/Y', strtotime($mov->fecha_movimiento)); ?>
                                                <br>
                                                <?php echo date('H:i', strtotime($mov->fecha_movimiento)); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <?php if ($mov->tipo_movimiento === 'entrada'): ?>
                                                <span class="badge bg-success">
                                                    <i class="fas fa-plus"></i> Entrada
                                                </span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">
                                                    <i class="fas fa-minus"></i> Salida
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <a href="<?php echo BASE_URL; ?>/utensilios/ver/<?php echo $mov->id_utensilio; ?>" 
                                               class="text-decoration-none">
                                                <strong><?php echo htmlspecialchars($mov->utensilio_nombre ?? 'N/A'); ?></strong>
                                            </a>
                                            <br>
                                            <small class="text-muted"><?php echo htmlspecialchars($mov->utensilio_tipo ?? ''); ?></small>
                                        </td>
                                        <td>
                                            <strong class="<?php echo $mov->tipo_movimiento === 'entrada' ? 'text-success' : 'text-danger'; ?>">
                                                <?php echo $mov->tipo_movimiento === 'entrada' ? '+' : '-'; ?><?php echo $mov->cantidad; ?>
                                            </strong>
                                        </td>
                                        <td>
                                            <div class="motivo-cell">
                                                <strong><?php echo htmlspecialchars($mov->motivo); ?></strong>
                                                <?php if ($mov->observaciones): ?>
                                                    <br>
                                                    <small class="text-muted">
                                                        <?php echo htmlspecialchars($mov->observaciones); ?>
                                                    </small>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td>
                                            <small>
                                                <?php echo htmlspecialchars($mov->usuario_nombre ?? 'Sistema'); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button type="button" 
                                                        class="btn btn-outline-info btn-sm" 
                                                        onclick="verDetalleMovimiento(<?php echo $mov->id; ?>)"
                                                        title="Ver detalle">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <?php if ($usuario->puedeGestionar('movimientos') && 
                                                          (strtotime($mov->fecha_movimiento) > strtotime('-1 day'))): ?>
                                                    <button type="button" 
                                                            class="btn btn-outline-warning btn-sm" 
                                                            onclick="corregirMovimiento(<?php echo $mov->id; ?>)"
                                                            title="Corregir">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Paginación -->
                    <div class="card-footer bg-light">
                        <div class="d-flex justify-content-between align-items-center">
                            <small class="text-muted">
                                Mostrando <?php echo count($movimientos); ?> movimientos
                            </small>
                            <nav>
                                <ul class="pagination pagination-sm mb-0">
                                    <?php if ($pagina > 1): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?<?php echo http_build_query(array_merge($filtros, ['pagina' => $pagina - 1])); ?>">
                                                Anterior
                                            </a>
                                        </li>
                                    <?php endif; ?>
                                    
                                    <li class="page-item active">
                                        <span class="page-link"><?php echo $pagina; ?></span>
                                    </li>
                                    
                                    <?php if (count($movimientos) === $limite): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?<?php echo http_build_query(array_merge($filtros, ['pagina' => $pagina + 1])); ?>">
                                                Siguiente
                                            </a>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </nav>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Modal para registrar movimiento -->
    <div class="modal fade" id="modalMovimiento" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <span id="iconoTipoMovimiento"></span>
                        Registrar <span id="textoTipoMovimiento"></span>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="formMovimiento">
                    <div class="modal-body">
                        <input type="hidden" id="tipoMovimiento" name="tipo_movimiento">
                        <input type="hidden" name="csrf_token" value="<?php echo $authService->getCSRFToken(); ?>">
                        
                        <div class="row">
                            <div class="col-md-8 mb-3">
                                <label for="utensilioMovimiento" class="form-label">
                                    Utensilio <span class="text-danger">*</span>
                                </label>
                                <select class="form-select" id="utensilioMovimiento" name="utensilio_id" required>
                                    <option value="">Seleccionar utensilio...</option>
                                    <?php foreach ($utensilios as $ut): ?>
                                        <option value="<?php echo $ut->id; ?>" 
                                                data-disponible="<?php echo $ut->cantidad_disponible ?? 0; ?>"
                                                data-total="<?php echo $ut->cantidad_total ?? 0; ?>"
                                                data-ubicacion="<?php echo htmlspecialchars($ut->ubicacion ?? ''); ?>">
                                            <?php echo htmlspecialchars($ut->nombre); ?> 
                                            (Disponible: <?php echo $ut->cantidad_disponible ?? 0; ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-4 mb-3">
                                <label for="cantidadMovimiento" class="form-label">
                                    Cantidad <span class="text-danger">*</span>
                                </label>
                                <input type="number" 
                                       class="form-control" 
                                       id="cantidadMovimiento" 
                                       name="cantidad" 
                                       required
                                       min="1"
                                       placeholder="0">
                                <div class="form-text" id="ayudaCantidad"></div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="motivoMovimiento" class="form-label">
                                    Motivo <span class="text-danger">*</span>
                                </label>
                                <select class="form-select" id="motivoMovimiento" name="motivo" required>
                                    <option value="">Seleccionar motivo...</option>
                                </select>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label for="ubicacionDestino" class="form-label">
                                    Ubicación destino
                                </label>
                                <input type="text" 
                                       class="form-control" 
                                       id="ubicacionDestino" 
                                       name="ubicacion_destino" 
                                       placeholder="Ubicación de destino (opcional)">
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="observacionesMovimiento" class="form-label">Observaciones</label>
                            <textarea class="form-control" 
                                      id="observacionesMovimiento" 
                                      name="observaciones" 
                                      rows="3" 
                                      placeholder="Detalles adicionales del movimiento..."></textarea>
                        </div>
                        
                        <!-- Información del utensilio seleccionado -->
                        <div id="infoUtensilio" class="alert alert-info d-none">
                            <h6><i class="fas fa-info-circle me-2"></i>Información del utensilio</h6>
                            <div id="detalleUtensilio"></div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn" id="btnConfirmarMovimiento">
                            <span id="textoBotonMovimiento"></span>
                            <span id="spinnerMovimiento" class="spinner-border spinner-border-sm ms-2 d-none"></span>
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal para ver detalle de movimiento -->
    <div class="modal fade" id="modalDetalleMovimiento" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-info-circle me-2"></i>
                        Detalle del Movimiento
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="contenidoDetalleMovimiento">
                    <!-- Contenido cargado dinámicamente -->
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const BASE_URL = '<?php echo BASE_URL; ?>';
        const USUARIO_PUEDE_GESTIONAR = <?php echo $usuario->puedeGestionar('movimientos') ? 'true' : 'false'; ?>;
    </script>
    <script src="<?php echo ASSETS_URL; ?>/js/movimientos.js"></script>
</body>
</html>