<?php
/**
 * PÁGINA DE AGREGAR USUARIO
 */

// Verificar autenticación y permisos de administrador
if (!isset($_SESSION['usuario_id']) || ($_SESSION['rol'] ?? '') !== 'administrador') {
    header('Location: /app-web-Enciso/public/usuarios');
    exit;
}

// Obtener configuración de base de datos
require_once __DIR__ . '/../../config/config.php';

try {
    $pdo = new PDO($dsn, $username, $password, $options);
    
    // Si es POST, procesar creación de usuario
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $nombre = trim($_POST['nombre'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $rol = trim($_POST['rol'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirmar_password = $_POST['confirmar_password'] ?? '';
        
        // Validaciones básicas
        if (empty($nombre) || empty($email) || empty($rol) || empty($password)) {
            $error = "Todos los campos son obligatorios";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = "El email no es válido";
        } elseif (strlen($password) < 6) {
            $error = "La contraseña debe tener al menos 6 caracteres";
        } elseif ($password !== $confirmar_password) {
            $error = "Las contraseñas no coinciden";
        } else {
            // Verificar que el email no esté en uso
            $stmt = $pdo->prepare("SELECT id FROM usuarios WHERE email = ?");
            $stmt->execute([$email]);
            
            if ($stmt->rowCount() > 0) {
                $error = "El email ya está siendo usado por otro usuario";
            } else {
                // Crear usuario
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("
                    INSERT INTO usuarios (nombre, email, password_hash, rol, activo, fecha_creacion) 
                    VALUES (?, ?, ?, ?, 1, NOW())
                ");
                
                if ($stmt->execute([$nombre, $email, $hashed_password, $rol])) {
                    $_SESSION['success_message'] = "Usuario creado exitosamente";
                    header('Location: /app-web-Enciso/public/usuarios');
                    exit;
                } else {
                    $error = "Error al crear el usuario";
                }
            }
        }
    }
    
} catch (Exception $e) {
    $error = "Error de base de datos: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Nuevo Usuario - Sistema de Gestión</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* === TEMA CAYEJERO GAMING === */
        * {
            background-color: transparent !important;
        }
        
        body, html {
            background: #0a0a0a !important;
            min-height: 100vh;
            color: #ffffff !important;
        }
        
        .card {
            background: #1a1a1a !important;
            border: 2px solid rgba(255, 140, 0, 0.3) !important;
            border-radius: 15px !important;
            box-shadow: 0 8px 32px rgba(255, 140, 0, 0.1) !important;
        }
        
        .card-header {
            background: linear-gradient(135deg, #FF8C00, #FFD700) !important;
            color: #000000 !important;
            font-weight: 800 !important;
            border-radius: 13px 13px 0 0 !important;
            text-shadow: 1px 1px 2px rgba(255, 255, 255, 0.3);
        }
        
        h1, h2, h3, h4, h5, h6 {
            color: #FFD700 !important;
            font-weight: 800 !important;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.8);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #FF8C00, #FFD700) !important;
            border: none !important;
            color: #000000 !important;
            font-weight: 700 !important;
            padding: 12px 30px !important;
            transition: all 0.3s ease !important;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px) !important;
            box-shadow: 0 6px 20px rgba(255, 140, 0, 0.4) !important;
            background: linear-gradient(135deg, #FFD700, #FF8C00) !important;
        }
        
        .btn-secondary {
            background: #6c757d !important;
            color: #ffffff !important;
            font-weight: 700 !important;
            border: none !important;
        }
        
        .form-control, .form-select {
            background: #2d2d2d !important;
            border: 2px solid rgba(255, 140, 0, 0.5) !important;
            color: #ffffff !important;
            font-weight: 600 !important;
            border-radius: 8px !important;
        }
        
        .form-control:focus, .form-select:focus {
            background: #2d2d2d !important;
            border-color: #FFD700 !important;
            box-shadow: 0 0 0 0.2rem rgba(255, 215, 0, 0.25) !important;
            color: #ffffff !important;
        }
        
        .form-control::placeholder {
            color: #999999 !important;
        }
        
        .form-label {
            color: #FFD700 !important;
            font-weight: 700 !important;
        }
        
        .alert-danger {
            background: rgba(220, 53, 69, 0.2) !important;
            border: 2px solid #dc3545 !important;
            color: #ffffff !important;
        }
        
        .text-primary {
            color: #FFD700 !important;
        }
        
        .password-strength {
            height: 5px;
            border-radius: 3px;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <?php include __DIR__ . '/../partials/navbar-simple.php'; ?>
    
    <div class="container py-4">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header bg-gradient text-white text-center py-4" style="background: linear-gradient(45deg, #667eea, #764ba2); border-radius: 20px 20px 0 0;">
                        <h3 class="mb-0">
                            <i class="fas fa-user-plus"></i> 
                            Crear Nuevo Usuario
                        </h3>
                    </div>
                    <div class="card-body p-4">
                        <?php if (isset($error)): ?>
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <i class="fas fa-exclamation-circle me-2"></i>
                                <?php echo htmlspecialchars($error); ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <form method="POST" id="userForm">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="nombre" class="form-label">
                                            <i class="fas fa-user text-primary"></i> Nombre Completo
                                        </label>
                                        <input type="text" class="form-control" id="nombre" name="nombre" 
                                               value="<?php echo htmlspecialchars($_POST['nombre'] ?? ''); ?>" 
                                               required maxlength="100" placeholder="Ej: Juan Pérez García">
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="email" class="form-label">
                                            <i class="fas fa-envelope text-primary"></i> Correo Electrónico
                                        </label>
                                        <input type="email" class="form-control" id="email" name="email" 
                                               value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" 
                                               required maxlength="150" placeholder="usuario@empresa.com">
                                    </div>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-12">
                                    <div class="mb-3">
                                        <label for="rol" class="form-label">
                                            <i class="fas fa-user-tag text-primary"></i> Rol del Usuario
                                        </label>
                                        <select class="form-control" id="rol" name="rol" required>
                                            <option value="">Seleccionar rol...</option>
                                            <option value="administrador" <?php echo ($_POST['rol'] ?? '') === 'administrador' ? 'selected' : ''; ?>>
                                                <i class="fas fa-crown"></i> Administrador - Control total del sistema
                                            </option>
                                            <option value="encargado" <?php echo ($_POST['rol'] ?? '') === 'encargado' ? 'selected' : ''; ?>>
                                                <i class="fas fa-user-tie"></i> Encargado - Gestión de inventario y reportes
                                            </option>
                                            <option value="auditor" <?php echo ($_POST['rol'] ?? '') === 'auditor' ? 'selected' : ''; ?>>
                                                <i class="fas fa-search"></i> Auditor - Solo visualización y reportes
                                            </option>
                                        </select>
                                        <small class="text-muted">
                                            <strong>Administrador:</strong> Acceso completo | 
                                            <strong>Encargado:</strong> Gestión operativa | 
                                            <strong>Auditor:</strong> Solo consulta
                                        </small>
                                    </div>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="password" class="form-label">
                                            <i class="fas fa-key text-primary"></i> Contraseña
                                        </label>
                                        <div class="input-group">
                                            <input type="password" class="form-control" id="password" name="password" 
                                                   required minlength="6" oninput="checkPasswordStrength()">
                                            <button class="btn btn-outline-secondary" type="button" onclick="togglePassword('password')">
                                                <i class="fas fa-eye" id="password_icon"></i>
                                            </button>
                                        </div>
                                        <div class="password-strength bg-light" id="password_strength"></div>
                                        <small class="text-muted">Mínimo 6 caracteres</small>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="confirmar_password" class="form-label">
                                            <i class="fas fa-check-circle text-success"></i> Confirmar Contraseña
                                        </label>
                                        <div class="input-group">
                                            <input type="password" class="form-control" id="confirmar_password" name="confirmar_password" 
                                                   required oninput="checkPasswordMatch()">
                                            <button class="btn btn-outline-secondary" type="button" onclick="togglePassword('confirmar_password')">
                                                <i class="fas fa-eye" id="confirmar_password_icon"></i>
                                            </button>
                                        </div>
                                        <small id="password_match" class="text-muted"></small>
                                    </div>
                                </div>
                            </div>

                            <div class="d-flex justify-content-between align-items-center mt-4">
                                <a href="/app-web-Enciso/public/usuarios" class="btn btn-outline-secondary">
                                    <i class="fas fa-arrow-left"></i> Cancelar
                                </a>
                                <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                                    <i class="fas fa-user-plus"></i> Crear Usuario
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Información sobre roles -->
                <div class="card mt-4">
                    <div class="card-body">
                        <h6 class="text-muted mb-3">
                            <i class="fas fa-info-circle text-primary"></i> Información sobre Roles
                        </h6>
                        <div class="row">
                            <div class="col-md-4">
                                <div class="p-3 bg-danger bg-opacity-10 rounded">
                                    <h6 class="text-danger">
                                        <i class="fas fa-crown"></i> Administrador
                                    </h6>
                                    <ul class="text-sm mb-0">
                                        <li>Gestión completa de usuarios</li>
                                        <li>Acceso a todas las funcionalidades</li>
                                        <li>Configuración del sistema</li>
                                    </ul>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="p-3 bg-primary bg-opacity-10 rounded">
                                    <h6 class="text-primary">
                                        <i class="fas fa-user-tie"></i> Encargado
                                    </h6>
                                    <ul class="text-sm mb-0">
                                        <li>Gestión de inventario completa</li>
                                        <li>Creación y visualización de reportes</li>
                                        <li>Movimientos de entrada y salida</li>
                                    </ul>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="p-3 bg-info bg-opacity-10 rounded">
                                    <h6 class="text-info">
                                        <i class="fas fa-search"></i> Auditor
                                    </h6>
                                    <ul class="text-sm mb-0">
                                        <li>Solo visualización de datos</li>
                                        <li>Acceso a reportes y estadísticas</li>
                                        <li>No puede modificar inventario</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = document.getElementById(fieldId + '_icon');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        function checkPasswordStrength() {
            const password = document.getElementById('password').value;
            const strengthBar = document.getElementById('password_strength');
            let strength = 0;

            if (password.length >= 6) strength += 1;
            if (password.length >= 8) strength += 1;
            if (/[a-z]/.test(password) && /[A-Z]/.test(password)) strength += 1;
            if (/\d/.test(password)) strength += 1;
            if (/[^A-Za-z0-9]/.test(password)) strength += 1;

            const colors = ['#e74c3c', '#e67e22', '#f39c12', '#27ae60', '#2ecc71'];
            const widths = [20, 40, 60, 80, 100];
            
            strengthBar.style.backgroundColor = colors[strength - 1] || '#e74c3c';
            strengthBar.style.width = (widths[strength - 1] || 0) + '%';
            
            checkPasswordMatch();
        }

        function checkPasswordMatch() {
            const password = document.getElementById('password').value;
            const confirm = document.getElementById('confirmar_password').value;
            const matchText = document.getElementById('password_match');
            const submitBtn = document.getElementById('submitBtn');
            
            if (confirm.length === 0) {
                matchText.textContent = '';
                submitBtn.disabled = true;
                return;
            }
            
            if (password === confirm && password.length >= 6) {
                matchText.textContent = '✓ Las contraseñas coinciden';
                matchText.className = 'text-success';
                submitBtn.disabled = false;
            } else {
                matchText.textContent = '✗ Las contraseñas no coinciden';
                matchText.className = 'text-danger';
                submitBtn.disabled = true;
            }
        }

        // Validar formulario completo
        function validateForm() {
            const nombre = document.getElementById('nombre').value.trim();
            const email = document.getElementById('email').value.trim();
            const rol = document.getElementById('rol').value;
            const password = document.getElementById('password').value;
            const confirm = document.getElementById('confirmar_password').value;
            const submitBtn = document.getElementById('submitBtn');
            
            const isValid = nombre && email && rol && password.length >= 6 && password === confirm;
            submitBtn.disabled = !isValid;
        }

        // Escuchadores de eventos
        document.getElementById('nombre').addEventListener('input', validateForm);
        document.getElementById('email').addEventListener('input', validateForm);
        document.getElementById('rol').addEventListener('change', validateForm);
        document.getElementById('password').addEventListener('input', function() {
            checkPasswordStrength();
            validateForm();
        });
        document.getElementById('confirmar_password').addEventListener('input', function() {
            checkPasswordMatch();
            validateForm();
        });
    </script>
</body>
</html>

