<?php
/**
 * PÁGINA DE CAMBIO DE CONTRASEÑA
 */

// Verificar autenticación
if (!isset($_SESSION['usuario_id'])) {
    header('Location: /app-web-Enciso/public/login');
    exit;
}

// Obtener configuración de base de datos
require_once __DIR__ . '/../../config/config.php';

// Obtener ID del usuario (puede ser el actual o uno específico)
$usuario_id = isset($_GET['id']) ? intval($_GET['id']) : $_SESSION['usuario_id'];

// Solo administradores pueden cambiar contraseña de otros usuarios
$es_admin = ($_SESSION['rol'] ?? '') === 'administrador';
if ($usuario_id !== $_SESSION['usuario_id'] && !$es_admin) {
    header('Location: /app-web-Enciso/public/usuarios');
    exit;
}

try {
    $pdo = new PDO($dsn, $username, $password, $options);
    
    // Si es POST, procesar cambio de contraseña
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $password_actual = $_POST['password_actual'] ?? '';
        $nueva_password = $_POST['nueva_password'] ?? '';
        $confirmar_password = $_POST['confirmar_password'] ?? '';
        
        // Validaciones
        if ($usuario_id === $_SESSION['usuario_id'] && empty($password_actual)) {
            $error = "Debe ingresar su contraseña actual";
        } elseif (strlen($nueva_password) < 6) {
            $error = "La nueva contraseña debe tener al menos 6 caracteres";
        } elseif ($nueva_password !== $confirmar_password) {
            $error = "Las contraseñas no coinciden";
        } else {
            // Si está cambiando su propia contraseña, verificar la actual
            if ($usuario_id === $_SESSION['usuario_id']) {
                $stmt = $pdo->prepare("SELECT password_hash FROM usuarios WHERE id = ?");
                $stmt->execute([$usuario_id]);
                $user = $stmt->fetch();
                
                if (!$user || !password_verify($password_actual, $user['password_hash'])) {
                    $error = "La contraseña actual es incorrecta";
                }
            }
            
            if (!isset($error)) {
                // Actualizar contraseña
                $hashed_password = password_hash($nueva_password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("
                    UPDATE usuarios 
                    SET password_hash = ?, fecha_actualizacion = NOW() 
                    WHERE id = ?
                ");
                
                if ($stmt->execute([$hashed_password, $usuario_id])) {
                    $_SESSION['success_message'] = "Contraseña actualizada exitosamente";
                    header('Location: /app-web-Enciso/public/usuarios');
                    exit;
                } else {
                    $error = "Error al actualizar la contraseña";
                }
            }
        }
    }
    
    // Obtener datos del usuario
    $stmt = $pdo->prepare("SELECT nombre, email FROM usuarios WHERE id = ?");
    $stmt->execute([$usuario_id]);
    $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$usuario) {
        header('Location: /app-web-Enciso/public/usuarios');
        exit;
    }
    
} catch (Exception $e) {
    $error = "Error de base de datos: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cambiar Contraseña - Sistema de Gestión</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* === TEMA CAYEJERO GAMING === */
        * {
            background-color: transparent !important;
        }
        
        body, html {
            background: #0a0a0a !important;
            min-height: 100vh;
            color: #ffffff !important;
        }
        
        .card {
            background: #1a1a1a !important;
            border: 2px solid rgba(255, 140, 0, 0.3) !important;
            border-radius: 15px !important;
            box-shadow: 0 8px 32px rgba(255, 140, 0, 0.1) !important;
        }
        
        .card-header {
            background: linear-gradient(135deg, #FF8C00, #FFD700) !important;
            color: #000000 !important;
            font-weight: 800 !important;
            border-radius: 13px 13px 0 0 !important;
            text-shadow: 1px 1px 2px rgba(255, 255, 255, 0.3);
        }
        
        h1, h2, h3, h4, h5, h6 {
            color: #FFD700 !important;
            font-weight: 800 !important;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.8);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #FF8C00, #FFD700) !important;
            border: none !important;
            color: #000000 !important;
            font-weight: 700 !important;
            padding: 12px 30px !important;
            transition: all 0.3s ease !important;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px) !important;
            box-shadow: 0 6px 20px rgba(255, 140, 0, 0.4) !important;
            background: linear-gradient(135deg, #FFD700, #FF8C00) !important;
        }
        
        .btn-secondary {
            background: #6c757d !important;
            color: #ffffff !important;
            font-weight: 700 !important;
            border: none !important;
        }
        
        .form-control {
            background: #2d2d2d !important;
            border: 2px solid rgba(255, 140, 0, 0.5) !important;
            color: #ffffff !important;
            font-weight: 600 !important;
            border-radius: 8px !important;
            padding: 12px 20px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            background: #2d2d2d !important;
            border-color: #FFD700 !important;
            box-shadow: 0 0 0 0.2rem rgba(255, 215, 0, 0.25) !important;
            color: #ffffff !important;
        }
        
        .form-control::placeholder {
            color: #999999 !important;
        }
        
        .form-label {
            color: #FFD700 !important;
            font-weight: 700 !important;
        }
        
        .alert-danger {
            background: rgba(220, 53, 69, 0.2) !important;
            border: 2px solid #dc3545 !important;
            color: #ffffff !important;
        }
        
        .alert-success {
            background: rgba(40, 167, 69, 0.2) !important;
            border: 2px solid #28a745 !important;
            color: #ffffff !important;
        }
        
        .text-primary {
            color: #FFD700 !important;
        }
        
        .password-strength {
            height: 5px;
            border-radius: 3px;
            margin-top: 5px;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .password-strength {
            height: 5px;
            border-radius: 3px;
            margin-top: 5px;
        }
        
        /* === TEXTOS Y LISTAS === */
        .text-sm {
            color: #ffffff !important;
            font-weight: 600 !important;
        }
        
        ul {
            color: #ffffff !important;
        }
        
        ul li {
            color: #ffffff !important;
            font-weight: 600 !important;
            margin-bottom: 8px !important;
        }
        
        h6, h6.text-muted {
            color: #FFD700 !important;
            font-weight: 800 !important;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.8);
        }
        
        h6 .text-primary, h6.text-muted .text-primary {
            color: #FF8C00 !important;
        }
        
        h6 i, h6.text-muted i {
            color: #FF8C00 !important;
        }
        
        small {
            color: #e0e0e0 !important;
            font-weight: 600 !important;
        }
        
        .opacity-75 {
            color: #e0e0e0 !important;
        }
        
        .btn-outline-secondary {
            background: #6c757d !important;
            border: 2px solid #6c757d !important;
            color: #ffffff !important;
            font-weight: 700 !important;
        }
        
        .btn-outline-secondary:hover {
            background: #5a6268 !important;
            border-color: #5a6268 !important;
        }
        
        .input-group .btn-outline-secondary {
            background: #2d2d2d !important;
            border: 2px solid rgba(255, 140, 0, 0.5) !important;
            color: #FFD700 !important;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <?php include __DIR__ . '/../partials/navbar-simple.php'; ?>
    
    <div class="container py-4">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header bg-gradient text-white text-center py-4" style="background: linear-gradient(45deg, #667eea, #764ba2); border-radius: 20px 20px 0 0;">
                        <h3 class="mb-0">
                            <i class="fas fa-key"></i> 
                            Cambiar Contraseña
                        </h3>
                        <p class="mb-0 mt-2 opacity-75">
                            <?php echo htmlspecialchars($usuario['nombre']); ?>
                        </p>
                    </div>
                    <div class="card-body p-4">
                        <?php if (isset($error)): ?>
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <i class="fas fa-exclamation-circle me-2"></i>
                                <?php echo htmlspecialchars($error); ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <form method="POST" id="passwordForm">
                            <?php if ($usuario_id === $_SESSION['usuario_id']): ?>
                            <div class="mb-3">
                                <label for="password_actual" class="form-label">
                                    <i class="fas fa-lock text-warning"></i> Contraseña Actual
                                </label>
                                <div class="input-group">
                                    <input type="password" class="form-control" id="password_actual" name="password_actual" required>
                                    <button class="btn btn-outline-secondary" type="button" onclick="togglePassword('password_actual')">
                                        <i class="fas fa-eye" id="password_actual_icon"></i>
                                    </button>
                                </div>
                            </div>
                            <?php endif; ?>

                            <div class="mb-3">
                                <label for="nueva_password" class="form-label">
                                    <i class="fas fa-key text-primary"></i> Nueva Contraseña
                                </label>
                                <div class="input-group">
                                    <input type="password" class="form-control" id="nueva_password" name="nueva_password" 
                                           required minlength="6" oninput="checkPasswordStrength()">
                                    <button class="btn btn-outline-secondary" type="button" onclick="togglePassword('nueva_password')">
                                        <i class="fas fa-eye" id="nueva_password_icon"></i>
                                    </button>
                                </div>
                                <div class="password-strength bg-light" id="password_strength"></div>
                                <small class="text-muted">Mínimo 6 caracteres</small>
                            </div>

                            <div class="mb-3">
                                <label for="confirmar_password" class="form-label">
                                    <i class="fas fa-check-circle text-success"></i> Confirmar Contraseña
                                </label>
                                <div class="input-group">
                                    <input type="password" class="form-control" id="confirmar_password" name="confirmar_password" 
                                           required oninput="checkPasswordMatch()">
                                    <button class="btn btn-outline-secondary" type="button" onclick="togglePassword('confirmar_password')">
                                        <i class="fas fa-eye" id="confirmar_password_icon"></i>
                                    </button>
                                </div>
                                <small id="password_match" class="text-muted"></small>
                            </div>

                            <div class="d-flex justify-content-between align-items-center mt-4">
                                <a href="/app-web-Enciso/public/usuarios" class="btn btn-outline-secondary">
                                    <i class="fas fa-arrow-left"></i> Cancelar
                                </a>
                                <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                                    <i class="fas fa-save"></i> Cambiar Contraseña
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Consejos de seguridad -->
                <div class="card mt-4">
                    <div class="card-body">
                        <h6 class="text-muted mb-3">
                            <i class="fas fa-shield-alt text-primary"></i> Consejos de Seguridad
                        </h6>
                        <ul class="text-sm text-muted mb-0">
                            <li>Use una combinación de letras mayúsculas y minúsculas</li>
                            <li>Incluya números y símbolos especiales</li>
                            <li>Evite usar información personal fácil de adivinar</li>
                            <li>No reutilice contraseñas de otras cuentas</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = document.getElementById(fieldId + '_icon');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        function checkPasswordStrength() {
            const password = document.getElementById('nueva_password').value;
            const strengthBar = document.getElementById('password_strength');
            let strength = 0;

            if (password.length >= 6) strength += 1;
            if (password.length >= 8) strength += 1;
            if (/[a-z]/.test(password) && /[A-Z]/.test(password)) strength += 1;
            if (/\d/.test(password)) strength += 1;
            if (/[^A-Za-z0-9]/.test(password)) strength += 1;

            const colors = ['#e74c3c', '#e67e22', '#f39c12', '#27ae60', '#2ecc71'];
            const widths = [20, 40, 60, 80, 100];
            
            strengthBar.style.backgroundColor = colors[strength - 1] || '#e74c3c';
            strengthBar.style.width = (widths[strength - 1] || 0) + '%';
            
            checkPasswordMatch();
        }

        function checkPasswordMatch() {
            const password = document.getElementById('nueva_password').value;
            const confirm = document.getElementById('confirmar_password').value;
            const matchText = document.getElementById('password_match');
            const submitBtn = document.getElementById('submitBtn');
            
            if (confirm.length === 0) {
                matchText.textContent = '';
                submitBtn.disabled = true;
                return;
            }
            
            if (password === confirm && password.length >= 6) {
                matchText.textContent = '✓ Las contraseñas coinciden';
                matchText.className = 'text-success';
                submitBtn.disabled = false;
            } else {
                matchText.textContent = '✗ Las contraseñas no coinciden';
                matchText.className = 'text-danger';
                submitBtn.disabled = true;
            }
        }

        // Validar en tiempo real
        document.getElementById('nueva_password').addEventListener('input', checkPasswordStrength);
        document.getElementById('confirmar_password').addEventListener('input', checkPasswordMatch);
    </script>
</body>
</html>

