<?php
// Verificar autenticación y permisos
$authService = new AuthService();
$authService->requierePermiso('utensilios_crear');
$usuario = $authService->getUsuarioActual();

// Obtener servicios
$inventarioService = new InventarioService();

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Validar token CSRF
        if (!$authService->verificarCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Token de seguridad inválido');
        }
        
        $datos = [
            'nombre' => $_POST['nombre'] ?? '',
            'tipo' => $_POST['tipo'] ?? '',
            'descripcion' => $_POST['descripcion'] ?? '',
            'cantidad_total' => (int)($_POST['cantidad_total'] ?? 0),
            'cantidad_disponible' => (int)($_POST['cantidad_disponible'] ?? 0),
            'cantidad_minima' => (int)($_POST['cantidad_minima'] ?? 1),
            'estado' => $_POST['estado'] ?? 'bueno',
            'ubicacion' => $_POST['ubicacion'] ?? '',
            'precio_unitario' => (float)($_POST['precio_unitario'] ?? 0)
        ];
        
        $id = $inventarioService->crear($datos);
        
        $_SESSION['success'] = 'Utensilio creado exitosamente';
        header('Location: ' . BASE_URL . '/utensilios/ver/' . $id);
        exit;
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Obtener ubicaciones para el select
try {
    $ubicaciones = $inventarioService->obtenerUbicaciones();
} catch (Exception $e) {
    $ubicaciones = [];
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Agregar Utensilio - <?php echo APP_NAME; ?></title>
    <meta name="csrf-token" content="<?php echo $authService->getCSRFToken(); ?>">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="<?php echo ASSETS_URL; ?>/css/form-utensilio.css" rel="stylesheet">
</head>
<body>
    <!-- Navbar -->
    <?php include VIEWS_PATH . '/partials/navbar.php'; ?>

    <div class="container py-4">
        
        <!-- Breadcrumb -->
        <nav aria-label="breadcrumb" class="mb-4">
            <ol class="breadcrumb">
                <li class="breadcrumb-item">
                    <a href="<?php echo BASE_URL; ?>/dashboard">
                        <i class="fas fa-home"></i> Dashboard
                    </a>
                </li>
                <li class="breadcrumb-item">
                    <a href="<?php echo BASE_URL; ?>/utensilios">Utensilios</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Agregar Utensilio</li>
            </ol>
        </nav>

        <!-- Encabezado -->
        <div class="row mb-4">
            <div class="col">
                <h1 class="h3 mb-0">
                    <i class="fas fa-plus text-primary me-2"></i>
                    Agregar Nuevo Utensilio
                </h1>
                <p class="text-muted">Complete la información del utensilio para agregarlo al inventario</p>
            </div>
        </div>

        <div class="row justify-content-center">
            <div class="col-lg-8">
                
                <!-- Mensaje de error -->
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo htmlspecialchars($error); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Formulario -->
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">
                            <i class="fas fa-utensils me-2"></i>
                            Información del Utensilio
                        </h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="<?php echo BASE_URL; ?>/utensilios/crear" id="formUtensilio" novalidate>
                            <input type="hidden" name="csrf_token" value="<?php echo $authService->getCSRFToken(); ?>">
                            
                            <!-- Información básica -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h6 class="border-bottom pb-2 mb-3">
                                        <i class="fas fa-info-circle text-primary me-2"></i>
                                        Información Básica
                                    </h6>
                                </div>
                                
                                <div class="col-md-8 mb-3">
                                    <label for="nombre" class="form-label">
                                        Nombre del Utensilio <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" 
                                           class="form-control" 
                                           id="nombre" 
                                           name="nombre" 
                                           required
                                           maxlength="100"
                                           value="<?php echo htmlspecialchars($_POST['nombre'] ?? ''); ?>"
                                           placeholder="Ej: Cuchillo Chef 20cm">
                                    <div class="invalid-feedback">
                                        El nombre es requerido y debe tener máximo 100 caracteres
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="tipo" class="form-label">
                                        Tipo <span class="text-danger">*</span>
                                    </label>
                                    <select class="form-select" id="tipo" name="tipo" required>
                                        <option value="">Seleccionar tipo</option>
                                        <?php foreach (Utensilio::getTipos() as $key => $value): ?>
                                            <option value="<?php echo $key; ?>" 
                                                    <?php echo ($_POST['tipo'] ?? '') === $key ? 'selected' : ''; ?>>
                                                <?php echo $value; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <div class="invalid-feedback">
                                        Debe seleccionar un tipo de utensilio
                                    </div>
                                </div>
                                
                                <div class="col-12 mb-3">
                                    <label for="descripcion" class="form-label">Descripción</label>
                                    <textarea class="form-control" 
                                              id="descripcion" 
                                              name="descripcion" 
                                              rows="3"
                                              placeholder="Descripción detallada del utensilio (opcional)"><?php echo htmlspecialchars($_POST['descripcion'] ?? ''); ?></textarea>
                                    <div class="form-text">Incluya características importantes, marca, material, etc.</div>
                                </div>
                            </div>

                            <!-- Inventario -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h6 class="border-bottom pb-2 mb-3">
                                        <i class="fas fa-boxes text-primary me-2"></i>
                                        Control de Inventario
                                    </h6>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="cantidad_total" class="form-label">
                                        Cantidad Total <span class="text-danger">*</span>
                                    </label>
                                    <input type="number" 
                                           class="form-control" 
                                           id="cantidad_total" 
                                           name="cantidad_total" 
                                           required
                                           min="0"
                                           value="<?php echo htmlspecialchars($_POST['cantidad_total'] ?? ''); ?>"
                                           placeholder="0">
                                    <div class="invalid-feedback">
                                        La cantidad total debe ser un número mayor o igual a 0
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="cantidad_disponible" class="form-label">
                                        Cantidad Disponible <span class="text-danger">*</span>
                                    </label>
                                    <input type="number" 
                                           class="form-control" 
                                           id="cantidad_disponible" 
                                           name="cantidad_disponible" 
                                           required
                                           min="0"
                                           value="<?php echo htmlspecialchars($_POST['cantidad_disponible'] ?? ''); ?>"
                                           placeholder="0">
                                    <div class="invalid-feedback">
                                        La cantidad disponible debe ser un número mayor o igual a 0
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="cantidad_minima" class="form-label">
                                        Stock Mínimo <span class="text-danger">*</span>
                                    </label>
                                    <input type="number" 
                                           class="form-control" 
                                           id="cantidad_minima" 
                                           name="cantidad_minima" 
                                           required
                                           min="1"
                                           value="<?php echo htmlspecialchars($_POST['cantidad_minima'] ?? '1'); ?>"
                                           placeholder="1">
                                    <div class="invalid-feedback">
                                        El stock mínimo debe ser al menos 1
                                    </div>
                                    <div class="form-text">Cantidad que activará alertas de stock bajo</div>
                                </div>
                            </div>

                            <!-- Estado y ubicación -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h6 class="border-bottom pb-2 mb-3">
                                        <i class="fas fa-map-marker-alt text-primary me-2"></i>
                                        Estado y Ubicación
                                    </h6>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="estado" class="form-label">
                                        Estado del Utensilio <span class="text-danger">*</span>
                                    </label>
                                    <select class="form-select" id="estado" name="estado" required>
                                        <?php foreach (Utensilio::getEstados() as $key => $value): ?>
                                            <option value="<?php echo $key; ?>" 
                                                    <?php echo ($_POST['estado'] ?? 'bueno') === $key ? 'selected' : ''; ?>>
                                                <?php echo $value; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <div class="invalid-feedback">
                                        Debe seleccionar un estado
                                    </div>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="ubicacion" class="form-label">
                                        Ubicación <span class="text-danger">*</span>
                                    </label>
                                    <div class="input-group">
                                        <input type="text" 
                                               class="form-control" 
                                               id="ubicacion" 
                                               name="ubicacion" 
                                               required
                                               maxlength="100"
                                               list="ubicacionesExistentes"
                                               value="<?php echo htmlspecialchars($_POST['ubicacion'] ?? ''); ?>"
                                               placeholder="Ej: Estación de Corte A">
                                        <button type="button" class="btn btn-outline-secondary" id="btnUbicacionActual">
                                            <i class="fas fa-crosshairs"></i>
                                        </button>
                                    </div>
                                    <datalist id="ubicacionesExistentes">
                                        <?php foreach ($ubicaciones as $ub): ?>
                                            <option value="<?php echo htmlspecialchars($ub); ?>">
                                        <?php endforeach; ?>
                                    </datalist>
                                    <div class="invalid-feedback">
                                        La ubicación es requerida
                                    </div>
                                </div>
                            </div>

                            <!-- Información financiera -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h6 class="border-bottom pb-2 mb-3">
                                        <i class="fas fa-dollar-sign text-primary me-2"></i>
                                        Información Financiera
                                    </h6>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="precio_unitario" class="form-label">Precio Unitario</label>
                                    <div class="input-group">
                                        <span class="input-group-text">$</span>
                                        <input type="number" 
                                               class="form-control" 
                                               id="precio_unitario" 
                                               name="precio_unitario" 
                                               step="0.01"
                                               min="0"
                                               value="<?php echo htmlspecialchars($_POST['precio_unitario'] ?? ''); ?>"
                                               placeholder="0.00">
                                    </div>
                                    <div class="form-text">Precio individual del utensilio (opcional)</div>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Valor Total Estimado</label>
                                    <div class="form-control-plaintext fw-bold text-success" id="valorTotal">
                                        $0.00
                                    </div>
                                    <div class="form-text">Cantidad total × Precio unitario</div>
                                </div>
                            </div>

                            <!-- Botones de acción -->
                            <div class="row">
                                <div class="col-12">
                                    <hr class="my-4">
                                    <div class="d-flex justify-content-between">
                                        <a href="<?php echo BASE_URL; ?>/utensilios" class="btn btn-outline-secondary">
                                            <i class="fas fa-arrow-left me-2"></i>
                                            Volver a la Lista
                                        </a>
                                        
                                        <div>
                                            <button type="reset" class="btn btn-outline-warning me-2">
                                                <i class="fas fa-undo me-2"></i>
                                                Limpiar Formulario
                                            </button>
                                            <button type="submit" class="btn btn-primary" id="btnGuardar">
                                                <i class="fas fa-save me-2"></i>
                                                <span id="textoGuardar">Guardar Utensilio</span>
                                                <span id="spinnerGuardar" class="spinner-border spinner-border-sm ms-2 d-none"></span>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Consejos -->
                <div class="row mt-4">
                    <div class="col-12">
                        <div class="card bg-light border-0">
                            <div class="card-body">
                                <h6 class="text-muted mb-3">
                                    <i class="fas fa-lightbulb me-2"></i>
                                    Consejos para completar el formulario
                                </h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <ul class="list-unstyled mb-0">
                                            <li class="mb-2">
                                                <i class="fas fa-check text-success me-2"></i>
                                                Use nombres descriptivos y específicos
                                            </li>
                                            <li class="mb-2">
                                                <i class="fas fa-check text-success me-2"></i>
                                                La cantidad disponible no debe exceder la total
                                            </li>
                                            <li class="mb-2">
                                                <i class="fas fa-check text-success me-2"></i>
                                                Configure un stock mínimo apropiado
                                            </li>
                                        </ul>
                                    </div>
                                    <div class="col-md-6">
                                        <ul class="list-unstyled mb-0">
                                            <li class="mb-2">
                                                <i class="fas fa-check text-success me-2"></i>
                                                Use ubicaciones consistentes y claras
                                            </li>
                                            <li class="mb-2">
                                                <i class="fas fa-check text-success me-2"></i>
                                                El precio ayuda a calcular el valor del inventario
                                            </li>
                                            <li class="mb-2">
                                                <i class="fas fa-check text-success me-2"></i>
                                                Revise toda la información antes de guardar
                                            </li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const BASE_URL = '<?php echo BASE_URL; ?>';
    </script>
    <script src="<?php echo ASSETS_URL; ?>/js/form-utensilio.js"></script>
</body>
</html>