<?php
// Verificar autenticación
$authService = new AuthService();
if (!$authService->estaAutenticado()) {
    header('Location: ' . BASE_URL . '/login');
    exit;
}
$usuario = $authService->getUsuarioActual();

// Obtener ID del utensilio
$id = $_GET['id'] ?? 0;
if (!$id) {
    header('Location: ' . BASE_URL . '/utensilios');
    exit;
}

// Obtener servicios
$inventarioService = new InventarioService();

try {
    // Obtener datos del utensilio
    $utensilio = $inventarioService->obtenerPorId($id);
    if (!$utensilio) {
        throw new Exception('Utensilio no encontrado');
    }
    
    // Obtener historial de movimientos recientes
    $movimientoService = new MovimientoService();
    $movimientos = $movimientoService->obtenerHistorialUtensilio($id, 10);
    
} catch (Exception $e) {
    $_SESSION['error'] = $e->getMessage();
    header('Location: ' . BASE_URL . '/utensilios');
    exit;
}

// Calcular estadísticas
$valorTotal = $utensilio->cantidad_total * $utensilio->precio_unitario;
$porcentajeDisponible = $utensilio->cantidad_total > 0 ? 
    ($utensilio->cantidad_disponible / $utensilio->cantidad_total) * 100 : 0;
$stockBajo = $utensilio->cantidad_disponible <= $utensilio->cantidad_minima;
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($utensilio->nombre); ?> - <?php echo APP_NAME; ?></title>
    <meta name="csrf-token" content="<?php echo $authService->getCSRFToken(); ?>">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="<?php echo ASSETS_URL; ?>/css/utensilio-detalle.css" rel="stylesheet">
</head>
<body>
    <!-- Navbar -->
    <?php include VIEWS_PATH . '/partials/navbar.php'; ?>

    <div class="container py-4">
        
        <!-- Breadcrumb -->
        <nav aria-label="breadcrumb" class="mb-4">
            <ol class="breadcrumb">
                <li class="breadcrumb-item">
                    <a href="<?php echo BASE_URL; ?>/dashboard">
                        <i class="fas fa-home"></i> Dashboard
                    </a>
                </li>
                <li class="breadcrumb-item">
                    <a href="<?php echo BASE_URL; ?>/utensilios">Utensilios</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">
                    <?php echo htmlspecialchars($utensilio->nombre); ?>
                </li>
            </ol>
        </nav>

        <!-- Encabezado con acciones -->
        <div class="row mb-4">
            <div class="col-md-8">
                <h1 class="h3 mb-2">
                    <i class="fas fa-utensils text-primary me-2"></i>
                    <?php echo htmlspecialchars($utensilio->nombre); ?>
                    
                    <!-- Badge de estado -->
                    <?php
                    $badgeClass = '';
                    switch($utensilio->estado) {
                        case 'excelente': $badgeClass = 'bg-success'; break;
                        case 'bueno': $badgeClass = 'bg-primary'; break;
                        case 'regular': $badgeClass = 'bg-warning'; break;
                        case 'malo': $badgeClass = 'bg-danger'; break;
                        case 'fuera_servicio': $badgeClass = 'bg-dark'; break;
                    }
                    ?>
                    <span class="badge <?php echo $badgeClass; ?> ms-2">
                        <?php echo ucfirst($utensilio->estado); ?>
                    </span>
                    
                    <?php if ($stockBajo): ?>
                        <span class="badge bg-warning text-dark ms-1">
                            <i class="fas fa-exclamation-triangle"></i> Stock Bajo
                        </span>
                    <?php endif; ?>
                </h1>
                
                <p class="text-muted mb-0">
                    <strong>Tipo:</strong> <?php echo ucfirst($utensilio->tipo); ?> |
                    <strong>Ubicación:</strong> <?php echo htmlspecialchars($utensilio->ubicacion); ?> |
                    <strong>Creado:</strong> <?php echo date('d/m/Y H:i', strtotime($utensilio->fecha_creacion)); ?>
                </p>
            </div>
            
            <div class="col-md-4 text-md-end">
                <div class="btn-group" role="group">
                    <?php if ($usuario->puedeEditar('utensilios')): ?>
                        <a href="<?php echo BASE_URL; ?>/utensilios/editar/<?php echo $utensilio->id; ?>" 
                           class="btn btn-outline-primary">
                            <i class="fas fa-edit"></i> Editar
                        </a>
                    <?php endif; ?>
                    
                    <?php if ($usuario->puedeGestionar('movimientos')): ?>
                        <div class="btn-group">
                            <button type="button" class="btn btn-success dropdown-toggle" data-bs-toggle="dropdown">
                                <i class="fas fa-exchange-alt"></i> Movimientos
                            </button>
                            <ul class="dropdown-menu">
                                <li>
                                    <a class="dropdown-item" href="#" onclick="registrarMovimiento('entrada')">
                                        <i class="fas fa-plus text-success"></i> Registrar Entrada
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="#" onclick="registrarMovimiento('salida')">
                                        <i class="fas fa-minus text-danger"></i> Registrar Salida
                                    </a>
                                </li>
                                <li><hr class="dropdown-divider"></li>
                                <li>
                                    <a class="dropdown-item" href="<?php echo BASE_URL; ?>/movimientos?utensilio=<?php echo $utensilio->id; ?>">
                                        <i class="fas fa-history"></i> Ver Historial Completo
                                    </a>
                                </li>
                            </ul>
                        </div>
                    <?php endif; ?>
                    
                    <button type="button" class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">
                        <i class="fas fa-ellipsis-v"></i>
                    </button>
                    <ul class="dropdown-menu">
                        <li>
                            <a class="dropdown-item" href="#" onclick="imprimirEtiqueta()">
                                <i class="fas fa-print"></i> Imprimir Etiqueta
                            </a>
                        </li>
                        <li>
                            <a class="dropdown-item" href="#" onclick="exportarDatos()">
                                <i class="fas fa-download"></i> Exportar Datos
                            </a>
                        </li>
                        <li><hr class="dropdown-divider"></li>
                        <?php if ($usuario->puedeEliminar('utensilios')): ?>
                            <li>
                                <a class="dropdown-item text-danger" href="#" onclick="confirmarEliminacion()">
                                    <i class="fas fa-trash"></i> Eliminar Utensilio
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Información principal -->
            <div class="col-lg-8">
                
                <!-- Tarjeta de información básica -->
                <div class="card mb-4 fade-in">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-info-circle me-2"></i>
                            Información General
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <dl class="row">
                                    <dt class="col-sm-4">Nombre:</dt>
                                    <dd class="col-sm-8"><?php echo htmlspecialchars($utensilio->nombre); ?></dd>
                                    
                                    <dt class="col-sm-4">Tipo:</dt>
                                    <dd class="col-sm-8">
                                        <span class="badge bg-light text-dark">
                                            <?php echo ucfirst(str_replace('_', ' ', $utensilio->tipo)); ?>
                                        </span>
                                    </dd>
                                    
                                    <dt class="col-sm-4">Estado:</dt>
                                    <dd class="col-sm-8">
                                        <span class="badge <?php echo $badgeClass; ?>">
                                            <?php echo ucfirst($utensilio->estado); ?>
                                        </span>
                                    </dd>
                                    
                                    <dt class="col-sm-4">Ubicación:</dt>
                                    <dd class="col-sm-8">
                                        <i class="fas fa-map-marker-alt text-muted me-1"></i>
                                        <?php echo htmlspecialchars($utensilio->ubicacion); ?>
                                    </dd>
                                </dl>
                            </div>
                            
                            <div class="col-md-6">
                                <dl class="row">
                                    <dt class="col-sm-5">Precio Unitario:</dt>
                                    <dd class="col-sm-7 fw-bold text-success">
                                        $<?php echo number_format($utensilio->precio_unitario, 2); ?>
                                    </dd>
                                    
                                    <dt class="col-sm-5">Valor Total:</dt>
                                    <dd class="col-sm-7 fw-bold text-primary">
                                        $<?php echo number_format($valorTotal, 2); ?>
                                    </dd>
                                    
                                    <dt class="col-sm-5">Creado por:</dt>
                                    <dd class="col-sm-7"><?php echo htmlspecialchars($utensilio->creado_por_nombre ?? 'Sistema'); ?></dd>
                                    
                                    <dt class="col-sm-5">Última actualización:</dt>
                                    <dd class="col-sm-7">
                                        <?php echo date('d/m/Y H:i', strtotime($utensilio->fecha_actualizacion)); ?>
                                    </dd>
                                </dl>
                            </div>
                        </div>
                        
                        <?php if ($utensilio->descripcion): ?>
                            <hr>
                            <div>
                                <strong>Descripción:</strong>
                                <p class="mt-2 mb-0"><?php echo nl2br(htmlspecialchars($utensilio->descripcion)); ?></p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Historial de movimientos -->
                <div class="card fade-in">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">
                            <i class="fas fa-history me-2"></i>
                            Movimientos Recientes
                        </h5>
                        <a href="<?php echo BASE_URL; ?>/movimientos?utensilio=<?php echo $utensilio->id; ?>" 
                           class="btn btn-sm btn-outline-primary">
                            Ver Todo
                        </a>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($movimientos)): ?>
                            <div class="text-center py-4 text-muted">
                                <i class="fas fa-inbox fa-2x mb-3"></i>
                                <p>No hay movimientos registrados para este utensilio</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Fecha</th>
                                            <th>Tipo</th>
                                            <th>Cantidad</th>
                                            <th>Usuario</th>
                                            <th>Observaciones</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($movimientos as $mov): ?>
                                            <tr>
                                                <td>
                                                    <small class="text-muted">
                                                        <?php echo date('d/m/Y H:i', strtotime($mov->fecha_movimiento)); ?>
                                                    </small>
                                                </td>
                                                <td>
                                                    <?php if ($mov->tipo_movimiento === 'entrada'): ?>
                                                        <span class="badge bg-success">
                                                            <i class="fas fa-plus"></i> Entrada
                                                        </span>
                                                    <?php else: ?>
                                                        <span class="badge bg-danger">
                                                            <i class="fas fa-minus"></i> Salida
                                                        </span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <strong><?php echo $mov->cantidad; ?></strong>
                                                </td>
                                                <td>
                                                    <small><?php echo htmlspecialchars($mov->usuario_nombre ?? 'Sistema'); ?></small>
                                                </td>
                                                <td>
                                                    <small class="text-muted">
                                                        <?php echo htmlspecialchars($mov->observaciones ?: '-'); ?>
                                                    </small>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Estadísticas y acciones rápidas -->
            <div class="col-lg-4">
                
                <!-- Estadísticas de inventario -->
                <div class="card mb-4 fade-in">
                    <div class="card-header">
                        <h6 class="mb-0">
                            <i class="fas fa-chart-bar me-2"></i>
                            Estado del Inventario
                        </h6>
                    </div>
                    <div class="card-body">
                        <!-- Cantidad total -->
                        <div class="stat-item mb-3">
                            <div class="d-flex justify-content-between align-items-center mb-1">
                                <span class="text-muted">Cantidad Total</span>
                                <span class="fw-bold"><?php echo $utensilio->cantidad_total; ?></span>
                            </div>
                        </div>
                        
                        <!-- Cantidad disponible -->
                        <div class="stat-item mb-3">
                            <div class="d-flex justify-content-between align-items-center mb-1">
                                <span class="text-muted">Disponible</span>
                                <span class="fw-bold text-success"><?php echo $utensilio->cantidad_disponible; ?></span>
                            </div>
                            <div class="progress" style="height: 6px;">
                                <div class="progress-bar bg-success" 
                                     style="width: <?php echo $porcentajeDisponible; ?>%"></div>
                            </div>
                            <small class="text-muted"><?php echo round($porcentajeDisponible, 1); ?>% disponible</small>
                        </div>
                        
                        <!-- En uso -->
                        <?php $enUso = $utensilio->cantidad_total - $utensilio->cantidad_disponible; ?>
                        <div class="stat-item mb-3">
                            <div class="d-flex justify-content-between align-items-center mb-1">
                                <span class="text-muted">En Uso</span>
                                <span class="fw-bold text-warning"><?php echo $enUso; ?></span>
                            </div>
                        </div>
                        
                        <!-- Stock mínimo -->
                        <div class="stat-item">
                            <div class="d-flex justify-content-between align-items-center mb-1">
                                <span class="text-muted">Stock Mínimo</span>
                                <span class="fw-bold"><?php echo $utensilio->cantidad_minima; ?></span>
                            </div>
                            <?php if ($stockBajo): ?>
                                <div class="alert alert-warning py-2 mt-2 mb-0">
                                    <small>
                                        <i class="fas fa-exclamation-triangle"></i>
                                        Stock por debajo del mínimo
                                    </small>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Información adicional -->
                <div class="card fade-in">
                    <div class="card-header">
                        <h6 class="mb-0">
                            <i class="fas fa-cogs me-2"></i>
                            Información Técnica
                        </h6>
                    </div>
                    <div class="card-body">
                        <dl class="mb-0">
                            <dt>ID del Sistema:</dt>
                            <dd class="text-muted"><?php echo $utensilio->id; ?></dd>
                            
                            <dt>Fecha de Creación:</dt>
                            <dd class="text-muted">
                                <?php echo date('d/m/Y H:i:s', strtotime($utensilio->fecha_creacion)); ?>
                            </dd>
                            
                            <dt>Última Modificación:</dt>
                            <dd class="text-muted">
                                <?php echo date('d/m/Y H:i:s', strtotime($utensilio->fecha_actualizacion)); ?>
                            </dd>
                            
                            <?php if (isset($utensilio->version)): ?>
                                <dt>Versión:</dt>
                                <dd class="text-muted"><?php echo $utensilio->version; ?></dd>
                            <?php endif; ?>
                        </dl>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal para movimientos -->
    <div class="modal fade" id="modalMovimiento" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Registrar <span id="tipoMovimientoTexto"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="formMovimiento">
                    <div class="modal-body">
                        <input type="hidden" id="tipoMovimiento" name="tipo_movimiento">
                        <input type="hidden" name="utensilio_id" value="<?php echo $utensilio->id; ?>">
                        <input type="hidden" name="csrf_token" value="<?php echo $authService->getCSRFToken(); ?>">
                        
                        <div class="mb-3">
                            <label for="cantidadMovimiento" class="form-label">Cantidad</label>
                            <input type="number" class="form-control" id="cantidadMovimiento" 
                                   name="cantidad" min="1" max="<?php echo $utensilio->cantidad_disponible; ?>" required>
                        </div>
                        
                        <div class="mb-3">
                            <label for="observacionesMovimiento" class="form-label">Observaciones</label>
                            <textarea class="form-control" id="observacionesMovimiento" 
                                      name="observaciones" rows="3" placeholder="Motivo del movimiento..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary" id="btnConfirmarMovimiento">
                            Registrar <span id="textoBotonMovimiento"></span>
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const BASE_URL = '<?php echo BASE_URL; ?>';
        const utensilioId = <?php echo $utensilio->id; ?>;
        const cantidadDisponible = <?php echo $utensilio->cantidad_disponible; ?>;
    </script>
    <script src="<?php echo ASSETS_URL; ?>/js/utensilio-detalle.js"></script>
</body>
</html>